package com.qiho.center.biz.service.impl.order;

import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.biz.event.FundInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.event.OrderAuditEvent;
import com.qiho.center.biz.event.OrderCreateEvent;
import com.qiho.center.biz.service.order.OrderSnapshotService;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DomainConstantUtil;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.dao.QihoShortUrlDAO;
import com.qiho.center.common.dao.QihoTemplateDAO;
import com.qiho.center.common.entity.QihoShortUrlEntity;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.util.ShortUrlUtil;
import com.qiho.center.common.util.SimpleTemplateUtil;

/**
 * Created by liuyao on 2017/6/3.
 */
@Service
@BizEventListener
public class OrderSnapshotServiceImpl implements OrderSnapshotService {

    private static final Logger LOG = LoggerFactory.getLogger(OrderSnapshotServiceImpl.class);
    
    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;
    
    @Autowired
    private QihoShortUrlDAO qihoShortUrlDAO;
    
    @Autowired
    private QihoTemplateDAO qihoTemplateDAO;
    
    @Autowired
    private SmsService smsService;

    @Override
    public PagenationDto<OrderSnapshotDto> findOrderPage(OrderPageParams params) {
        PagenationDto<OrderSnapshotDto> page = new PagenationDto<>();
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.findOrderPage(params);
        page.setTotal(qihoOrderSnapshotDAO.findOrderPageCount(params));
        DtoTransform transform = new DtoTransform();
        List<OrderSnapshotDto> dtoList = Lists.transform(list,transform);
        page.setList(dtoList);
        return page;
    }

    @Override
    public List<OrderSnapshotDto> findOrderPageForExport(OrderPageParams params) {
        List<QihoOrderSnapshotEntity> list = Lists.newArrayList();
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null){
                list.add(order);
            }
        }else{
            list.addAll(qihoOrderSnapshotDAO.exportOrderPage(params));
        }
        DtoTransform transform = new DtoTransform();
        return Lists.transform(list,transform);
    }

    @Override
    public Integer findAllPageCount(OrderPageParams params) {
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null) {
                return 1;
            }else{
                return 0;
            }
        }else{
            return qihoOrderSnapshotDAO.findOrderPageCount(params);
        }
    }


    private class DtoTransform implements Function<QihoOrderSnapshotEntity,OrderSnapshotDto>{

        @Override
        public OrderSnapshotDto apply(QihoOrderSnapshotEntity input) {
            if(input==null){
                return null;
            }
            OrderSnapshotDto dto = new OrderSnapshotDto();
            BeanUtils.copy(input,dto);
            dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(input.getFundStatus()));
            dto.setOrderStatusEnum(OrderStatusEnum.getByCode(input.getOrderStatus()));
            dto.setDeliveryEnum(DeliveryEnum.getByCode(input.getDelivery()));
            return dto;
        }
    }

    @Override
    public Integer exportOrderPageCount(OrderPageParams params) {
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null) {
                return 1;
            }else{
                return 0;
            }
        }else{
            return qihoOrderSnapshotDAO.exportOrderPageCount(params);
        }
    }

    @Override
    public Integer updateOrderStatus(String orderId, OrderStatusEnum orderStatus) {
        return qihoOrderSnapshotDAO.updateOrderStatus(orderId, orderStatus);
    }

    @Override
    public OrderSnapshotDto findByOrderId(String orderId) {
        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if(entity == null){
            return null;
        }
        OrderSnapshotDto dto = BeanUtils.copy(entity, OrderSnapshotDto.class);
        dto.setOrderStatusEnum(OrderStatusEnum.getByCode(entity.getOrderStatus()));
        dto.setDeliveryEnum(DeliveryEnum.getByCode(entity.getDelivery()));
        dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(entity.getFundStatus()));
        return dto;
    }
    
    /**
     * 监听订单创建事件
     * @author zhanglihui
     * @param orderCreateEvent
     */
    @Subscribe
    public void createOrderListener(OrderCreateEvent orderCreateEvent){
        String orderId = orderCreateEvent.getOrderId();
        // TODO 防作弊规则扫描
        
        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(orderId);
        
        sendMessage(order);
        
    }

    /**
     *
     * @author zhanglihui
     * @param order
     */
    private void sendMessage(QihoOrderSnapshotEntity order) {
        String qihoWebUrl = DomainConstantUtil.getQihoWebUrl();
        // 手机端域名
        String domain = StringUtils.substring(qihoWebUrl, StringUtils.indexOf(qihoWebUrl, "//")+2);
        // 订单详情页长链接完整地址
        String longUrl = qihoWebUrl + "/order/detail?orderId="+order.getOrderId();
        // 短链接地址
        String shortUrl = ShortUrlUtil.generateShortUrl(longUrl);
        // 保存长链接和短链接的映射关系
        QihoShortUrlEntity qihoShortUrlEntity = new QihoShortUrlEntity();
        qihoShortUrlEntity.setShortUrl(shortUrl);
        qihoShortUrlEntity.setLongUrl(longUrl);
        qihoShortUrlDAO.insert(qihoShortUrlEntity);
        String mobile = order.getMobile();
        // 用户下单成功短信通知
        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCCESS.getCode());
        if(template.getTemplateEnable()){
            Map<String, String> context = Maps.newHashMap();
            context.put("orderDetailUrl", domain+"/"+shortUrl);
            context.put("itemName", order.getItemShortName());
            String text = SimpleTemplateUtil.render(template.getTemplateContext(), context);
            smsService.singleSend(text, mobile);
        }
        
        try {
            Thread.sleep(2000l);
        } catch (InterruptedException e) {
            throw new QihoException("发送短信线程中断", e);
        }
        // 发送下单后温馨提示
        template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_NOTIFY.getCode());
        if(template.getTemplateEnable()){
            smsService.singleSend(template.getTemplateContext(), mobile);
        }
    }

    /**
     * 监听订单审核事件
     * @param event
     */
    @Subscribe
    public void OrderAuditEventListener(OrderAuditEvent event){
        if(Objects.equal(event.getAuditResult(), OrderAuditParams.AUDIT_SUCCESS)){
            qihoOrderSnapshotDAO.updateOrderStatus(event.getOrderId(),OrderStatusEnum.TO_SEND);
        }else{
            qihoOrderSnapshotDAO.updateOrderStatus(event.getOrderId(),OrderStatusEnum.CLOSED);
        }
    }

    /**
     * 物流子订单更新事件
     * @param event
     */
    @Subscribe
    public void LogisticsInfoUpdateListener(LogisticsInfoUpdateEvent event) {
        
        // 发送发货短信
        String postId = event.getPostId();
        String mobile = event.getMobile();
        
        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SEND.getCode());
        if(template.getTemplateEnable()){
            Map<String, String> context = Maps.newHashMap();
            context.put("postId", postId);
            
            String qihoWebUrl = DomainConstantUtil.getQihoWebUrl();
            // 手机端域名
            String domain = StringUtils.substring(qihoWebUrl, StringUtils.indexOf(qihoWebUrl, "//")+2);
            // 订单详情页长链接完整地址
            String longUrl = qihoWebUrl + "/order/detail?orderId="+event.getOrderId();
            // 短链接地址
            String shortUrl = ShortUrlUtil.generateShortUrl(longUrl);
            context.put("orderDetailUrl", domain+"/"+shortUrl);
            String text = SimpleTemplateUtil.render(template.getTemplateContext(), context);
            smsService.singleSend(text, mobile);
        }
    }


    @Subscribe
    public void fundInfoUpdateListener(FundInfoUpdateEvent event){
        QihoOrderSnapshotEntity entity = new QihoOrderSnapshotEntity();
        BeanUtils.copy(event,entity);
        qihoOrderSnapshotDAO.updataFundInfo(entity);
    }
    
    /**
     * 监听物流状态签收操作事件
     * @author chencheng
     * @param event
     */
    @Subscribe
    public void updateLogisticsListener(LogisticsUpdateEvent event) {
        LogisticsOrderStatusEnum status = event.getOrderStatus();
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(event.getOrderId());
        orderSnapshot.setLogisticsStatus(event.getLogisticsStatus().getDesc());
        switch (status) {
            case SUCCESS:
                orderSnapshot.setOrderStatus(OrderStatusEnum.SUCCESS.getCode());
                qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
                break;
            case FAILED:
                orderSnapshot.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
                qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
                break;
            default:
                LOG.info("不需要更改订单快照表，状态status:{}"+status.getCode());
                break;
        }
    }
}
