package com.qiho.center.biz.job;

import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.tuia.core.api.remoteservice.advert.RemoteAdvertStopService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.alibaba.fastjson.JSONObject;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.dao.QihoItemExtDAO;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entity.item.QihoItemExtEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.params.KeyPair;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by danke on 2017/9/18.
 * 定时任务,用于定时扫描所有配置了阈值的商品,发现达到阈值,则关闭配置的广告计划
 * PS:tuia接口28号上线,任务暂时不配置定时
 */
@Component
public class ItemAdvertStopJob extends AbstractQihoSimpleElasticJob {

    @Resource
    private QihoItemExtDAO qihoItemExtDAO;

    @Resource
    private QihoItemSkuDAO qihoItemSkuDAO;

    @Resource
    private QihoItemDAO qihoItemDAO;

    @Resource
    private RemoteStockService remoteStockService;

    @Resource
    private RemoteAdvertStopService remoteAdvertStopService;

    private static final Logger LOGGER = LoggerFactory.getLogger(ItemAdvertStopJob.class);

    @Override
    protected void doProcess(JobExecutionMultipleShardingContext shardingContext) {
        LOGGER.warn("扫描阈值商品 and 停止对应广告的定时任务 start");
        StopWatch times = new StopWatch();
        times.start();
        Boolean resultBase = stopAdvertAboutItem();
        times.stop();
        LOGGER.warn("扫描阈值商品 and 停止对应广告的定时任务 end wasteTime={} result={}",times.getTime(),resultBase);
    }

    private Boolean stopAdvertAboutItem(){
        //查询所有商品,不包含逻辑删除的
        List<QihoItemDetailEntity> qihoIt = qihoItemDAO.queryItemByStatus(new QihoItemEntity());
        if (CollectionUtils.isEmpty(qihoIt))
            return Boolean.TRUE;
        List<Long> itemIds = qihoIt.stream().map(QihoItemDetailEntity::getId).collect(Collectors.toList());
        List<QihoItemExtEntity> extEntityList = qihoItemExtDAO.queryAllItemExt(itemIds);
        //配置的阈值和广告id对应商品id的map
        Map<Long,KeyPair<String,Long>> itemIdsRule = new HashMap<>();
        //获取配置了阈值的商品id,并且获取itemid为key的配置规则
        List<Long> thresholdItemIds = extEntityList.stream()
                                                        .filter(this::needStopAdvert)//过滤未配置阈值的商品
                                                        .peek(e -> itemIdsRule.put(e.getItemId(),itemAboutAdvertRule(e)))//抽取商品和规则对应关系
                                                        .map(QihoItemExtEntity::getItemId)//抽取商品id集合,用于查询库存
                                                        .collect(Collectors.toList());//收集器
        if (CollectionUtils.isEmpty(thresholdItemIds) || itemIdsRule.isEmpty())//配置为空,不往下走
            return Boolean.TRUE;
        //通过商品id查询库存id
        List<QihoItemSkuEntity> skuEntityList = qihoItemSkuDAO.getSkuByItemIds(thresholdItemIds);
        //因为商品id和库存id是多对多的关系,所以使用map<itemId,List<stockId>>来标示关系
        Map<Long,List<Long>> itemIdWithStockIds = Maps.newHashMap();
        //收集库存id集合,去重复,并且填充商品id和库存ids的map关系
        List<Long> stockIds = skuEntityList.stream().peek(skuEntity -> {
            Long itemId = skuEntity.getItemId();
            List<Long> stockId = itemIdWithStockIds.get(itemId);
            if (CollectionUtils.isEmpty(stockId)){
                stockId = new ArrayList<>();
                itemIdWithStockIds.put(itemId,stockId);
            }
            stockId.add(skuEntity.getStockId());
        }).map(QihoItemSkuEntity::getStockId).distinct().collect(Collectors.toList());
        try{
            //查询旺店通库存
            DubboResult<List<StockDto>> dubboResult = remoteStockService.findBatchByIds(stockIds);
            if (!dubboResult.isSuccess() || CollectionUtils.isEmpty(dubboResult.getResult())){
                throw new QihoException("批量查询旺店通库存失败");
            }
            List<StockDto> stockDtos = dubboResult.getResult();
            //收集库存id和库存数量的map集合
            Map<Long,Long> stockIdWithNum = stockDtos.stream().collect(Collectors.toMap(StockDto::getStockID,StockDto::getStock));
            //获取每个商品id的总库存
            Map<Long,Long> itemIdWithAllStock = this.getAllStockAboutItem(itemIdWithStockIds,stockIdWithNum);
            //获取需要停止的广告id
            List<Long> advertIds = this.getNeedStopAdvertIds(itemIdWithAllStock,itemIdsRule);
            if (CollectionUtils.isEmpty(advertIds))
                return Boolean.TRUE;
            remoteAdvertStopService.batchStopAdvertByList(advertIds);
            return Boolean.TRUE;
        }catch (Exception e){
            LOGGER.error("定时任务扫描商品阈值任务error msg={}",e);
            return Boolean.FALSE;
        }
    }

    /**
     * 判断商品是否配置了关闭广告的阈值
     */
    private boolean needStopAdvert(QihoItemExtEntity param){
        JSONObject metaVal = JSONObject.parseObject(param.getExtParam());
        return StringUtils.isNotBlank(metaVal.getString(ItemConstants.ItemExtConstans.ADVERTIDS))
                && StringUtils.isNotBlank(metaVal.getString(ItemConstants.ItemExtConstans.THRESHOLD));
    }

    /**
     * 将扩展字段中的值取出来,Key:itemId,Value:keypair->FirstKey是广告计划String,SecondKey是配置的阈值
     */
    private KeyPair<String,Long> itemAboutAdvertRule(QihoItemExtEntity itemExtEntity){
        KeyPair<String,Long> keyPair = new KeyPair<>();
        JSONObject extParam = JSONObject.parseObject(itemExtEntity.getExtParam());
        keyPair.setFirstKey(extParam.getString(ItemConstants.ItemExtConstans.ADVERTIDS));
        keyPair.setSecondKey(extParam.getLong(ItemConstants.ItemExtConstans.THRESHOLD));
        return keyPair;
    }

    /**
     * 计算商品的总库存
     * allStock = stock1+stock2+stock3 ...
     */
    private Map<Long,Long> getAllStockAboutItem(Map<Long,List<Long>> itemIdWithStockIds,Map<Long,Long> stockIdWithNum){
        Map<Long,Long> itemIdWithAllStock = Maps.newHashMap();
        itemIdWithStockIds.entrySet().stream().forEach(e -> {
            Long itemId = e.getKey();
            Long allStock = 0l;
            for (Long stockId : e.getValue()){
                allStock += stockIdWithNum.get(stockId);
            }
            itemIdWithAllStock.put(itemId,allStock);
        });
        return itemIdWithAllStock;
    }

    /**
     * 获取需要停止的广告ids
     * @param itemStockMap 商品和库存对应的map
     * @param itemIdsRule 商品和配置规则对应的map
     * @return
     */
    private List<Long> getNeedStopAdvertIds(Map<Long,Long> itemStockMap,Map<Long,KeyPair<String,Long>> itemIdsRule){
        if (null == itemIdsRule || null == itemStockMap)
            return new ArrayList<>();
        List<List<Long>> advertIdList = itemIdsRule.entrySet().stream().filter(e ->{//过滤掉不需要停止广告的商品
            Long itemId = e.getKey();//商品id
            Long stock = itemStockMap.get(itemId);//库存
            KeyPair<String,Long> rule = e.getValue();//配置规则
            Long threshold = rule.getSecondKey();//第二个key是阈值
            return threshold >= stock;//小于等于阈值的时候都会被推送
        }).map(this::advertStrToLongList).collect(Collectors.toList());
        List<Long> advertIds = new ArrayList<>();
        for (List<Long> advertId : advertIdList){
            advertIds.addAll(advertId);
        }
        return advertIds.stream().distinct().collect(Collectors.toList());
    }

    /**
     * 处理配置的广告计划id,str:id1,id2 to List<Long></>
     */
    private List<Long> advertStrToLongList(Map.Entry<Long,KeyPair<String,Long>> pairEntry){
        String advertIdStr = pairEntry.getValue().getFirstKey();
        String[] advertIdStrs = advertIdStr.split(",");
        return Stream.of(advertIdStrs).map(Long::valueOf).collect(Collectors.toList());
    }
}
