package com.qiho.center.biz.paychannel.pay;

import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.domain.AlipayTradeFastpayRefundQueryModel;
import com.alipay.api.domain.AlipayTradeQueryModel;
import com.alipay.api.domain.AlipayTradeRefundModel;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.internal.util.AlipaySignature;
import com.alipay.api.request.AlipayTradeFastpayRefundQueryRequest;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.request.AlipayTradeRefundRequest;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.alipay.api.response.AlipayTradeFastpayRefundQueryResponse;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.alipay.api.response.AlipayTradeRefundResponse;
import com.alipay.api.response.AlipayTradeWapPayResponse;
import com.qiho.center.api.constant.AliPayConstant;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.remoteservice.order.RemoteOrderService;
import com.qiho.center.api.util.AliPayAppUtil;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.common.params.AliPayParam;
import com.qiho.center.common.util.AppLogUtil;

/**
 * Created by chensong on 2017/7/24.
 */
@Component
public class AlipayProcessor extends PayChannelProcessor implements InitializingBean {

    private static final Logger LOG          = LoggerFactory.getLogger(AlipayProcessor.class);

    @Value("${alipay.serverUrl}")
    private String              serverUrl;

    @Value("${alipay.appId}")
    private String              appId;

    @Value("${alipay.privateKey}")
    private String              privateKey;

    @Value("${alipay.publicKey}")
    private String              alipayPulicKey;

    /** 请求和签名使用的编码 */
    private static final String CHARSET      = "UTF-8";

    /** 签名和算法类型 */
    private static final String SIGN_TYPE    = "RSA";

    /** 数据交换格式 */
    private static final String FORMAT       = "json";

    /** 销售产品码、用于标识手机网站支付、固定 */
    private static final String PRODUCT_CODE = "QUICK_WAP_PAY";

    @Autowired
    private AliPayParam         aliPayParam;

    @Autowired
    private RemoteOrderService  remoteOrderService;

    private AlipayClient        alipayClient;

    @Override
    public String payExecute(String orderId,Map<String,String>params) {
        OrderDto dto = remoteOrderService.findByOrderId(orderId).getResult();
        AlipayTradeWapPayRequest alipayRequest = new AlipayTradeWapPayRequest();

        // 封装请求支付信息
        AlipayTradeWapPayModel model = new AlipayTradeWapPayModel();
        // 外部流水号，奇货系统生成
        model.setOutTradeNo(dto.getOrderId());
        // 订单名称
        model.setSubject(dto.getOrderItem().getItemName());
        // 订单金额
        model.setTotalAmount(AliPayAppUtil.fenToYuan(dto.getOrderAmt()));
        // 商品描述
        model.setBody(dto.getOrderItem().getSkuName() + "*" + dto.getOrderItem().getQuantity());
        // 销售产品码
        model.setProductCode(PRODUCT_CODE);

        alipayRequest.setReturnUrl(aliPayParam.getReturnUrl() + "?orderId=" + orderId);
        alipayRequest.setNotifyUrl(aliPayParam.getNotifyUrl());
        alipayRequest.setBizModel(model);

        String form = "";
        try {
            AlipayTradeWapPayResponse response = alipayClient.pageExecute(alipayRequest);
            if(response.isSuccess()){
                form = response.getBody();
                updateFundOrder(orderId, response);
            }
            
        } catch (AlipayApiException e) {
            AppLogUtil.error(LOG, "支付宝发起付款失败", e);
        }
        return form;
    }
    
    /**
     *
     * @author zhanglihui
     * @param orderId
     * @param response
     */
    private void updateFundOrder(String orderId, AlipayTradeWapPayResponse response) {
        List<FundOrderDto> list = fundOrderService.findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
        if(CollectionUtils.isNotEmpty(list)){
            FundOrderDto fundOrder = list.get(0);
            fundOrder.setOutSeqNo(response.getTradeNo());
            fundOrderService.update(fundOrder);
        }
    }

    @Override
    public PayDto queryPayResult(String orderId) {
        AlipayTradeQueryRequest request = new AlipayTradeQueryRequest();
        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(orderId);
        request.setBizModel(model);
        PayDto result = new PayDto();
        result.setOrderId(orderId);
        try {
            AlipayTradeQueryResponse response = alipayClient.execute(request);
            if (response.isSuccess()) {
                result.setPayerId(response.getBuyerUserId());
                result.setPayAmt(AliPayAppUtil.yuanToFen(response.getTotalAmount()));
                result.setOutTradeNo(response.getTradeNo());
                result.setSuccess(StringUtils.containsAny(response.getTradeStatus(),
                                                          AliPayConstant.TradeStatus.TRADE_SUCCESS,
                                                          AliPayConstant.TradeStatus.TRADE_FINISHED));
            } else {
                AppLogUtil.warn(LOG, response.getSubMsg());
            }
        } catch (AlipayApiException e) {
            AppLogUtil.error(LOG, "查询支付宝付款结果失败", e);
        }
        return result;
    }

    @Override
    public void refund(String orderId, String fundId, Integer refundAmt) {
        AlipayTradeRefundRequest request = new AlipayTradeRefundRequest();
        AlipayTradeRefundModel model = new AlipayTradeRefundModel();
        FundOrderDto dto = fundOrderService.findByFundId(fundId);
        model.setOutTradeNo(orderId);
        model.setRefundAmount(AliPayAppUtil.fenToYuan(refundAmt));
        model.setOutRequestNo(fundId);
        request.setBizModel(model);
        try {
            AlipayTradeRefundResponse response = alipayClient.execute(request);
            String code = response.getCode();
            String refundFee = response.getRefundFee();
            if (response.isSuccess() && StringUtils.equals(refundFee, AliPayAppUtil.fenToYuan(refundAmt))) {
                dto.setFundStatus(FundStatusEnum.SUCCESS.getCode());
            } else if (!AliPayConstant.ResponseCode.SUCCESS.equals(code)) {
                dto.setFundStatus(FundStatusEnum.FAILED.getCode());
                AppLogUtil.warn(LOG, "订单【orderId={}】请求退款失败 msg={}", orderId, response.getSubMsg());
            }
            dto.setAccount(response.getBuyerUserId());
            dto.setOutSeqNo(response.getTradeNo());
            fundOrderService.update(dto);
        } catch (AlipayApiException e) {
            AppLogUtil.error(LOG, "订单【orderId={}】请求退款异常", orderId, e);
        }
    }
    
    @Override
    public RefundResult refundQuery(String orderId, String fundId) {
        AlipayTradeFastpayRefundQueryRequest request = new AlipayTradeFastpayRefundQueryRequest();
        AlipayTradeFastpayRefundQueryModel model = new AlipayTradeFastpayRefundQueryModel();
        FundOrderDto dto = fundOrderService.findByFundId(fundId);
        model.setOutTradeNo(orderId);
        model.setOutRequestNo(fundId);
        request.setBizModel(model);
        RefundResult result = new RefundResult();
        result.setFundId(fundId);
        result.setOrderId(orderId);
        try {
            AlipayTradeFastpayRefundQueryResponse response = alipayClient.execute(request);
            result.setRefundAmt(AliPayAppUtil.yuanToFen(response.getRefundAmount()));
            //接口调用成功
            if (response.isSuccess() && StringUtils.equals(response.getRefundAmount(), AliPayAppUtil.fenToYuan(dto.getAmt()))) {
                result.setIsSuccess(true);
                result.setRefundStatus(RefundResult.REFUND_STATUS_SUCCESS);
                result.setOutTradeNo(response.getTradeNo());
            } else {
                result.setIsSuccess(false);
                AppLogUtil.warn(LOG, "支付宝查询退款失败msg={}", response.getSubMsg());
            }

        } catch (AlipayApiException e) {
            AppLogUtil.error(LOG, "支付宝查询退款异常,fundId={}", fundId, e);
        }
        return result;
    }

    @Override
    PayTypeEnum getPayType() {
        return PayTypeEnum.ALIPAY;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        PayChannelDecider.registPaychannel(getPayType(), this);
        // 初始化支付宝客户端
        alipayClient = new DefaultAlipayClient(serverUrl, appId, privateKey, FORMAT, CHARSET, alipayPulicKey, SIGN_TYPE);
    }

    /**
     * @see com.qiho.center.biz.paychannel.pay.PayChannelProcessor#verify(java.util.Map)
     */
    @Override
    public boolean verify(Map<String, String> params) {
        boolean verifyResult = false;
        try {
            verifyResult = AlipaySignature.rsaCheckV1(params, alipayPulicKey, CHARSET, SIGN_TYPE);
        } catch (AlipayApiException e) {
            AppLogUtil.warn(LOG, "支付宝验签失败，params={}", params, e);
        }
        return verifyResult;
    }

    @Override
    public String getOpenIdByCode(String code) {
        return null;
    }
}
