package com.qiho.center.biz.paychannel.pay;

import com.google.common.base.Objects;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.params.AsyncAcceptParams;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.util.AppLogUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * ClassName:PayChannelProcessor <br/>
 * 支付渠道处理器. <br/>
 * Date: 2017年7月17日 下午8:40:14 <br/>
 *
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
public abstract class PayChannelProcessor {

    private static final Logger LOG = LoggerFactory.getLogger(PayChannelProcessor.class);

    @Autowired
    protected FundOrderService fundOrderService;

    @Resource
    protected OrderService orderService;

    /**
     * 参数验签
     *
     * @author zhanglihui
     * @param params  需要参与验签的参数,通常需要剔除签名值和签名类型两个参数,签名密钥交给实现类去获取
     * @return 签名是否通过
     */
    public abstract boolean verify(Map<String, String> params);
    
    /**
     * 根据订单ID构建支付请求，并发起支付流程
     *
     * @author zhanglihui
     * @param orderId 订单ID
     * @return 输出的表单html
     */
    public abstract String payExecute(String orderId,Map<String,String> params);

    /**
     * 支付结果通知处理，调用本方法前需要验签</br> 注意：
     * <ul>
     * <li>只有付款成功的通知才会处理</li>
     * <li>付款失败可以重新支付</li>
     * <li>支付超时会关闭支付订单</li>
     * <li>处理结果失败会抛出异常,需要重新接收通知再尝试处理</li>
     * </ul>
     *
     * @author zhanglihui
     * @param params 
     */
    public void processNotify(AsyncAcceptParams params) {
        List<FundOrderDto> list = fundOrderService.findByOrderIdAndBizType(params.getOrderId(), FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isEmpty(list)) {
            AppLogUtil.warn(LOG, "查不到相关订单信息，忽略支付通知，orderId={}, outSeqNo={}", params.getOrderId(), params.getOutSqeNo());
            return;
        }
        // 一个订单只有一笔付款子订单
        FundOrderDto payOrder = list.get(0);
        String originalFundStatus = payOrder.getFundStatus();
        if (!Objects.equal(params.getReceiptAmount(), payOrder.getAmt())) {
            AppLogUtil.warn(LOG, "支付金额【{}】与订单金额【{}】不一致", params.getReceiptAmount(), payOrder.getAmt());
            return;
        }
        if (needRefund(payOrder.getPayType(), payOrder.getFundStatus())) {
            AppLogUtil.warn(LOG, "订单支付方式为货到付款，或已支付。发起退款,orderId={}", params.getOrderId());
            // 发起退款流程
            fundOrderService.refund(params.getOrderId(), getPayType().getCode(), payOrder.getAmt());
        } else if (StringUtils.equals(payOrder.getPayType(), getPayType().getCode())
                && StringUtils.equals(originalFundStatus, FundStatusEnum.SUCCESS.getCode())) {
            AppLogUtil.warn(LOG, "收到重复消息，忽略，orderId={}, outSeqNo={}", params.getOrderId(), params.getOutSqeNo());
            return;
        } else {
            // 更新主订单状态、资金订单状态，自动审批通过
            payOrder.setPayType(getPayType().getCode());
            payOrder.setFundStatus(FundStatusEnum.SUCCESS.getCode());
            payOrder.setOutSeqNo(params.getOutSqeNo());
            payOrder.setAccount(params.getAccount());
            doProcess(payOrder, originalFundStatus);
        }
    }

    /**
     * @author zhanglihui
     * @param payOrder
     * @param originalFundStatus
     */
    private void doProcess(FundOrderDto payOrder, String originalFundStatus) {
        fundOrderService.notifySuccess(payOrder, originalFundStatus);
    }

    /**
     * 付款子订单已失败，或付款子订单是货到付款，或者支付类型与通知类型不一致 & 已支付成功，需要发起退款流程
     *
     * @author zhanglihui
     * @param payType
     * @param fundStatus
     * @return
     */
    private boolean needRefund(String payType, String fundStatus) {
        return StringUtils.equals(fundStatus, FundStatusEnum.FAILED.getCode())
                || StringUtils.equals(PayTypeEnum.COD.getCode(), payType)
                || (!StringUtils.equals(payType, getPayType().getCode()) && StringUtils.equals(fundStatus,
                FundStatusEnum.SUCCESS.getCode()));
    }

    /**
     * 构建退款参数，发起退款
     *
     * @author zhanglihui
     * @param orderId   主订单ID
     * @param fundId    资金流水ID，标识一次退款请求，同一笔交易多次退款需要保证唯一，如需部分退款，则此参数必传
     * @param refundAmt 退款金额
     */
    public abstract void refund(String orderId, String fundId, Integer refundAmt);

    /**
     * 查询退款结果
     *
     * @author zhanglihui
     * @param orderId 主订单ID
     * @param fundId 资金订单ID，请求退款接口时，传入的退款请求号，如果在退款请求时未传入，则该值为主订单ID
     * @return 退款结果
     */
    public abstract RefundResult refundQuery(String orderId, String fundId);

    abstract PayTypeEnum getPayType();

    public abstract PayDto queryPayResult(String orderId);

    public abstract String getOpenIdByCode(String code);

    /**
     * 发起转账
     * @param account  转账账户
     * @param fundId   资金订单ID
     * @param amount   转账金额，单位:分
     */
    public abstract void transAccount(String account, String fundId, Integer amount);

}