package com.qiho.center.biz.paychannel.pay;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.WeChatConstant;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.util.StringRandUtil;
import com.qiho.center.api.util.WechatPayAppUtil;
import com.qiho.center.api.util.XmlTranformUtil;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.HttpClientUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * Created by qianjue on 2017/8/14.
 */
@Component
public class WechatPayOfficialAccountProcessor extends WechatPayBaseProcessor implements InitializingBean{

	private Logger logger = LoggerFactory.getLogger(WechatPayOfficialAccountProcessor.class);

	private static final String SUFFIX = "p";

	@Override
	public void afterPropertiesSet() throws Exception {
		PayChannelDecider.registPaychannel(PayTypeEnum.WECHATPAY_PUB, this);
	}


	@Override
	public String payExecute(String orderId, Map<String, String> params) {
		Map<String,String> map = Maps.newHashMap();
		try{
			//封装一下支付相关的基础信息
			Map<String,String> data = buildPayParams(orderId,params);
			String parmaStr = XmlTranformUtil.mapToXml(data);
			//发送支付请求
			String httpResultStr = HttpClientUtil.postData(WeChatConstant.Url.PAY_URL,parmaStr);
			//解析返回的支付结果
			Map<String,String> resultMap = XmlTranformUtil.xmlToMap(httpResultStr);
			//如果返回成功有结果,则返回前端跳转的url地址
			//通信返回成功结果
			if(WeChatConstant.ResponseCode.SUCCESS.equals(resultMap.get("return_code"))
				&& WeChatConstant.ResponseCode.SUCCESS.equals(resultMap.get("result_code"))){
				//如果返回结果正常,
				map.put("appId",resultMap.get(WeChatConstant.WechatKey.APP_ID));
				map.put("nonceStr",resultMap.get(WeChatConstant.WechatKey.NONCE_STR));
				map.put("signType",WeChatConstant.WechatDefaultValue.SIGN_TYPE_MD5);
				map.put(WeChatConstant.WechatKey.PACKAGE,WeChatConstant.WechatKey.PREPAY_ID+"="+resultMap.get(WeChatConstant.WechatKey.PREPAY_ID));
				map.put(WeChatConstant.WechatKey.TIMESTAMP,String.valueOf(System.currentTimeMillis() / 1000));
				map.put("paySign",sign(map,WechatPayAppUtil.getSecret()));
				return JSON.toJSONString(map);
			}else{
				AppLogUtil.error(logger,"微信支付发起失败,orderId = {},支付报文 = {}, 返回报文 = {}",orderId,parmaStr,httpResultStr);
			}
		}catch (Exception e){
			AppLogUtil.error(logger,"唤起微信支付异常,orderId = {}",orderId,e);
		}
		return StringUtils.EMPTY;
	}

	private Map<String,String> buildPayParams(String orderId, Map<String, String> params) {
		OrderDto orderDto = orderService.findByOrderId(orderId);
		if (null == orderDto){
			throw new QihoException("无效的订单ID");
		}
		Map<String,String> dataMap = Maps.newHashMap();
		dataMap.put(WeChatConstant.WechatKey.APP_ID, WechatPayAppUtil.getAppId());
		dataMap.put(WeChatConstant.WechatKey.MCH_ID,WechatPayAppUtil.getMchId());
		dataMap.put(WeChatConstant.WechatKey.NONCE_STR, StringRandUtil.getRandomString(32));
		dataMap.put(WeChatConstant.WechatKey.BOBY,"奇货—"+StringUtils.trim(orderDto.getOrderItem().getItemName()));
		dataMap.put(WeChatConstant.WechatKey.OUT_TRADE_NO,orderId+SUFFIX);
		dataMap.put(WeChatConstant.WechatKey.TOTAL_FEE,orderDto.getOrderAmt().toString());
		dataMap.put(WeChatConstant.WechatKey.SPBILL_CREATE_IP,params.get("ip"));
		dataMap.put(WeChatConstant.WechatKey.NOTIFY_URL,WechatPayAppUtil.getNotifyUrl());
		dataMap.put(WeChatConstant.WechatKey.TRADE_TYPE,WeChatConstant.WechatDefaultValue.TRADE_TYPE_JSAPI);
		dataMap.put(WeChatConstant.WechatKey.SCENE_INFO,WeChatConstant.WechatDefaultValue.SCENE_INFO_H5);
		dataMap.put(WeChatConstant.WechatKey.OPEN_ID,params.get("openId"));
		dataMap.put(WeChatConstant.WechatKey.SIGN,sign(dataMap,WechatPayAppUtil.getSecret()));
		return dataMap;
	}

	@Override
	public RefundResult refundQuery(String orderId, String fundId) {
		RefundResult result =  super.refundQuery(orderId+SUFFIX,fundId);
		result.setOrderId(orderId);
		return result;
	}

	@Override
	public PayDto queryPayResult(String orderId) {
		return super.queryPayResult(orderId+SUFFIX);
	}

	@Override
	public void refund(String orderId, String fundId, Integer refundAmt) {
		 super.refund(orderId+SUFFIX,fundId,refundAmt);
	}


	@Override
	PayTypeEnum getPayType() {
		return PayTypeEnum.WECHATPAY_PUB;
	}
}
