package com.qiho.center.biz.service.impl.order;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.params.AsyncAcceptParams;
import com.qiho.center.api.params.FundPageParam;
import com.qiho.center.api.params.OrderQueryParams;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.biz.paychannel.pay.PayChannelDecider;
import com.qiho.center.biz.paychannel.pay.PayChannelProcessor;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.service.order.OrderTaskService;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoTemplateDAO;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.util.ShortUrlUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

@Service
public class OrderTaskServiceImpl implements OrderTaskService {
	
	private  static Logger log = LoggerFactory.getLogger(OrderTaskServiceImpl.class);
	
	private  static int pageSize = 100;
	
	private  SimpleDateFormat format = new SimpleDateFormat("MM月dd日HH时mm分");
	
	@Autowired
	private SmsService  smsService;
	@Autowired
	private  QihoOrderDAO  qihoOrderDAO;
	@Autowired
	private  QihoTemplateDAO qihoTemplateDAO;
	@Autowired
	private  FundOrderService  fundOrderService;
	@Autowired
	private	 OrderService  orderService;
	@Autowired
	private  PayChannelDecider  payChannelDecider;

	@Override
	public void remainOrderPay() {
		//查询状态为待付款状态下的订单总数
		OrderQueryParams params = new OrderQueryParams();
		params.setOrderStatus(OrderStatusEnum.TO_PAY.getCode());
		//将时间（秒）清零，防止出现秒级的时间误差
		params.setGmtCreateStart(DateUtils.addMinutes(DateUtils.setSeconds(new Date(), 0), -20));
		params.setGmtCreateEnd(DateUtils.addMinutes(DateUtils.setSeconds(new Date(), 0), -15));
       //查询短信模板
      	QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.PAY_REMIND.getCode());
       //分页处理相关数据
		params.setOffset(0);
		params.setMax(pageSize);
        Long minId = 0L;
      	while (true) {
	        params.setMinId(minId);
        	List<QihoOrderEntity> orderList = qihoOrderDAO.findOrderPageListByParams(params);
    		if(CollectionUtils.isEmpty(orderList)){
    			break;
    		}
	        minId = orderList.get(orderList.size()-1).getId();
    		//循环处理数据
    		for(QihoOrderEntity entity:orderList){
    			try{
    		        Map<String, String> context = Maps.newHashMap();
    		        context.put("payLimitTime",format.format(DateUtils.addMinutes(entity.getGmtCreate(),30)));
    		        context.put("orderDetailUrl", ShortUrlUtil.getShortUrlByOrderId(entity.getOrderId()));
    				smsService.singleSend(template,context,entity.getMobile());
    			}catch(Exception e){
    				log.error("OrderTaskServiceImpl call remainOrderPay error, entity => {}",JSON.toJSONString(entity),e);
    			}
    		}
        }
	}

	@Override
	public void closeOverTimeOrder() {
		OrderQueryParams params = new OrderQueryParams();
		params.setOrderStatus(OrderStatusEnum.TO_PAY.getCode());
		params.setGmtCreateEnd(DateUtils.addHours(new Date(), -1));
		Long minId = 0l;
		while(true){
			params.setMinId(minId);
			params.setOffset(0);
        	params.setMax(pageSize);
        	List<QihoOrderEntity> orderList = qihoOrderDAO.findOrderPageListByParams(params);
    		if(CollectionUtils.isEmpty(orderList)){
    			break;
    		}
			minId = orderList.get(orderList.size()-1).getId();
    		for(QihoOrderEntity entity:orderList){
    			try{
    		       //查询一下订单的支付单信息
    				List<FundOrderDto> fundOrderList = fundOrderService.findByOrderIdAndBizType(entity.getOrderId(), FundBizTypeEnum.PAY.getCode());
    			   //如果没有查询到支付单的信息，则是数据不正常，可以直接关闭订单
    				if(CollectionUtils.isEmpty(fundOrderList)){
    					orderService.closeOrder(entity.getOrderId(), "付款失败",null); 
    					continue;
    				}
    			   //根据支付单信息确认是否已经处于支付中的状态
    				FundOrderDto fundOrder = fundOrderList.get(0);
    				PayChannelProcessor payChannelProcessor = payChannelDecider.decidePayChannel(PayTypeEnum.getEnumByCode(fundOrder.getPayType()));
    				PayDto result = payChannelProcessor.queryPayResult(entity.getOrderId());
    				//如果查询返回订单是支付成功的，应该属于消息未接收到，需要更新订单状态为付款成功
    				if(result.isSuccess()){
    					AsyncAcceptParams param = new AsyncAcceptParams();
    					param.setAccount(result.getPayerId());
    					param.setOrderId(entity.getOrderId());
    					param.setOutSqeNo(result.getOutTradeNo());
    					param.setReceiptAmount(result.getPayAmt());
    					payChannelProcessor.processNotify(param);
    				//关闭未支付的订单和其关联的支付单	
    				}else{
    					orderService.closeOrder(entity.getOrderId(), FundStatusEnum.FAILED.getDesc(),fundOrder.getFundId()); 
    				}
    			}catch(Exception e){
    				log.error("OrderTaskServiceImpl call remainOrderPay error, entity => {}",JSON.toJSONString(entity),e);
    			}
    		}
		}
	}

	@Override
	public void syncFundOrderStatus() {
        FundPageParam param = new FundPageParam();
        param.setBizType(FundBizTypeEnum.REFUND.getCode());
        param.setFundStatus(FundStatusEnum.TO_PAY.getCode());
		Long minId = 0L;
        while (true) {
        	param.setMinId(minId);
        	List<FundOrderDto> list= fundOrderService.queryFundOrderToSync(param,pageSize,1);
        	if (CollectionUtils.isEmpty(list)) {
                break;
            }
	        minId = list.get(list.size()-1).getId();
        	for(FundOrderDto dto : list){
        		try{
			        doProcess(dto);
        		}catch(Exception e){
        			log.error("OrderTaskServiceImpl call syncFundOrderStatus error, entity => {}",JSON.toJSONString(dto),e);
        		}
        	}
        }
	}

	private void doProcess(FundOrderDto dto) {
		if(Objects.equal(PayTypeEnum.getEnumByCode(dto.getPayType()), PayTypeEnum.COD)){
			return;
		}
		PayChannelProcessor payChannelProcessor = payChannelDecider.decidePayChannel(PayTypeEnum.getEnumByCode(dto.getPayType()));
		RefundResult result = payChannelProcessor.refundQuery(dto.getOrderId(), dto.getFundId());
		//调用成功
		if(null == result.getIsSuccess() || !result.getIsSuccess()){
			return;
		}
		//退款处理中,下次在查询
		if(StringUtils.equals(result.getRefundStatus(),RefundResult.REFUND_STATUS_PROCESSING)){
			return;
		}
		dto.setFundStatus(StringUtils.equals(result.getRefundStatus(),RefundResult.REFUND_STATUS_SUCCESS) ? FundStatusEnum.SUCCESS.getCode() : FundStatusEnum.FAILED.getCode());
		dto.setOutSeqNo(StringUtils.isBlank(result.getOutTradeNo()) ? dto.getOutSeqNo() : result.getOutTradeNo());
		fundOrderService.update(dto);
	}

}
