package com.qiho.center.biz.bo;

import java.util.Collections;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.enums.KuaiDi100LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.KuaiDi100OrderStatusEnum;
import com.qiho.center.api.util.BizLog;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.service.impl.LogisticsOrderServiceImpl;
import com.qiho.center.biz.service.logistics.Express100LogisticsService;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.entityd.qiho.logistics.Express100LogisticsEntity;
import com.qiho.center.common.params.kuaidi100.Result;
import com.qiho.center.common.params.kuaidi100.ResultItem;
import com.qiho.center.common.params.kuaidi100.TaskResponse;
import com.qiho.center.common.util.KuaiDi100Util;

/**
 * 快递100
 * Author: zhanghuifeng
 * Date:   11:58 2017/11/22
 */
@Service
public class KuaiDi100Bo {

    private static final Logger LOGGER = LoggerFactory.getLogger(KuaiDi100Bo.class);

    @Resource
    private LogisticsOrderServiceImpl logisticsOrderService;
    @Resource
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;
    @Resource
    private Express100LogisticsService express100LogisticsService;
    @Resource
    private LogisticsService logisticsService;

    /**
     * 处理快递100物流推送的信息
     *
     * @param com    快递100中快递公司编码
     * @param postId 快递单号
     * @param status 快递单当前状态
     * @return boolean
     */
    public Boolean dealKuaiDi100Status(String com, String postId, String status, String condition) {

        BizLog.log("快递100推送信息：快递公司编码：{}， 快递单号:{}，快递单当前状态: {}, 快递单明细状态标记:{}", com, postId, status, condition);

        LogisticsOrderDto dto = new LogisticsOrderDto();
        KuaiDi100LogisticsOrderStatusEnum kuaiDi100LogisticsOrderStatusEnum = KuaiDi100LogisticsOrderStatusEnum
            .getByState(status);
        if (kuaiDi100LogisticsOrderStatusEnum != null) {
            dto.setLogisticsStatus(kuaiDi100LogisticsOrderStatusEnum.getDesc());
        } else {
            dto.setLogisticsStatus(status);
        }
        KuaiDi100OrderStatusEnum statusEnum = KuaiDi100OrderStatusEnum.getByKuaiDi100Status(status);
        if (statusEnum == null) {
            return true;
        }
        //快递100拒签状态特殊处理
        if (statusEnum == KuaiDi100OrderStatusEnum.UNDER_WAY && "JQ001".equals(condition)) {
            dto.setOrderStatus(KuaiDi100OrderStatusEnum.REJECTED.getStatus());
        } else {
            dto.setOrderStatus(statusEnum.getStatus());
        }
        dto.setPostId(postId);
        Express100LogisticsEntity express100LogisticsEntity = express100LogisticsService.findByCode(com);
        if (express100LogisticsEntity == null) {
            LOGGER.error("快递100推送的快递公司代码表中不存在:", com);
            return false;
        }
        BaiqiLogisticsEntity baiqiLogisticsEntity = logisticsService
            .findById(express100LogisticsEntity.getBaiqiLogisticsId());
        if (baiqiLogisticsEntity == null) {
            LOGGER.error("快递100推送的快递公司所映射的百奇物流公司不存在:", express100LogisticsEntity.getBaiqiLogisticsId());
            return false;
        }
        dto.setLogisticsCode(baiqiLogisticsEntity.getLogisticsCode());
        logisticsOrderService.update(dto);
        return true;
    }

    /**
     * 向快递100订阅
     *
     * @param list
     */
    public void dealpostOrder(List<LogisticsOrderEntity> list) {
        //以每20单为一组
        List<List<LogisticsOrderEntity>> groupList = Lists.partition(list, 20);

        Express100LogisticsEntity express100LogisticsEntity = null;

        for (List<LogisticsOrderEntity> listFinal : groupList) {
            long start = System.currentTimeMillis();
            for (LogisticsOrderEntity entity : listFinal) {
                express100LogisticsEntity = express100LogisticsService
                    .findByBaiqiLogisticsCode(entity.getLogisticsCode());
                if (express100LogisticsEntity == null) {
                    LOGGER.error("在快递100物流表中不存在百奇快递公司代码:{}", entity.getLogisticsCode());
                } else {
                    postOrder(express100LogisticsEntity.getLogisticsCode(), entity.getPostId());
                }
            }
            //快递100最高请求数是20单/秒
            long end = System.currentTimeMillis();
            long useTime = end - start;
            if (useTime < 1000) {
                try {
                    Thread.sleep(1000 - useTime);
                } catch (InterruptedException e) {
                    LOGGER.warn("Thread sleep error ", e);
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    /**
     * 向快递100订阅
     *
     * @param list
     */
    public void dealpostOrders(List<LogisticsOrderDo> list) {
        //以每20单为一组
        List<List<LogisticsOrderDo>> groupList = Lists.partition(list, 20);

        Express100LogisticsEntity express100LogisticsEntity = null;

        for (List<LogisticsOrderDo> listFinal : groupList) {
            long start = System.currentTimeMillis();
            for (LogisticsOrderDo logisticsOrderDo : listFinal) {
                express100LogisticsEntity = express100LogisticsService
                        .findByBaiqiLogisticsCode(logisticsOrderDo.getLogisticsCode());
                if (express100LogisticsEntity == null) {
                    LOGGER.error("在快递100物流表中不存在百奇快递公司代码:{}", logisticsOrderDo.getLogisticsCode());
                } else {
                    postOrder(express100LogisticsEntity.getLogisticsCode(), logisticsOrderDo.getPostId());
                }
            }
            //快递100最高请求数是20单/秒
            long end = System.currentTimeMillis();
            long useTime = end - start;
            if (useTime < 1000) {
                try {
                    Thread.sleep(1000 - useTime);
                } catch (InterruptedException e) {
                    LOGGER.warn("Thread sleep error ", e);
                    Thread.currentThread().interrupt();
                }
            }
        }
    }


    private void postOrder(String kuaiDi100Code, String postId) {
        TaskResponse response = KuaiDi100Util.postOrder(kuaiDi100Code, postId);
        if (response != null && StringUtils.isNotBlank(response.getReturnCode())) {
            qihoLogisticsOrderDAO.updatePostReturnCode(Integer.parseInt(response.getReturnCode()), postId);
        }
    }

    public List<LogisticsProcessDto> queryWaybillTrace(String logisticsCode, String postId) {
        if (StringUtils.isBlank(postId)) {
            return Collections.emptyList();
        }

        // 查询快递100公司信息
        Express100LogisticsEntity express100LogisticsEntity = express100LogisticsService.findByBaiqiLogisticsCode(logisticsCode);
        if (express100LogisticsEntity == null) {
            LOGGER.error("获取不到快递100物流公司信息");

            return Collections.emptyList();
        }

        List<LogisticsProcessDto> resultList = null;
        try {

            // 获取快递100查询结果
            Result result = KuaiDi100Util.queryWaybillTrace(express100LogisticsEntity.getLogisticsCode(), postId);

            if (result != null && CollectionUtils.isNotEmpty(result.getData())) {

                //
                List<ResultItem> list = result.getData();

                resultList = Lists.newArrayListWithCapacity(list.size());
;
                for (ResultItem resultItem : list) {
                    LogisticsProcessDto logisticsProcessDto = new LogisticsProcessDto();
                    logisticsProcessDto.setWaybillNo(postId);
                    logisticsProcessDto.setUploadTime(resultItem.getFtime());
                    logisticsProcessDto.setProcessInfo(resultItem.getContext());

                    resultList.add(logisticsProcessDto);
                }
            }
        } catch (Exception e) {
            LOGGER.warn("快递100运单走件流程异常， postId = " + postId, e);
        }

        return resultList;
    }
}
