package com.qiho.center.biz.job;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.merchant.MerchantSimpleDto;
import com.qiho.center.common.dao.QihoMerchantDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.util.DingTalkUtil;


/**
 * 订单物流监控提醒
 * @author: wangjin
 * @create 2018-05-11 14:12
 **/
@Component
public class OrderLogisticsMonitorJob extends AbstractQihoSimpleElasticJob{

    private final static Logger LOGGER = LoggerFactory.getLogger(OrderLogisticsMonitorJob.class);

    @Autowired
    private QihoMerchantDAO qihoMerchantDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Value("${dingtalk.logistics.url}")
    private String dingUrl;


    /**
     * 五天前
     */
    private final static Integer FIVE_DAY = -5;

    /**
     * 5% 五天前
     */
    private final static Integer FIVE_PERCENT = 5;

    /**
     * 两天前
     */
    private final static Integer TWO_DAY = -2;

    /**
     * 10% 两天前
     */
    private final static Integer TEN_PERCENT = 10;

    /**
     * 1、商家为维度，以两天和五天为时间条件，查询商家订单的物流状态信息
     * 2、对于待发货与待审核的订单做占比统计
     * 3、符合一定占比的数据给以钉钉告警
     *
     * @param shardingContext
     */
    @Override
    protected void doProcess(JobExecutionMultipleShardingContext shardingContext) {

        LOGGER.info("订单物流监控提醒，定时任务开始。。。");

        // 1、查询所有商家
        List<MerchantSimpleDto> merchantsList = qihoMerchantDAO.findAllIdAndNames();
        if(CollectionUtils.isEmpty(merchantsList)){
            return;
        }

        // 2、构建告警信息
        List<String> alertMsgList = buildMsgList(merchantsList);

        // 3、发送告警
        if(CollectionUtils.isNotEmpty(alertMsgList)){
            sendDingTalkMsg(alertMsgList);
        }

        LOGGER.info("订单物流监控提醒，定时任务结束。。。");

    }

    /**
     * 发送钉钉报警
     *
     * @param alertMsgList
     */
    private void sendDingTalkMsg(List<String> alertMsgList) {

        StringBuilder msg = new StringBuilder();

        msg.append("产生一条新的订单监控预警！！预警内容如下：\n").append("日期：")
            .append(getDay(LocalDate.now().toDate())).append("\n")
            .append(StringUtils.join(alertMsgList,""));

        DingTalkUtil.sendTextMessageWith(String.valueOf(msg), dingUrl,true);
    }

    /**
     * 构建告警信息列表
     *
     *  查询 两天前与五天前的 订单物流状态
     *
     *  计算占比
     *
     *  构建消息
     *
     * @param merchantsList
     * @return
     */
    private List<String> buildMsgList(List<MerchantSimpleDto> merchantsList) {

        List<String> alertMsgList = Lists.newArrayListWithExpectedSize(merchantsList.size());

        // 循环每个商家处理
        merchantsList.forEach(merchant ->{
            StringBuilder content = new StringBuilder();
            content.append("商家id: ").append(merchant.getId()).append(", 商家名称: ").append(merchant.getMerchantName()).append(", ");

            boolean needAlert = false;

            // 两天前
            int twoDayPercent = searchPercent(merchant, TWO_DAY);
            // 两天前的占比大于10% --〉告警
            if(twoDayPercent > TEN_PERCENT){
                needAlert = true;
                // 两天前的日期
                String date = getDay(getStartTime(TWO_DAY));
                // 添加告警内容
                content.append(date).append("未处理订单").append(twoDayPercent).append("% ");
            }

            // 五天前
            int fiveDayPercent = searchPercent(merchant, FIVE_DAY);
            // 5天前的占比大于5% --〉告警
            if(fiveDayPercent > FIVE_PERCENT){
                needAlert = true;
                // 五天前的日期
                String date = getDay(getStartTime(FIVE_DAY));
                // 添加告警内容
                content.append(date).append("未处理订单").append(fiveDayPercent).append("% ");
            }

            if(needAlert){
                alertMsgList.add(content.append("\n").toString());
            }
        });

        return alertMsgList;
    }

    /**
     * 获得开始/结束时间
     * @param day -2表示两天前  -5表示5天前
     * @return
     */
    private static Date getStartTime(int day){
        Calendar cal = Calendar.getInstance();

        cal.add(Calendar.DATE, day);
        // 时
        cal.set(Calendar.HOUR_OF_DAY, 0);
        // 分
        cal.set(Calendar.MINUTE, 0);
        // 秒
        cal.set(Calendar.SECOND, 0);

        return cal.getTime();
    }

    private static String getDay(Date date) {
        if (date == null) {
            return "";
        }
        return new SimpleDateFormat("MM月dd日").format(date);
    }

    /**
     * 查询是否超过百分比
     *
     * @param merchant
     */
    private int searchPercent(MerchantSimpleDto merchant, int day){
        // 商家id
        Long merchantId = merchant.getId();

        // 开始时间
        Date startDay = getStartTime(day);

        // 结束时间
        Date endDay = getStartTime(day + 1);

        // 总订单数
        int dayCountAll = qihoOrderSnapshotDAO.countByMerchantIdAndOrderStatusAndTime(merchantId,"", startDay, endDay);

        // 待审核的订单数
        int dayCountAPPROVE = qihoOrderSnapshotDAO.countByMerchantIdAndOrderStatusAndTime(merchantId,"TO_APPROVE", startDay, endDay);

        // 代发货的订单数
        int dayCountSend = qihoOrderSnapshotDAO.countByMerchantIdAndOrderStatusAndTime(merchantId,"TO_SEND", startDay,endDay);

        // 计算百分比
        int execPercent;
        if(dayCountAll == 0){
            execPercent = 0;
        }else {
            execPercent =  (dayCountAPPROVE + dayCountSend) * 100 / dayCountAll;
        }

        return execPercent;

    }

}
