package com.qiho.center.biz.paychannel.pay;

import com.google.common.collect.Maps;
import com.qiho.center.api.constant.WeChatConstant;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.util.StringRandUtil;
import com.qiho.center.api.util.WechatPayAppUtil;
import com.qiho.center.api.util.XmlTranformUtil;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.HttpClientUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * Created by qianjue on 2017/8/9.
 */
@Component
public class WeChatPayProcessor extends WechatPayBaseProcessor implements InitializingBean{

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	private static String suffix = "w";

	@Override
	public void afterPropertiesSet() throws Exception {
		PayChannelDecider.registPaychannel(getPayType(), this);
	}

	@Override
	public String payExecute(String orderId,Map<String,String> params) {
		try{
			//封装一下支付相关的基础信息
			Map<String,String> data = buildPayParams(orderId,params);
			String parmaStr = XmlTranformUtil.mapToXml(data);
			//发送支付请求
			String httpResultStr = HttpClientUtil.postData(WeChatConstant.Url.PAY_URL,parmaStr);
			//解析返回的支付结果
			Map<String,String> resultMap = XmlTranformUtil.xmlToMap(httpResultStr);
			//如果返回成功有结果,则返回前端跳转的url地址
			//通信返回成功结果
			if(WeChatConstant.ResponseCode.SUCCESS.equals(resultMap.get("return_code"))){
				//如果返回结果正常,
				if(WeChatConstant.ResponseCode.SUCCESS.equals(resultMap.get("result_code"))){
					return resultMap.get("mweb_url");
				}else{
					AppLogUtil.error(logger,"微信支付发起失败,支付报文请求报文={},返回报文={}",parmaStr,httpResultStr);
				}
			}else{
				AppLogUtil.error(logger,"微信支付发起失败,支付报文请求报文={},返回报文={}",parmaStr,httpResultStr);
			}
		}catch (Exception e){
			AppLogUtil.error(logger,"唤起微信支付异常,orderId = {}",orderId,e);
		}
		return StringUtils.EMPTY;
	}

	private Map<String,String> buildPayParams(String orderId,Map<String,String>params) {
		OrderDto orderDto = orderService.findByOrderId(orderId);
		if (null == orderDto){
			throw new QihoException("无效的订单ID");
		}
		Map<String,String> dataMap = Maps.newHashMap();
		dataMap.put(WeChatConstant.WechatKey.APP_ID,WechatPayAppUtil.getAppId());
		dataMap.put(WeChatConstant.WechatKey.MCH_ID,WechatPayAppUtil.getMchId());
		dataMap.put(WeChatConstant.WechatKey.NONCE_STR, StringRandUtil.getRandomString(32));
		dataMap.put(WeChatConstant.WechatKey.BOBY,"奇货—"+StringUtils.trim(orderDto.getOrderItem().getItemName()));
		//增加后缀,用来区分是webH5发起的支付请求
		dataMap.put(WeChatConstant.WechatKey.OUT_TRADE_NO,orderId+suffix);
		dataMap.put(WeChatConstant.WechatKey.TOTAL_FEE,orderDto.getOrderAmt().toString());
		dataMap.put(WeChatConstant.WechatKey.SPBILL_CREATE_IP,params.get("ip"));
		dataMap.put(WeChatConstant.WechatKey.NOTIFY_URL,WechatPayAppUtil.getNotifyUrl());
		dataMap.put(WeChatConstant.WechatKey.TRADE_TYPE,WeChatConstant.WechatDefaultValue.TRADE_TYPE_H5);
		dataMap.put(WeChatConstant.WechatKey.SCENE_INFO,WeChatConstant.WechatDefaultValue.SCENE_INFO_H5);
		dataMap.put(WeChatConstant.WechatKey.SIGN,sign(dataMap,WechatPayAppUtil.getSecret()));
		return dataMap;
	}


	@Override
	PayTypeEnum getPayType() {
		return PayTypeEnum.WECHATPAY;
	}

	@Override
	public RefundResult refundQuery(String orderId, String fundId) {
		RefundResult result =  super.refundQuery(orderId+suffix,fundId);
		result.setOrderId(orderId);
		return result;
	}

	@Override
	public PayDto queryPayResult(String orderId) {
		return super.queryPayResult(orderId+suffix);
	}

	@Override
	public void refund(String orderId, String fundId, Integer refundAmt) {
		super.refund(orderId+suffix,fundId,refundAmt);
	}
}
