package com.qiho.center.biz.remoteservice.impl.campaign;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSON;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.campaign.ChannelDto;
import com.qiho.center.api.dto.campaign.MediaPlanDto;
import com.qiho.center.api.enums.ResultErrorEnum;
import com.qiho.center.api.params.ChannelParams;
import com.qiho.center.api.params.MediaPlanParams;
import com.qiho.center.api.remoteservice.campaign.RemoteChannelService;
import com.qiho.center.biz.service.ChannelService;
import com.qiho.center.biz.service.MediaPlanService;

import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * Created by danke on 2017/6/16.
 */
@RestController
public class RemoteChannelServiceImpl implements RemoteChannelService {

    private static final Logger logger = LoggerFactory.getLogger(RemoteChannelServiceImpl.class);

    @Resource
    private ChannelService channelService;

    @Resource
    private MediaPlanService mediaPlanService;

    @Override
    public DubboResult<PagenationDto<ChannelDto>> queryChannelByPage(ChannelParams params, Integer pageNum,
        Integer pageSize) {
        if (null == params)
            return DubboResult.failResult(ResultErrorEnum.PARAMS_ERROR.getMessage());
        try {
            PagenationDto<ChannelDto> pagenationDto = channelService.findByQuery(params, pageNum, pageSize);
            return DubboResult.successResult(pagenationDto);
        } catch (Exception e) {
            logger.error("渠道分页查询异常 params={}", params.toString(), e);
            return DubboResult.failResult(ResultErrorEnum.SERVICE_ERROR.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> deleteChannelById(Long id, String updator) {
        if (null == id || id < 0)
            return DubboResult.failResult(ResultErrorEnum.PARAMS_ERROR.getMessage());
        try {
            ChannelDto resultDto = channelService.findById(id);
            if (null == resultDto)//id有误,需更新记录为null
                return DubboResult.failResult("需要删除的渠道不存在");
            Integer resultIte = channelService.deleteChannelById(id, updator);
            if (null != resultIte && resultIte > 0) {
                //级联删除媒体计划,删除渠道失败则直接返回异常
                MediaPlanParams planParams = new MediaPlanParams();
                planParams.setChannelId(id);
                PagenationDto<MediaPlanDto> mediaPlanDtos = mediaPlanService.findByQuery(planParams, null, null);
                if (!CollectionUtils.isEmpty(mediaPlanDtos.getList())) {
                    mediaPlanDtos.getList().stream().forEach(e -> mediaPlanService.deleteMediaById(e.getId(), updator));
                }
                return DubboResult.successResult(Boolean.TRUE);
            }
            return DubboResult.failResult(ResultErrorEnum.RESULT_ERROR.getMessage());
        } catch (Exception e) {
            logger.error("通过id删除渠道失败 id={}", id, e);
            return DubboResult.failResult(ResultErrorEnum.SERVICE_ERROR.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> updateChannelByParam(ChannelDto channelDto) {
        if (null == channelDto || null == channelDto.getId() || StringUtils.isBlank(channelDto.getChannelName()))
            return DubboResult.failResult(ResultErrorEnum.PARAMS_ERROR.getMessage());
        try {
            ChannelDto resultDto = channelService.findById(channelDto.getId());
            if (null == resultDto)//id有误,需更新记录为null
                return DubboResult.failResult("需要更新的渠道不存在");
            if (checkNameRepeat(channelDto))
                return DubboResult.failResult("渠道名称已经存在");
            Integer resultBase = channelService.updateChannel(channelDto);
            if (null != resultBase && resultBase > 0) {
                //级联更新渠道下的媒体计划的渠道名称
                MediaPlanDto mediaPlanDto = new MediaPlanDto();
                mediaPlanDto.setChannelId(channelDto.getId());
                mediaPlanDto.setChannelName(channelDto.getChannelName());
                mediaPlanService.updateMediaPlanByChannelId(mediaPlanDto);
                return DubboResult.successResult(Boolean.TRUE);
            }
            return DubboResult.failResult(ResultErrorEnum.RESULT_ERROR.getMessage());
        } catch (Exception e) {
            logger.error("更新渠道信息失败 channelDto={}", channelDto.getId(), e);
            return DubboResult.failResult(ResultErrorEnum.SERVICE_ERROR.getMessage());
        }
    }

    @Override
    public DubboResult<Integer> insertChannel(ChannelDto channelDto) {
        if (valuteParamsNull(channelDto))
            return DubboResult.failResult(ResultErrorEnum.PARAMS_ERROR.getMessage());
        if (checkNameRepeat(channelDto))
            return DubboResult.failResult("渠道名称已经存在");
        try {
            Integer channelId = channelService.insertChannel(channelDto);
            if (null != channelId || channelId > 0)
                return DubboResult.successResult(channelId);
            return DubboResult.failResult(ResultErrorEnum.RESULT_ERROR.getMessage());
        } catch (Exception e) {
            logger.error("新增渠道信息失败 channelDto={}", channelDto.getChannelName(), e);
            return DubboResult.failResult(ResultErrorEnum.SERVICE_ERROR.getMessage());
        }
    }

    @Override
    public DubboResult<Integer> countByParam(ChannelParams params) {
        if (null == params)
            return DubboResult.failResult("查询条件参数为空");
        try {
            Integer total = channelService.countChannelByQuery(params);
            return DubboResult.successResult(total);
        } catch (Exception e) {
            logger.error("条件查询记录数error, params={}", JSON.toJSONString(params), e);
            return DubboResult.failResult("条件查询记录数error");
        }
    }

    /**
     * 校验新增操作dto数据完整
     * 渠道不为null信息
     * id,name,creator,status,gmt_created
     */
    private Boolean valuteParamsNull(ChannelDto channelDto) {
        if (null == channelDto)
            return Boolean.TRUE;
        if (null != channelDto.getId() && channelDto.getId() > 0)
            return Boolean.TRUE;
        if (StringUtils.isBlank(channelDto.getChannelName()) || StringUtils.isBlank(channelDto.getCreator()))
            return Boolean.TRUE;
        return Boolean.FALSE;
    }

    /**
     * 校验对应渠道下的媒体计划名称是否重复
     */
    private Boolean checkNameRepeat(ChannelDto channelDto) {
        ChannelParams channelParams = new ChannelParams();
        channelParams.setChannelName(channelDto.getChannelName());
        PagenationDto<ChannelDto> channelResult = channelService.findByQuery(channelParams, null, null);
        if (CollectionUtils.isEmpty(channelResult.getList()))
            return Boolean.FALSE;
        Long resultCopy = channelResult.getList().stream()
            .filter(e -> channelDto.getChannelName().equals(e.getChannelName())).count();
        if (resultCopy <= 0)
            return Boolean.FALSE;
        return Boolean.TRUE;
    }
}
