package com.qiho.center.biz.runnable;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.enums.DealTypeEnum;
import com.qiho.center.api.enums.blacklist.BaiqiBlackListTypeEnum;
import com.qiho.center.api.enums.blacklist.BlackListUseTypeEnum;
import com.qiho.center.api.enums.blacklist.SourceTypeEnum;
import com.qiho.center.api.params.blacklist.BlackListBatchDealParam;
import com.qiho.center.common.daoh.qiho.blacklist.BaiqiBlackListMapper;
import com.qiho.center.common.entityd.qiho.blacklist.BaiqiBlackListEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author chensong
 * @create 2018-04-28 14:28
 **/
@Component
@Scope("prototype")
public class BaiqiBlackListRunnable implements Runnable{

    private Logger logger = LoggerFactory.getLogger(BaiqiBlackListRunnable.class);

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    private List<BlackListBatchDealParam> params = Lists.newArrayList();

    private BoundHashOperations<String, String, Object> ops;

    private Map<String,String> remarkMap = Maps.newHashMap();

    /** 商家id */
    private Long merchantId;

    /** 操作人 */
    private String operator;

    @Resource
    private BaiqiBlackListMapper baiqiBlackListMapper;

    @Override
    public void run() {
        int failCount = 0;
        try{
            // 要新增的黑名单数据
            List<String> valueList = params.stream().map(BlackListBatchDealParam::getBkValue).collect(Collectors.toList());

            // 要新增的黑名单数据已存在数据库的
            List<String> existValueList = baiqiBlackListMapper.findByValueAndMerchantId(valueList,this.merchantId);
            if (CollectionUtils.isNotEmpty(existValueList)) {

                // 在新增的数据中去除 已存在数据库的
                failCount = failCount + existValueList.size();
                valueList.removeAll(existValueList);
            }
            if (CollectionUtils.isEmpty(valueList)) {
                return ;
            }

            // 将数据写入数据库
            List<BaiqiBlackListEntity> insertData = bulidInsertParam(valueList);
            int successResult = baiqiBlackListMapper.insertBatch(insertData);
            ops.increment("successCount", successResult);

        } catch (Exception e){
            logger.warn("批量添加黑名单异常,黑名单数据为:"+params.toString(),e);

        } finally {
            ops.increment("failCount", failCount);
            ops.increment("taskCount", -1L);
        }
    }

    /**
     * 将list转成黑名单数据对象
     * @param list
     * @return
     */
    private List<BaiqiBlackListEntity> bulidInsertParam(List<String> list){
        return Lists.transform(list, e->{
            BaiqiBlackListEntity entity = new BaiqiBlackListEntity();
            entity.setBkType(BaiqiBlackListTypeEnum.PHONE.getNum());
            entity.setBkValue(e);
            entity.setDealType(DealTypeEnum.FORBID_SUBMIT.getNum());
            entity.setOperator(this.operator);
            entity.setMerchantId(this.merchantId);
            entity.setRemark(remarkMap.get(e));
            entity.setUseType(merchantId.longValue() == 0L ? BlackListUseTypeEnum.ALL.getNum():BlackListUseTypeEnum.PART.getNum());
            entity.setSourceType(SourceTypeEnum.MANUAL.getNum());
            return entity;
        });
    }

    public void setProgressKey(String progressKey) {
        this.ops = redisTemplate.boundHashOps(progressKey);
    }

    public void setParams(List<BlackListBatchDealParam> params) {
        this.params.addAll(params);
    }

    public void setMerchantId(Long merchantId) {
        this.merchantId = merchantId;
    }

    public void setOperator(String operator) {
        this.operator = operator;
    }

    /**
     * 对传入数据进行简单校验
     * @return
     */
    public Integer enableFilter() {
        List<BlackListBatchDealParam> newList = Lists.newArrayList();
        for (BlackListBatchDealParam param : this.params) {
            if (StringUtils.isBlank(param.getBkValue())) {
                continue;
            }
            newList.add(param);
            remarkMap.put(param.getBkValue(),param.getRemarks());
        }
        params.clear();
        params.addAll(newList);
        return newList.size();
    }



}
