package com.qiho.center.biz.runnable;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.qiho.center.api.dto.order.DeliveryRecordDto;
import com.qiho.center.api.enums.DeliveryRecordStateEnum;
import com.qiho.center.api.params.OrderFileDeliveryParam;
import com.qiho.center.api.util.BizLog;
import com.qiho.center.biz.bo.ErpOrderBo;
import com.qiho.center.biz.service.order.OrderFileDeliveryService;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.entityd.qiho.order.DeliveryRecordEntity;

/**
 * 批量上传发物流单号-批量发货
 *
 * @Created by zhangshun on 2018-03-10 15:20:17
 */
@Component
@Scope("prototype")
public class OrderFileCancelDeliveryRunnable implements Runnable {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrderFileCancelDeliveryRunnable.class);

    /**
     * 任务ID
     */
    private final Long recordId;

    /**
     * 商家ID
     */
    private final Long merchantId;

    /**
     * 总条数
     */
    private final Integer totalCount;

    /**
     * 物流记录
     */
    private final List<OrderFileDeliveryParam> deliveryBeanList;

    @Autowired
    private ErpOrderBo erpOrderBo;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Resource
    private OrderFileDeliveryService orderFileDeliveryService;

    public OrderFileCancelDeliveryRunnable(DeliveryRecordDto recordDto, List<OrderFileDeliveryParam> deliveryBeanList) {
        this.recordId = recordDto.getId();
        this.merchantId = recordDto.getMerchantId();
        this.totalCount = recordDto.getTotalCount();
        this.deliveryBeanList = deliveryBeanList;
    }

    @Override
    public void run() {

        if (CollectionUtils.isEmpty(deliveryBeanList) || Objects.equals(null, recordId) || Objects
            .equals(null, merchantId)) {
            BizLog.log("[取消发货]参数错误，数据、任务ID、商家ID为空");
            return;
        }

        if (Objects.equals(null, totalCount) || totalCount.equals(0)) {
            BizLog.log("[取消发货]参数错误，总条数为空");
            return;
        }

        int failedCount = 0;
        int succeedCount = 0;
        Set<Integer> failIds = new HashSet<>();

        try {

            for (OrderFileDeliveryParam bean : deliveryBeanList) {
                try {

                    //检验业务参数
                    boolean checkFlag = this.checkParam(bean);
                    if (!checkFlag) {
                        failedCount++;
                        failIds.add(bean.getRowNumber());
                        continue;
                    }

                    // 主逻辑
                    erpOrderBo.cancelDeliver(bean.getErpId());

                    succeedCount++;

                    BizLog
                        .log("[取消发货]成功. {}, recordId:{}, merchantId:{}, {}", Thread.currentThread().getName(), recordId,
                            merchantId, bean);

                } catch (Exception e) {
                    failedCount++;
                    failIds.add(bean.getRowNumber());
                    LOGGER.error("[取消发货]失败. {}, recordId:{}, merchantId:{}, {}", Thread.currentThread().getName(),
                        recordId, merchantId, bean, e);
                }
            }
        } finally {
            this.completeHandler(failedCount, succeedCount, failIds);
        }

    }

    /**
     * 检查参数
     *
     * @param bean
     * @return
     */
    public boolean checkParam(OrderFileDeliveryParam bean) {

        if (StringUtils.isBlank(bean.getErpId()) || Objects.isNull(bean.getRowNumber())) {
            BizLog.log("[取消发货]失败，参数验证失败");
            return false;
        }

        if (!isMerchantByOrderId(bean.getErpId(), String.valueOf(merchantId))) {
            BizLog.log("[取消发货]订单与商家不对应，erpId:{}, merchantId:{}", bean.getErpId(), merchantId);
            return false;
        }

        return true;
    }

    /**
     * 判断订单是否这个商家
     *
     * @param erpId
     * @return
     */
    private boolean isMerchantByOrderId(String erpId, String merchantId) {

        // 查询ERP订单信息
        final QihoErpOrderEntity erpOrderEntity = qihoErpOrderDAO.findByErpId(erpId);
        if (Objects.isNull(erpOrderEntity) || Objects.isNull(erpOrderEntity.getOrderId())) {
            return Boolean.FALSE;
        }

        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(erpOrderEntity.getOrderId());
        if (null == entity) {
            return Boolean.FALSE;
        }
        if (!StringUtils.equals(String.valueOf(entity.getMerchantId()), merchantId)) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    private void completeHandler(int failedCount, int succeedCount, Set<Integer> failIds) {

        DeliveryRecordEntity entity = new DeliveryRecordEntity();

        entity.setId(recordId);
        entity.setFailedCount(failedCount);
        entity.setSucceedCount(succeedCount);
        entity.setUpdatedCount(0);
        entity.setRemark(getCancelFailMsg(failIds));
        entity.setState(DeliveryRecordStateEnum.COMPLETE.getCode());

        // 更新数据库
        if (!orderFileDeliveryService.updateByIdSelective(entity)) {
            LOGGER.error("[取消发货]批量上传发物流单号-更新数据失败, id:{}", entity.getId());
        }
    }

    private String getCancelFailMsg(Set<Integer> failIds) {
        String failMsg = StringUtils.join(failIds.iterator(), ",");
        if (2040 < failMsg.length()) {
            failMsg = failMsg.substring(0, 2040) + "......";
        }
        return failMsg;
    }

}
