package com.qiho.center.biz.runnable;

import com.google.common.collect.Lists;
import com.qiho.center.api.enums.ErpStatusEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.YTOLogisticsStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.WdtOrderAuditParams;
import com.qiho.center.biz.bo.KuaiDi100Bo;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoItemMerchantDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.SequenceNoBuilder;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by weny.cai on 2017/11/19.
 */
@Component
@Scope("prototype")
public class WdtAuditOrderRunnable implements Runnable {

    private static final Logger LOGGER = LoggerFactory.getLogger(WdtAuditOrderRunnable.class);

    private static final String NUM_AND_LETTER_REG_EX = "^[A-Za-z0-9]{1,40}$";

    private final List<WdtOrderAuditParams> params = Lists.newArrayList();

    private BoundHashOperations<String, String, String> ops;

    private final ThreadLocal<List<LogisticsOrderEntity>> logisticsOrderEntityListLocal = new ThreadLocal() {
        @Override
        protected List<LogisticsOrderEntity> initialValue() {
            return new ArrayList<>();
        }
    };

    private final ThreadLocal<StringBuffer> failIdsLocal = new ThreadLocal() {
        @Override
        protected StringBuffer initialValue() {
            return new StringBuffer();
        }
    };

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    @Autowired
    private LogisticsOrderService logisticsOrderService;

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    @Autowired
    private SequenceNoBuilder builder;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private ErpOrderService erpOrderService;

    @Autowired
    private QihoItemMerchantDAO qihoItemMerchantDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private KuaiDi100Bo kuaiDi100Bo;

    @Override
    public void run() {

        long failCount = 0L;
        long updateCount = 0L;
        long successCount = 0L;

        try {

            int paramsSize = params.size();

            String failRowMsg = null;
            WdtOrderAuditParams param = null;

            for (int i = 0; i < paramsSize; i++) {
                try {
                    param = params.get(i);
                    failRowMsg = String.valueOf(param.getRowNum());

                    //检验业务参数
                    boolean checkFlag = this.checkParam(param);
                    if (!checkFlag) {
                        failCount++;
                        failIdsLocal.get().append(",").append(failRowMsg);
                        continue;
                    }

                    // 主逻辑
                    int result = this.auditOrder(param);

                    if (0 == result) {
                        updateCount++;
                    } else if (1 == result) {
                        failCount++;
                        failIdsLocal.get().append(",").append(failRowMsg);
                    } else if (2 == result) {
                        successCount++;
                    }

                } catch (Exception e) {
                    failCount++;
                    failIdsLocal.get().append(",").append(failRowMsg);
                    LOGGER.error("旺店通订单审核失败", e);
                }
            }
        } finally {
            // 更新缓存值
            this.updateCacheOps(failCount, updateCount, successCount, failIdsLocal.get().toString());
        }
    }

    /**
     * 更新缓存值
     *
     * @param failCount
     * @param updateCount
     * @param successCount
     */
    private void updateCacheOps(long failCount, long updateCount, long successCount, String failIds) {
        ops.increment("failCount", failCount);
        ops.increment("updateCount", updateCount);
        ops.increment("deliveryCount", successCount);
        ops.increment("taskCurrent", 1);

        String failMsgKey = "failMsg";

        if (StringUtils.isNotBlank(failIds)) {

            String fm = ops.get(failMsgKey);

            if (StringUtils.isBlank(fm)) {
                ops.put(failMsgKey, failIds.substring(1));
            } else {
                ops.put(failMsgKey, fm + "," + failIds.substring(1));
            }
        }

        int totalCount = Integer.parseInt(ops.get("totalCount"));

        int proCount = Integer.parseInt(ops.get("deliveryCount")) + Integer.parseInt(ops.get("failCount")) + Integer
            .parseInt(ops.get("updateCount"));

        // 处理完成
        if (proCount >= totalCount) {

            ops.put("success", "true");

            long endTime = System.currentTimeMillis();
            ops.put("endTime", String.valueOf(endTime));

            long startTime = Long.parseLong(ops.get("startTime"));

            if (endTime - startTime > 1000) {
                ops.put("useTime", ((endTime - startTime) / 1000) + "秒");
            } else {
                ops.put("useTime", (endTime - startTime) + "毫秒");
            }
        }


        //重新订阅快递100
        try {
            kuaiDi100Bo.dealpostOrder(logisticsOrderEntityListLocal.get());
        } catch (Exception e) {
            LOGGER.error("订阅快递100接口错误", e);
        }
    }

    /**
     * 检查参数
     *
     * @param param
     *
     * @return
     */
    public boolean checkParam(WdtOrderAuditParams param) {

        //格式不完整 不处理 返回失败
        if (StringUtils.isBlank(param.getPostId()) || StringUtils.isBlank(param.getErpId()) || StringUtils
            .isBlank(param.getMerchantId())) {
            return false;
        }

        // 验证物流信息
        if (StringUtils.isBlank(param.getLogisticsCode()) || StringUtils.isBlank(param.getLogisticsName())) {
            return false;
        }

        // 忽略大小写的写法
        Pattern pat = Pattern.compile(NUM_AND_LETTER_REG_EX, Pattern.CASE_INSENSITIVE);
        Matcher matcher = pat.matcher(param.getPostId().trim());
        // 字符串是否与正则表达式相匹配
        boolean rs = matcher.matches();
        if (!rs) {//数字和字母的检验
            return false;
        }

        //对商家进行校验 操作权限
        int count = qihoItemMerchantDAO.findErpOrderMerchant(param.getErpId(), param.getMerchantId());
        if (count == 0) {
            return false;
        }

        return true;
    }

    /**
     * 单个同步物流信息
     *
     * @param param
     *
     * @return 说明：
     * 0 = 更新成功
     * 1 = 失败
     * 2 = 确认成功
     */
    @Transactional(rollbackFor = Exception.class)
    public int auditOrder(WdtOrderAuditParams param) throws QihoException {

        // 查询ERP订单信息
        final QihoErpOrderEntity erpOrderEntity = qihoErpOrderDAO.findByErpId(param.getErpId());
        if (null == erpOrderEntity) {
            LOGGER.warn("查询ERP订单信息,ErpId:{}", param.getErpId());
            return 1;
        }

        LogisticsOrderEntity logisticsOrderEntity = qihoLogisticsOrderDAO.findByErpId(param.getErpId());

        //物流子订单号已经存在  则更新物流单号
        if (logisticsOrderEntity != null) {

            LogisticsOrderEntity entity = new LogisticsOrderEntity();
            entity.setLogisticsCode(param.getLogisticsCode());
            entity.setLogisticsName(param.getLogisticsName());
            entity.setPostId(param.getPostId());
            entity.setErpId(param.getErpId());

            //更新物流表和快照表的信息
            if (0 == qihoLogisticsOrderDAO.updateLogistics(entity)) {
                throw new QihoException("更新物流表和快照表的信息失败");
            }

            QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
            orderSnapshot.setPostId(param.getPostId());
            orderSnapshot.setLogisticsName(param.getLogisticsName());
            orderSnapshot.setLogisticsCode(param.getLogisticsCode());
            orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());

            // 更新订单快照信息
            if (0 == qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot)) {
                throw new QihoException("更新订单快照信息");
            }

            // 重新订阅快递100
            LogisticsOrderEntity logisticsOrderEntity1 = new LogisticsOrderEntity();
            logisticsOrderEntity1.setLogisticsCode(param.getLogisticsCode());
            logisticsOrderEntity1.setPostId(param.getPostId());

            logisticsOrderEntityListLocal.get().add(logisticsOrderEntity1);

            // 返回更新成功
            return 0;
        } else {
            final LogisticsOrderEntity entity = this.getLogisticsEntity(param, erpOrderEntity);

            //订单置为返回成功
            try {
                erpOrderService
                    .updateErpStatus(entity.getErpId(), ErpStatusEnum.SUCCESS, ErpStatusEnum.TO_NOTIFY, null);

                logisticsOrderService.createLogisticsOrderBatchDelivery(entity);
            } catch (Exception e) {
                throw new QihoException("确认发货处理错误");
            }

            // 返回确认成功
            return 2;
        }
    }

    /**
     * 生成物流子订单实体类
     *
     * @param wdtOrderAuditParam
     * @param erpOrder
     *
     * @return
     */
    private LogisticsOrderEntity getLogisticsEntity(WdtOrderAuditParams wdtOrderAuditParam,
        QihoErpOrderEntity erpOrder) {

        LogisticsOrderEntity entity = new LogisticsOrderEntity();

        entity.setErpId(wdtOrderAuditParam.getErpId());
        String postId = wdtOrderAuditParam.getPostId();

        int num = postId.indexOf(',');
        if (num > -1) {
            wdtOrderAuditParam.setPostId(postId.substring(0, num).trim());
        } else {
            entity.setPostId(postId.trim());
        }

        entity.setLogisticsCode(wdtOrderAuditParam.getLogisticsCode());
        entity.setLogisticsName(wdtOrderAuditParam.getLogisticsName());

        entity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getDesc());
        // 物流子订单初始化状态为接单
        entity.setOrderStatus(LogisticsOrderStatusEnum.ACCEPT.getCode());
        entity.setConsumerName(erpOrder.getConsumerName());
        entity.setItemId(erpOrder.getItemId());
        entity.setItemName(erpOrder.getItemName());
        entity.setItemNo(erpOrder.getItemNo());
        entity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, erpOrder.getItemId()));
        entity.setMobile(erpOrder.getMobile());
        entity.setOrderId(erpOrder.getOrderId());
        entity.setSkuNo(erpOrder.getSkuNo());
        entity.setProvince(erpOrder.getProvince());

        return entity;
    }

    public void setProgressKey(String progressKey) {
        this.ops = redisTemplate.boundHashOps(progressKey);
    }

    public void setParams(List<WdtOrderAuditParams> params) {
        this.params.addAll(params);
    }

}
