/**
 * Project Name:qiho-center-biz
 * File Name:ItemRecommendServiceImpl.java
 * Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月22日上午10:36:42
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 *
*/

package com.qiho.center.biz.service.impl;

import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.biz.event.ItemDeleteEvent;
import com.qiho.center.biz.service.ItemRecommendService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemRecommendDAO;
import com.qiho.center.common.entity.item.QihoItemRecommendEntity;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

/**
 * ClassName:ItemRecommendServiceImpl <br/>
 * Date:     2017年5月22日 上午10:36:42 <br/>
 * @author   zhanglihui
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service
@BizEventListener
public class ItemRecommendServiceImpl implements ItemRecommendService {

    @Autowired
    private QihoItemRecommendDAO qihoItemRecommendDAO;
    
    @Override
    public void recommendBatch(List<Long> itemIds) {
        if(CollectionUtils.isEmpty(itemIds)){
            return;
        }
        Set<Long> existItemIds = qihoItemRecommendDAO.findAllRecommendItemId();
        //去重
        Set<Long> insertItems = Sets.difference(Sets.newHashSet(itemIds),existItemIds);
        if(CollectionUtils.isEmpty(insertItems)){
            return;
        }
        int payload = qihoItemRecommendDAO.findMaxPlayload();
        List<QihoItemRecommendEntity> list = Lists.newArrayList();
        for(Long itemId : insertItems){
            QihoItemRecommendEntity entity = new QihoItemRecommendEntity();
            entity.setItemId(itemId);
            entity.setPayload(++payload);
            list.add(entity);
        }
        qihoItemRecommendDAO.insertBatch(list);
    }

    @Override
    public List<QihoItemRecommendEntity> findAllItemRecommend() {
        return qihoItemRecommendDAO.findAllRecommend();
    }

    @Override
    public int deleteItemRecommend(Set<Long> itemIds) {
        List<Long> ids = Lists.newArrayList(itemIds);
        return qihoItemRecommendDAO.deleteItemRecommend(ids);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public int sortItemRecommend(List<Long> itemIds) {

        List<QihoItemRecommendEntity> list = qihoItemRecommendDAO.findAllRecommend();
        if(itemIds.size()!=list.size()){
            throw new QihoException("传入id和推荐商品总数不一致");
        }
        ItemRecommendOrdering ordering = new ItemRecommendOrdering();
        ordering.setOrderItemIds(itemIds);
        list = Lists.transform(list,ordering);

        int count = 0;
        for(QihoItemRecommendEntity it : list){
            int ret = qihoItemRecommendDAO.updatePayload(it.getItemId(),it.getPayload());
            count = count + ret;
        }
        return count;
    }

    private class ItemRecommendOrdering implements Function<QihoItemRecommendEntity,QihoItemRecommendEntity>{

        private List<Long> orderItemIds;

        private int maxPayload;

        public void setOrderItemIds(List<Long> orderItemIds) {
            this.orderItemIds = orderItemIds;
            this.maxPayload = orderItemIds.size()-1;
        }

        @Override
        public QihoItemRecommendEntity apply(QihoItemRecommendEntity input) {
            int index = orderItemIds.indexOf(input.getItemId());
            input.setPayload(maxPayload-index);
            return input;
        }
    }

    @Subscribe
    public void itemDeleteListener(ItemDeleteEvent event){

        // 添加超时打印
        DBTimeProfile.enter("itemDeleteListener");

        //商品被删除的时候,清除该商品的推荐
        deleteItemRecommend(event.getItems());

        DBTimeProfile.release();
    }

}

