package com.qiho.center.biz.service.impl.blacklist;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.bklist.BlackListStrategyDto;
import com.qiho.center.api.dto.bklist.StrategyDto;
import com.qiho.center.api.enums.DealTypeEnum;
import com.qiho.center.api.enums.blacklist.BaiqiBlackListTypeEnum;
import com.qiho.center.api.enums.blacklist.BlackListUseTypeEnum;
import com.qiho.center.biz.service.blacklist.BlackListStrategyService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.daoh.qiho.blacklist.BaiqiBKStrategyMCMapper;
import com.qiho.center.common.daoh.qiho.blacklist.BaiqiBlackListStrategyMapper;
import com.qiho.center.common.entityd.qiho.blacklist.BaiqiBKStrategyMCEntity;
import com.qiho.center.common.entityd.qiho.blacklist.BaiqiBlackListStrategyEntity;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author chensong
 * @create 2018-04-25 14:58
 **/
@Service
public class BlackListStrategyServiceImpl implements BlackListStrategyService {

    @Autowired
    private BaiqiBlackListStrategyMapper baiqiBlackListStrategyMapper;

    @Autowired
    private MerchantService merchantService;

    @Autowired
    private BaiqiBKStrategyMCMapper baiqiBKStrategyMCMapper;

    @Override
    public List<BlackListStrategyDto> listByMerchantId(Long merchantId) {
        if (null == merchantId) {
            return Lists.newArrayList();
        }

        List<BaiqiBlackListStrategyEntity> strategyEntities = Lists.newArrayList();
        if (merchantId.longValue() == -1L) {

            // 获取所有的黑名单策略记录
            strategyEntities = baiqiBlackListStrategyMapper.listAllStrategy();

        } else if (merchantId.longValue() == 0L) {

            // 获取全部商家的策略记录
            strategyEntities = baiqiBlackListStrategyMapper.listCommonStrategy();

        } else if (merchantId.longValue() > 0L) {

            // 先从关联表查询
            List<BaiqiBKStrategyMCEntity> baiqiBKStrategyMCEntities = baiqiBKStrategyMCMapper.listByMerchantId(merchantId);
            if (CollectionUtils.isEmpty(baiqiBKStrategyMCEntities)) {
                return Lists.newArrayList();
            }
            // 再去黑名单策略表查询
            List<Long> strategyIds = baiqiBKStrategyMCEntities.stream().map(BaiqiBKStrategyMCEntity::getStrategyId).collect(Collectors.toList());
            strategyEntities = baiqiBlackListStrategyMapper.listByIds(strategyIds);
        }

        return convertStrategyToDto(strategyEntities);
    }


    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> createBlackStrategy(BlackListStrategyDto param) {
        // 区分是全部商家还是部分商家
        if (param.getUseType() == BlackListUseTypeEnum.ALL) {
            // 新建全部商家的黑名单策略
            return insertCommonStrategy(param);

        } else if (param.getUseType() == BlackListUseTypeEnum.PART){
            // 新增部分商家的黑名单策略
            return insertPartMerchatStrategy(param);

        } else {
            return ResultDto.failResult("适用商家类型错误");
        }
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> deleteBlackListStrategy(Long id, String name) {
        BaiqiBlackListStrategyEntity entity = baiqiBlackListStrategyMapper.findById(id);
        if (entity == null){
            return ResultDto.failResult("策略不存在");
        }
        // 删除黑名单策略
        baiqiBlackListStrategyMapper.deleteBlackListStrategy(id, name);
        if (entity.getUseType().intValue() == BlackListUseTypeEnum.PART.getNum()){
            // 同步删除策略和商家的关联数据
            baiqiBKStrategyMCMapper.deletedByStrategyId(id);
        }
        return ResultDto.successResult(Boolean.TRUE);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> updateBlackListStrategy(BlackListStrategyDto param) {
        Long strategyId = param.getId();
        if (strategyId == null) {
            return ResultDto.failResult("策略id不能为空");
        }
        //区分多种适用商家类型
        BaiqiBlackListStrategyEntity blackListStrategyEntity = baiqiBlackListStrategyMapper.findById(strategyId);
        if (null == blackListStrategyEntity) {
            return ResultDto.failResult("策略不存在");
        }

        // 原先的适用商家类型
        BlackListUseTypeEnum oldUseType =   BlackListUseTypeEnum.getByNum(blackListStrategyEntity.getUseType().intValue());
        BlackListUseTypeEnum newUseType = param.getUseType();
        if (oldUseType == BlackListUseTypeEnum.ALL && newUseType == BlackListUseTypeEnum.ALL) {

            // 依旧是全部是商家类型
            return allToAll(param);

        } else if (oldUseType == BlackListUseTypeEnum.ALL && newUseType == BlackListUseTypeEnum.PART){

            // 全部商家变成部分商家
            return allToPart(param);

        } else if(oldUseType == BlackListUseTypeEnum.PART && newUseType == BlackListUseTypeEnum.PART){

            // 依旧是全部商家类型
            return partToPart(param);

        } else if (oldUseType == BlackListUseTypeEnum.PART && newUseType == BlackListUseTypeEnum.ALL){

            // 部分商家变成全部商家
            return partToAll(param);

        }else {
            return ResultDto.failResult("适用商家参数错误");
        }

    }

    @Override
    public BlackListStrategyDto findById(Long id) {
        if (null == id || id.longValue() <= 0L) {
            return null;
        }
        BaiqiBlackListStrategyEntity entity = baiqiBlackListStrategyMapper.findById(id);
        if (null == entity){
            return null;
        }
        return convertStrategyToDto(Lists.newArrayList(entity)).get(0);
    }


    /**
     * 全部商家变成部分商家
     * @param blackListStrategyDto
     * @return
     */
    private ResultDto<Boolean> allToPart(BlackListStrategyDto blackListStrategyDto){
        List<Long> merchantIdList = blackListStrategyDto.getMerchantIdList();
        if (CollectionUtils.isEmpty(merchantIdList)){
            return ResultDto.failResult("传入的商家id为空");
        }

        // 校验商家是否已有关联关系
        List<BaiqiBKStrategyMCEntity> bkStrategyMCEntities = baiqiBKStrategyMCMapper.listByMerchantIds(merchantIdList);
        if (CollectionUtils.isNotEmpty(bkStrategyMCEntities)){
            List<Long> existIds = bkStrategyMCEntities.stream().map(BaiqiBKStrategyMCEntity::getMerchantId).collect(Collectors.toList());
            return ResultDto.failResult("商d为："+existIds.toString()+"已存在黑名单策略");
        }

        // 更新黑名单策略
        BaiqiBlackListStrategyEntity entity = convertStrategyToEntity(blackListStrategyDto);
        baiqiBlackListStrategyMapper.update(entity);
        // 插入黑名单策略商家关系表
        Long strategyId = blackListStrategyDto.getId();
        List<BaiqiBKStrategyMCEntity> bkStrategyMCEntityList = merchantIdList.stream().map(e -> {
            BaiqiBKStrategyMCEntity strategyMCEntity = new BaiqiBKStrategyMCEntity();
            strategyMCEntity.setMerchantId(e);
            strategyMCEntity.setStrategyId(strategyId);
            return strategyMCEntity;
        }).collect(Collectors.toList());
        baiqiBKStrategyMCMapper.insetBatch(bkStrategyMCEntityList);

        return ResultDto.successResult(Boolean.TRUE);
    }

    /**
     * 全部商家类型不变
     * @param blackListStrategyDto
     * @return
     */
    private ResultDto<Boolean> allToAll(BlackListStrategyDto blackListStrategyDto){
        BaiqiBlackListStrategyEntity entity = convertStrategyToEntity(blackListStrategyDto);
        baiqiBlackListStrategyMapper.update(entity);
        return ResultDto.successResult();
    }

    /**
     * 部分商家变成全部商家
     * @param blackListStrategyDto
     * @return
     */
    private ResultDto<Boolean> partToAll(BlackListStrategyDto blackListStrategyDto){
        // 校验全部商家的策略是否存在
        List<BaiqiBlackListStrategyEntity> commonStrategy = baiqiBlackListStrategyMapper.listCommonStrategy();
        if (CollectionUtils.isNotEmpty(commonStrategy)) {
            return ResultDto.failResult("适用全部商家的策略已存在");
        }
        // 更新黑名单策略
        baiqiBlackListStrategyMapper.update(convertStrategyToEntity(blackListStrategyDto));
        // 删除商家信息
        baiqiBKStrategyMCMapper.deletedByStrategyId(blackListStrategyDto.getId());
        return ResultDto.successResult(Boolean.TRUE);
    }

    /**
     * 部分商家类型不变
     * @param blackListStrategyDto
     * @return
     */
    private ResultDto<Boolean> partToPart(BlackListStrategyDto blackListStrategyDto){
        List<Long> newMerchantIds = blackListStrategyDto.getMerchantIdList();
        if (CollectionUtils.isEmpty(newMerchantIds)) {
            return ResultDto.failResult("传入的商家id列表为空");
        }

        // 获取原有的黑名单策略-商家关系
        List<Long> originMerchantIds = baiqiBKStrategyMCMapper.listByStrategyId(blackListStrategyDto.getId())
                .stream().map(BaiqiBKStrategyMCEntity::getMerchantId).collect(Collectors.toList());
        List<Long> copyOriginMerchantIds = originMerchantIds.stream().collect(Collectors.toList());

        // 新旧关系的交集
        copyOriginMerchantIds.retainAll(newMerchantIds);
        // 获取需要删除的商家id
        originMerchantIds.removeAll(copyOriginMerchantIds);
        // 获得需要新增的商家id
        newMerchantIds.removeAll(copyOriginMerchantIds);

        // 修改黑名单策略
        baiqiBlackListStrategyMapper.update(convertStrategyToEntity(blackListStrategyDto));
        // 移除不需要的商家id
        if (CollectionUtils.isNotEmpty(originMerchantIds)) {
            baiqiBKStrategyMCMapper.deleteByMerchantIds(originMerchantIds);
        }
        // 新增商家id
        if (CollectionUtils.isNotEmpty(newMerchantIds)){
            List<BaiqiBKStrategyMCEntity> insertEntities = newMerchantIds.stream().map(e -> {
                return new BaiqiBKStrategyMCEntity(blackListStrategyDto.getId(), e);
            }).collect(Collectors.toList());
            baiqiBKStrategyMCMapper.insetBatch(insertEntities);
        }

        return ResultDto.successResult(Boolean.TRUE);
    }

    /**
     * 新增部分商家的黑名单策略
     * @param param
     * @return
     */
    private ResultDto<Boolean> insertPartMerchatStrategy(BlackListStrategyDto param){
        if (CollectionUtils.isEmpty(param.getMerchantIdList())){
            return ResultDto.failResult("传入的商家id为空");
        }

        // 先查询商家是否已有黑名单策略
        List<BaiqiBKStrategyMCEntity> bkStrategyMCEntityList = baiqiBKStrategyMCMapper.listByMerchantIds(param.getMerchantIdList());
        if (CollectionUtils.isNotEmpty(bkStrategyMCEntityList)){
            List<Long> existMerchantIds = bkStrategyMCEntityList.stream().map(BaiqiBKStrategyMCEntity::getMerchantId).collect(Collectors.toList());
            return ResultDto.failResult("id为："+existMerchantIds.toString()+"的商家已存在黑名单策略");
        }

        // 新增黑名单策略
        BaiqiBlackListStrategyEntity entity = convertStrategyToEntity(param);
        baiqiBlackListStrategyMapper.insertStrategy(entity);
        // 新增黑名单策略商家关联数据
        Long strategyId = entity.getId();
        List<BaiqiBKStrategyMCEntity> bkStrategyMCEntities = Lists.newArrayList();
        param.getMerchantIdList().stream().forEach(e -> {
            bkStrategyMCEntities.add(new BaiqiBKStrategyMCEntity(strategyId, e));
        });
        baiqiBKStrategyMCMapper.insetBatch(bkStrategyMCEntities);

        return ResultDto.successResult(Boolean.TRUE);
    }

    /**
     * 新建全部商家的黑名单策略
     * @param param
     * @return
     */
    private ResultDto<Boolean> insertCommonStrategy(BlackListStrategyDto param){
        List<BaiqiBlackListStrategyEntity> blackListStrategyEntities = baiqiBlackListStrategyMapper.listCommonStrategy();
        if (CollectionUtils.isNotEmpty(blackListStrategyEntities)) {
            return ResultDto.failResult("适用于全部商家的黑名单策略已存在");
        }

        // 插入黑名单策略数据
        BaiqiBlackListStrategyEntity entity = convertStrategyToEntity(param);
        baiqiBlackListStrategyMapper.insertStrategy(entity);
        return ResultDto.successResult(Boolean.TRUE);
    }


    /**
     * 将黑名单策略Dto转换成Entity
     * @param dto
     * @return
     */
    private BaiqiBlackListStrategyEntity convertStrategyToEntity(BlackListStrategyDto dto){
        if (null == dto){
            return null;
        }
        BaiqiBlackListStrategyEntity entity = BeanUtils.copy(dto, BaiqiBlackListStrategyEntity.class);
        entity.setBkType(dto.getBkType().getNum());
        entity.setDealType(dto.getDealType().getNum());
        entity.setStrategyValue(JSONObject.toJSONString(dto.getStrategy()));
        entity.setUseType(dto.getUseType().getNum());
        entity.setOperator(dto.getOperator());
        return entity;
    }

    /**
     * 将黑名单策略Entity转换成Dto
     * @param list
     * @return
     */
    private List<BlackListStrategyDto> convertStrategyToDto(List<BaiqiBlackListStrategyEntity> list){
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }

        List<BlackListStrategyDto> blackListStrategyDtoList = Lists.newArrayList();
        // 将需要添加商家信息的策略id放在这
        List<Long> needMerchantInfoIds = Lists.newArrayList();
        for (BaiqiBlackListStrategyEntity entity : list) {
            BlackListStrategyDto dto = new BlackListStrategyDto();
            dto.setId(entity.getId());
            dto.setBkType(BaiqiBlackListTypeEnum.getByNum(entity.getBkType()));
            dto.setDealType(DealTypeEnum.FORBID_SUBMIT);
            dto.setOperator(entity.getOperator());
            dto.setStrategy(JSON.parseObject(entity.getStrategyValue(), StrategyDto.class));

            dto.setUseType(BlackListUseTypeEnum.getByNum(entity.getUseType()));
            if (entity.getUseType().intValue() == BlackListUseTypeEnum.PART.getNum()) {
                needMerchantInfoIds.add(entity.getId());
            }
            blackListStrategyDtoList.add(dto);
        }

        // 如果有需要添加商家信息的策略
        if (CollectionUtils.isNotEmpty(needMerchantInfoIds)){
            List<BaiqiBKStrategyMCEntity> bkStrategyMCEntities = baiqiBKStrategyMCMapper.listByStrategyIds(needMerchantInfoIds);
            Map<Long, String> merchantMap = merchantService.fetchNamesWithCache();
            bkStrategyMCEntities.stream().forEach(e -> e.setMerchantName(merchantMap.get(e.getMerchantId())));

            // 循环比对，添加商家信息
            for (BlackListStrategyDto dto : blackListStrategyDtoList){
                for (BaiqiBKStrategyMCEntity strategyMCEntity : bkStrategyMCEntities) {
                    if (dto.getId().longValue() == strategyMCEntity.getStrategyId().longValue()){
                        dto.setMerchant(strategyMCEntity.getMerchantId(), strategyMCEntity.getMerchantName());
                    }
                }
            }
        }

        return blackListStrategyDtoList;
    }








}
