package com.qiho.center.biz.service.impl.coupon;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.coupon.CouponConsumerDto;
import com.qiho.center.api.enums.coupon.ActiveRuleEnum;
import com.qiho.center.api.enums.coupon.AstrictMark;
import com.qiho.center.api.enums.coupon.CouponAstrictEnum;
import com.qiho.center.api.enums.coupon.CouponStatusEnum;
import com.qiho.center.api.params.AstrictCouponParams;
import com.qiho.center.api.params.CouponConsumerQuery;
import com.qiho.center.api.params.CouponConsumerQueryParams;
import com.qiho.center.biz.bo.CouponBo;
import com.qiho.center.biz.engine.coupon.AstrictContextDecider;
import com.qiho.center.biz.engine.coupon.AstrictFactory;
import com.qiho.center.biz.engine.coupon.AstrictTemplate;
import com.qiho.center.biz.event.OrderClosedEvent;
import com.qiho.center.biz.service.coupon.CouponConsumerService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.dao.coupon.QihoActiveDAO;
import com.qiho.center.common.dao.coupon.QihoCouponActiveDAO;
import com.qiho.center.common.dao.coupon.QihoCouponAstrictDAO;
import com.qiho.center.common.dao.coupon.QihoCouponConsumerDAO;
import com.qiho.center.common.dao.coupon.QihoCouponDAO;
import com.qiho.center.common.dao.coupon.QihoCouponOrderDAO;
import com.qiho.center.common.entity.coupon.QihoActiveEntity;
import com.qiho.center.common.entity.coupon.QihoCouponActiveEntity;
import com.qiho.center.common.entity.coupon.QihoCouponAstrictEntity;
import com.qiho.center.common.entity.coupon.QihoCouponConsumerEntity;
import com.qiho.center.common.entity.coupon.QihoCouponEntity;
import com.qiho.center.common.entity.coupon.QihoCouponOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.util.AppLogUtil;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

/**
 * Created by danke on 2017/10/31.
 */
@Service
@BizEventListener
public class CouponConsumerServiceImpl implements CouponConsumerService {

    private static final Logger LOGGER = LoggerFactory.getLogger(CouponConsumerServiceImpl.class);

    @Resource
    private QihoCouponConsumerDAO qihoCouponConsumerDAO;

    @Resource
    private QihoActiveDAO qihoActiveDAO;

    @Resource
    private QihoCouponActiveDAO qihoCouponActiveDAO;

    @Resource
    private QihoCouponDAO qihoCouponDAO;

    @Resource
    private QihoCouponAstrictDAO qihoCouponAstrictDAO;

    @Resource
    private AstrictContextDecider astrictContextDecider;

    @Resource
    private CouponBo couponBo;

    @Resource
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Resource
    private QihoCouponOrderDAO qihoCouponOrderDAO;

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer insert(CouponConsumerDto entity) {
        return qihoCouponConsumerDAO.insert(BeanUtils.copy(entity,QihoCouponConsumerEntity.class));
    }

    @Override
    public List<CouponConsumerDto> findAllCouponByMobile(String mobile) {
        CouponConsumerQuery query = new CouponConsumerQuery();
        query.setMobile(mobile);
        List<QihoCouponConsumerEntity> allCoupon = qihoCouponConsumerDAO.findByQuery(query);
        return BeanUtils.copyList(allCoupon,CouponConsumerDto.class);
    }

    @Override
    public PagenationDto<CouponConsumerDto> findCouponConsumerByQuery(CouponConsumerQueryParams params) {
        PagenationDto<CouponConsumerDto> page = new PagenationDto<>();
        Integer count = qihoCouponConsumerDAO.countCouponConsumerByQuery(params);
        page.setTotal(count);
        if (count == 0) {
            return page;
        }

        List<QihoCouponConsumerEntity> couponConsumerEntityList = qihoCouponConsumerDAO.findCouponConsumerByQuery(params);
        List<CouponConsumerDto> couponConsumerList = BeanUtils.copyList(couponConsumerEntityList, CouponConsumerDto.class);
        page.setList(couponConsumerList);
        return page;
    }


    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public List<CouponConsumerDto> consumerGetCoupon(AstrictCouponParams params) {
        List<QihoCouponConsumerEntity> qihoCouponConsumerEntities = sentDownCoupon(params);
        return BeanUtils.copyList(qihoCouponConsumerEntities,CouponConsumerDto.class);
    }

    @Override
    public Boolean verfiyConsumer(AstrictCouponParams params) {
        //校验用户资格
        Long activeId = params.getActiveId();
        QihoActiveEntity activeEntity = qihoActiveDAO.findByActiveId(activeId);
        String activeRule = activeEntity.getActiveRule();
        if (StringUtils.isBlank(activeRule))
            return Boolean.TRUE;
        AstrictMark astrictEnum = ActiveRuleEnum.getByCode(activeRule);
        AstrictTemplate astrictTemplate = astrictContextDecider.decidePayChannel(astrictEnum);
        if (null == astrictTemplate){
            AppLogUtil.error(LOGGER,"活动规则配置出现脏数据 activeId={} activeRule={}",activeId,activeRule);
            return Boolean.FALSE;
        }
        return astrictTemplate.astrictProcess(AstrictFactory.createContextAsNotVerfiy(params));
    }

    @Override
    public List<CouponConsumerDto> findCouponUseAble(AstrictCouponParams params) {
        String mobile = params.getMobile();
        CouponConsumerQuery query = new CouponConsumerQuery();
        query.setMobile(mobile);
        query.setCouponStatus(CouponStatusEnum.UN_USE.getVal());
        query.setNowTime(new Date());
        List<QihoCouponConsumerEntity> ableCoupon = qihoCouponConsumerDAO.findByQuery(query);
        //过滤出符合条件的优惠券模板
        List<Long> filterCouponIds = new ArrayList<>();
        List<QihoCouponConsumerEntity> sentCoupon = ableCoupon.stream().filter(e ->{
            //根据配置的限制对每条优惠券模板的校验
            String employConditType = e.getEmployConditType();
            AstrictMark astrictMark = CouponAstrictEnum.getByCode(employConditType);
            AstrictTemplate astrictTemplate = astrictContextDecider.decidePayChannel(astrictMark);
            return astrictTemplate.astrictProcess(AstrictFactory.createAstrictContext(params,e));
        }).peek(e -> filterCouponIds.add(e.getCouponId())).collect(Collectors.toList());
        //优惠券限制过滤
        if (!params.getInWechat() && CollectionUtils.isNotEmpty(sentCoupon)){
            //查询配置了限制的优惠券,现在只有不在微信一种限制场景,暂时直接过滤
            List<QihoCouponAstrictEntity> couponAstricts = qihoCouponAstrictDAO.batchFindByCouponIds(filterCouponIds);
            if (CollectionUtils.isNotEmpty(couponAstricts)){
                for (QihoCouponAstrictEntity couponAstrict : couponAstricts){
                    Long couponAstrictId = couponAstrict.getCouponId();
                    int index = filterCouponIds.indexOf(couponAstrictId);
                    sentCoupon.remove(index);
                    filterCouponIds.remove(index);
                }
            }
        }
        return BeanUtils.copyList(sentCoupon,CouponConsumerDto.class);
    }

    @Override
    public List<CouponConsumerDto> findByOrderId(String orderId) {
        List<QihoCouponConsumerEntity> entityList = qihoCouponConsumerDAO.listByOrderId(orderId);

        return BeanUtils.copyList(entityList, CouponConsumerDto.class);
    }

    /**
     * 接受订单关闭事件,补充优惠券
     */
    @Subscribe
    public void closedOrderEventLister(OrderClosedEvent orderClosedEvent){

        /// 添加超时打印
        DBTimeProfile.enter("closedOrderEventLister");

        if (orderClosedEvent == null){
            DBTimeProfile.release();
            return;
        }

        String orderId = orderClosedEvent.getOrderId();
        QihoOrderSnapshotEntity orderSnapshot = qihoOrderSnapshotDAO.findByOrderId(orderId);
        String mobile = orderSnapshot.getMobile();
        //通过订单号,查询优惠券和订单关联信息
        List<QihoCouponOrderEntity> couponOrders = qihoCouponOrderDAO.listByOrderId(orderId);
        if(CollectionUtils.isEmpty(couponOrders)){
            DBTimeProfile.release();
            return;
        }

        //现在的业务场景订单和优惠券一对一
        QihoCouponOrderEntity couponOrder = couponOrders.get(0);
        //查询优惠券模板信息
        Long couponId = couponOrder.getCouponId();
        QihoCouponEntity qihoCouponEntity = qihoCouponDAO.findByCouponId(couponId);
        //生成优惠券实体
        QihoCouponConsumerEntity qihoCouponConsumer = BeanUtils.copy(qihoCouponEntity,QihoCouponConsumerEntity.class);
        qihoCouponConsumer.setActiveId(0L);
        qihoCouponConsumer.setCouponId(qihoCouponEntity.getId());
        qihoCouponConsumer.setCouponStatus(CouponStatusEnum.UN_USE.getVal());
        qihoCouponConsumer.setMobile(mobile);
        //开始时间和到期时间
        //获取已使用优惠券的到期时间
        CouponConsumerQuery couponConsumerQuery = new CouponConsumerQuery();
        couponConsumerQuery.setCouponId(couponId);
        couponConsumerQuery.setMobile(mobile);
        List<QihoCouponConsumerEntity> couponConsumers = qihoCouponConsumerDAO.findByQuery(couponConsumerQuery);
        Date couponBeUsedTime = couponConsumers.get(0).getEmployTime();
        Date couponExpireTime = couponConsumers.get(0).getExpireTime();
        int subDays = DateUtils.daysBetween(couponBeUsedTime,couponExpireTime);
        qihoCouponConsumer.setStartTime(new Date());
        qihoCouponConsumer.setExpireTime(getExpireTime(subDays+1));
        //优惠券使用限制
        QihoCouponAstrictEntity couponAstrict = qihoCouponAstrictDAO.findByCouponId(couponId);
        qihoCouponConsumer.setAstrictRule(couponAstrict.getAstrictRule());
        qihoCouponConsumerDAO.insert(qihoCouponConsumer);

        DBTimeProfile.release();
    }

    /**
     * 通过活动id,查询所有活动下挂着的优惠券模板,并实体化
     * 返回值为实体化的模板对象
     */
    private List<QihoCouponConsumerEntity> sentDownCoupon(AstrictCouponParams params){
        //查询出所有的优惠券模板
        List<QihoCouponActiveEntity> couponActiveEntities = qihoCouponActiveDAO.findByActiveId(params.getActiveId());
        if (CollectionUtils.isEmpty(couponActiveEntities)){
            AppLogUtil.error(LOGGER,"该活动下无可以领取的优惠券 activeId={}",params.getActiveId());
            return Lists.newArrayList();
        }
        //获取优惠券id集合
        List<Long> couponIds = couponActiveEntities.stream().map(QihoCouponActiveEntity::getCouponId).collect(Collectors.toList());
        //批量查询优惠券模板
        List<QihoCouponEntity> couponEntities = qihoCouponDAO.batchFindCouponByIds(couponIds);
        return insertCouponConsumerEntity(couponEntities,params);
    }

    /**
     * 插入优惠券实体
     * 插入操作异步进行,同步返回构建好的优惠券Entity
     * 异常直接外跑,客户端catch,数据库回滚
     */
    private List<QihoCouponConsumerEntity> insertCouponConsumerEntity(List<QihoCouponEntity> couponEntities,AstrictCouponParams verfiyParam){
        List<QihoCouponConsumerEntity> couponConsumers = new ArrayList<>();
        if (CollectionUtils.isEmpty(couponEntities))
            return couponConsumers;
        String mobile = verfiyParam.getMobile();
        Long activeId = verfiyParam.getActiveId();
        Date nowTime = new Date();
        List<Long> couponIds = couponEntities.stream().map(QihoCouponEntity::getId).collect(Collectors.toList());
        //批量查询限制
        List<QihoCouponAstrictEntity> couponAstricts = qihoCouponAstrictDAO.batchFindByCouponIds(couponIds);
        //填充优惠券实体对象
        for (QihoCouponEntity couponEntity : couponEntities){
            QihoCouponConsumerEntity qihoCouponConsumer = BeanUtils.copy(couponEntity,QihoCouponConsumerEntity.class);
            qihoCouponConsumer.setActiveId(activeId);
            qihoCouponConsumer.setCouponId(couponEntity.getId());
            qihoCouponConsumer.setCouponStatus(CouponStatusEnum.UN_USE.getVal());
            qihoCouponConsumer.setMobile(mobile);
            //开始时间和到期时间
            qihoCouponConsumer.setStartTime(nowTime);
            qihoCouponConsumer.setExpireTime(getExpireTime(couponEntity.getAbsoluteEmployTime()));
            //优惠券使用限制
            for (QihoCouponAstrictEntity couponAstrict : couponAstricts){
                if (couponAstrict.getCouponId().toString().equals(couponEntity.getId().toString())){
                    qihoCouponConsumer.setAstrictRule(couponAstrict.getAstrictRule());
                }
            }
            couponConsumers.add(qihoCouponConsumer);
        }
        couponBo.batchInsertCoupon(couponConsumers);
        return couponConsumers;
    }

    private Date getExpireTime(int absoluteEmployTime){
        Calendar calendar=GregorianCalendar.getInstance();
        calendar.add(Calendar.DATE, absoluteEmployTime);
        return calendar.getTime();
    }
}
