package com.qiho.center.biz.service.impl.merchant;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.OpenInterfaceDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.dto.merchant.MerchantSimpleDto;
import com.qiho.center.api.dto.merchant.MerchantSpacialDto;
import com.qiho.center.api.enums.finance.BaiqiCommercialTenantTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.MerchantQueryParams;
import com.qiho.center.api.params.OpenInterfaceQueryParams;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoMerchantDAO;
import com.qiho.center.common.daoh.qiho.OpenInterfaceMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceMapper;
import com.qiho.center.common.entity.merchant.QihoMerchantEntity;
import com.qiho.center.common.entityd.qiho.OpenInterfaceEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author chensong
 * @create 2017-11-15 10:53
 **/
@Service
public class MerchantServiceImpl implements MerchantService {

    @Autowired
    private QihoMerchantDAO qihoMerchantDAO;

    @Autowired
    private OpenInterfaceMapper openInterfaceMapper;

    @Resource
    private BaiqiFinanceMapper baiqiFinanceMapper;

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    private final static String MERCHANT_CACHAE_KEY = "BAIQI_MERCHANT_NAMES";

    /**
     * 商家id - 商家Name 缓存
     */
    private LoadingCache<String, Map<Long, String>> merchantCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).build(new CacheLoader<String, Map<Long, String>>() {
        @Override
        public Map<Long, String> load(String key) throws Exception {
            List<MerchantSimpleDto> merchantList = qihoMerchantDAO.findAllIdAndNames();
            Map<Long, String> map = Maps.newHashMap();  // merchantId - merchantName
            merchantList.stream().forEach(e -> {
                map.put(e.getId(), e.getMerchantName());
            });
            return map;
        }
    });




    @Override
    public MerchantDto findById(Long merchantId) {
        return BeanUtils.copy(qihoMerchantDAO.getById(merchantId), MerchantDto.class);
    }

    @Override
    public Boolean saveMerchant(MerchantDto dto) throws BizException {
        if (null == dto) {
            return Boolean.FALSE;
        }
        QihoMerchantEntity merchantEntity = BeanUtils.copy(dto,QihoMerchantEntity.class);
        QihoMerchantEntity oldMerchant = qihoMerchantDAO.findByMerchantName(dto.getMerchantName());
        if (merchantEntity.getId() == null){
            if (oldMerchant != null) {
                throw new BizException("商家名称已存在，请勿重复添加");
            }
            // 创建商家
            createMerchant(merchantEntity);
        } else {
            if (oldMerchant != null && merchantEntity.getId().longValue() != oldMerchant.getId().longValue()) {
                throw new BizException("商家名称已存在，请勿重复保存");
            }
            qihoMerchantDAO.updateMerchant(merchantEntity);
        }
        return Boolean.TRUE;
    }

    /**
     * 创建商家信息
     * @param merchantEntity
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    private void createMerchant(QihoMerchantEntity merchantEntity){
        // 创建商家
        merchantEntity.setVersionSign("v2.0");
        qihoMerchantDAO.insertMerchant(merchantEntity);
        // 同步创建商家资金账户
        BaiqiFinanceEntity financeEntity = new BaiqiFinanceEntity();
        financeEntity.setRelationId(merchantEntity.getId());
        financeEntity.setRelationType(BaiqiCommercialTenantTypeEnum.MERCHANT.getType());
        baiqiFinanceMapper.insertAccountFinance(financeEntity);
    }

    @Override
    public PagenationDto<MerchantDto> queryPage(MerchantQueryParams params) {
        PagenationDto<MerchantDto> page = new PagenationDto<>();
        if (params.getOffset() == null || params.getMax() == null) {
            page.setTotal(0);
            return page;
        }
        Integer count = qihoMerchantDAO.countByQuery(params);
        page.setTotal(count);
        if (count.intValue() == 0) {
            page.setList(Lists.newArrayList());
            return page;
        }

        List<MerchantDto> merchantDtoList = BeanUtils.copyList(qihoMerchantDAO.listByQuery(params), MerchantDto.class);
        page.setList(merchantDtoList);
        return page;
    }

    @Override
    public List<MerchantDto> findByName(String name) {
        if (StringUtils.isBlank(name)) {
            return new ArrayList<>();
        }
        return BeanUtils.copyList(qihoMerchantDAO.listByName(name.trim()), MerchantDto.class);
    }

    @Override
    public MerchantDto findByItemId(Long itemId) {
        if (itemId == null || itemId < 1) {
            throw new QihoException("商品id无效");
        }
        QihoMerchantEntity entity = qihoMerchantDAO.getByItemId(itemId);
        return BeanUtils.copy(entity, MerchantDto.class);
    }

    @Override
    public Map<Long, MerchantDto> findByIdList(List<Long> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyMap();
        }
        List<QihoMerchantEntity> qihoMerchantEntities = qihoMerchantDAO.listByIds(idList);
        Map<Long, MerchantDto> map = new HashMap<>();
        if (CollectionUtils.isEmpty(qihoMerchantEntities)) {
            return map;
        }
        for (QihoMerchantEntity qihoMerchantEntity: qihoMerchantEntities) {
            MerchantDto merchantDto = BeanUtils.copy(qihoMerchantEntity, MerchantDto.class);
            map.put(merchantDto.getId(), merchantDto);
        }
        return map;
    }


    @Override
    public List<MerchantSimpleDto> findAllIdAndNames() {
        return qihoMerchantDAO.findAllIdAndNames();
    }

    @Override
    public List<MerchantSimpleDto> findIdAndNamesByIds(List<Long> ids) {
        return qihoMerchantDAO.findIdAndNamesByIds(ids);
    }

    @Override
    public List<MerchantSpacialDto> findAll() {
        return BeanUtils.copyList(qihoMerchantDAO.findAll(),MerchantSpacialDto.class);
    }

    @Override
    public Map<Long, String> fetchNamesWithCache() {
        return merchantCache.getUnchecked("");
    }

    @Override
    public Map<Long, String> fetchNamesWithCache(Set<Long> idSet) {
        if (CollectionUtils.isEmpty(idSet)) {
            throw new IllegalArgumentException("参数idSet不能为空");
        }
        Map<Long, String> allNames = merchantCache.getUnchecked("");
        Map<Long, String> resMap = Maps.newHashMap();
        idSet.stream().forEach(e -> {
            resMap.put(e, allNames.get(e));
        });
        return resMap;
    }

    @Override
    public PagenationDto<OpenInterfaceDto> openInterfacePageQuery(OpenInterfaceQueryParams params) {
        PagenationDto<OpenInterfaceDto> page = new PagenationDto<>();
        Integer count = openInterfaceMapper.countByQuery(params);
        page.setTotal(count);
        if (count.intValue() == 0) {
            return page;
        }

        List<OpenInterfaceDto> openInterfaceDtos = openInterfaceMapper.listByQuery(params);
        for (OpenInterfaceDto dto : openInterfaceDtos) {
            QihoMerchantEntity merchantEntity = qihoMerchantDAO.getById(dto.getMerchantId());
            if (merchantEntity != null && merchantEntity.getMerchantName() != null) {
                dto.setMerchantName(merchantEntity.getMerchantName());
            }
        }
        page.setList(openInterfaceDtos);
        return page;
    }

    @Override
    public Boolean saveOpenInterface(OpenInterfaceDto dto) {
        if (null == dto) {
            return Boolean.FALSE;
        }
        OpenInterfaceEntity openInterfaceEntity = BeanUtils.copy(dto, OpenInterfaceEntity.class);
        return openInterfaceMapper.saveOpenInterface(openInterfaceEntity) == 1;
    }

    @Override
    public OpenInterfaceDto findByMerchantId(Long merchantId) {
        return BeanUtils.copy(openInterfaceMapper.findByMerchantId(merchantId), OpenInterfaceDto.class);
    }

    @Override
    public OpenInterfaceDto findByAppKey(String appKey) {
        return BeanUtils.copy(openInterfaceMapper.findByAppKey(appKey), OpenInterfaceDto.class);
    }

    @Override
    public List<MerchantSimpleDto> findNoOpenInterface(String merchantName) {
        // 查询所有名称包含字符的商家信息
        List<QihoMerchantEntity> entityList = qihoMerchantDAO.listByName(merchantName);

        List<MerchantSimpleDto> simpleDtoList = new ArrayList<>();
        if (entityList == null || StringUtils.isBlank(merchantName)) {
            return simpleDtoList;
        }

        // 将商家详情信息转化为简单信息
        for (int i = 0; i < entityList.size(); i++) {
            MerchantSimpleDto simpleDto = new MerchantSimpleDto();
            simpleDto.setId(entityList.get(i).getId());
            simpleDto.setMerchantName(entityList.get(i).getMerchantName());
            simpleDtoList.add(simpleDto);
        }

        OpenInterfaceQueryParams params = new OpenInterfaceQueryParams();
        params.setMax(openInterfaceMapper.countByQuery(params));
        List<OpenInterfaceDto> openInterfaceDtoList = openInterfacePageQuery(params).getList();

        // 去除已存在验证信息的商家
        if (openInterfaceDtoList != null) {
            List<MerchantSimpleDto> tempList = new ArrayList<>();
            for (int i = 0; i < openInterfaceDtoList.size(); i++) {
                MerchantSimpleDto simpleDto = new MerchantSimpleDto();
                simpleDto.setId(openInterfaceDtoList.get(i).getMerchantId());
                simpleDto.setMerchantName(openInterfaceDtoList.get(i).getMerchantName());
                tempList.add(simpleDto);
            }
            simpleDtoList.removeAll(tempList);
        }
        return simpleDtoList;
    }

    @Override
    public Boolean updateOpenInterface(OpenInterfaceDto dto) {
        if (null == dto) {
            return Boolean.FALSE;
        }
        OpenInterfaceEntity openInterfaceEntity = BeanUtils.copy(dto, OpenInterfaceEntity.class);
        return openInterfaceMapper.updateOpenInterface(openInterfaceEntity) == 1;
    }

    @Override
    public Integer findHasAppKeyCount() {
        return openInterfaceMapper.countByQuery(null);
    }

    @Override
    public OpenInterfaceDto checkedAppKey(String appKey, String appSecret) {
        OpenInterfaceEntity openInterfaceEntity = new OpenInterfaceEntity();
        openInterfaceEntity.setAppkey(appKey);
        openInterfaceEntity.setAppsecret(appSecret);
        return openInterfaceMapper.checkedAppKey(openInterfaceEntity);
    }

    @Override
    public boolean updateVersionSign(String versionSign,Long id) {
        qihoMerchantDAO.updateVersionSign(versionSign,id);
        return Boolean.TRUE;
    }
}
