/**
 * Project Name:qiho-center-biz File Name:FundOrderServiceImpl.java Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月6日下午1:03:34 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl.order;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.FundPageParam;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.biz.event.FundInfoUpdateEvent;
import com.qiho.center.biz.event.UpdateToCodEvent;
import com.qiho.center.biz.paychannel.pay.PayChannelDecider;
import com.qiho.center.biz.paychannel.pay.PayChannelProcessor;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * ClassName:FundOrderServiceImpl <br/>
 * Date: 2017年6月6日 下午1:03:34 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class FundOrderServiceImpl implements FundOrderService {


    @Autowired
    private QihoFundOrderDAO     qihoFundOrderDAO;
    @Autowired
    private QihoOrderDAO         qihoOrderDAO;
    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;
    @Autowired
    private SequenceNoBuilder    builder;
    @Autowired
    private BizEventBus          bizEventBus;
    @Autowired
    private OrderService         orderService;
    @Autowired
    private PayChannelDecider payChannelDecider;

    @Override
    public List<FundOrderDto> findByOrderId(String orderId) {
        return BeanUtils.copyList(qihoFundOrderDAO.findByOrderId(orderId), FundOrderDto.class);
    }

    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#queryFundPage(com.qiho.center.api.params.FundPageParam)
     */
    @Override
    public PagenationDto<FundOrderDto> queryFundPage(FundPageParam param) {
        PagenationDto<FundOrderDto> pagenation = new PagenationDto<>();
        int count = qihoFundOrderDAO.countByParam(param);
        pagenation.setTotal(count);
        if (count > 0) {
            List<QihoFundOrderEntity> list = qihoFundOrderDAO.queryPage(param);
            List<FundOrderDto> fundList = BeanUtils.copyList(list, FundOrderDto.class);
            pagenation.setList(fundList);
        }
        return pagenation;
    }

    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#findByOrderIdAndBizType(java.lang.String,
     * java.lang.String)
     */
    @Override
    public List<FundOrderDto> findByOrderIdAndBizType(String orderId, String bizType) {
        return BeanUtils.copyList(qihoFundOrderDAO.findByOrderIdAndBizType(orderId, bizType), FundOrderDto.class);
    }

    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#notifySuccess(com.qiho.center.api.dto.FundOrderDto)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void notifySuccess(FundOrderDto fundOrder, String originalFundStatus) {
        updateFundOrder(fundOrder, originalFundStatus, OrderStatusEnum.TO_APPROVE);
        // 自动审批通过
        OrderAuditParams params = new OrderAuditParams();
        params.setOrderId(fundOrder.getOrderId());
        params.setAuditResult(OrderAuditParams.AUDIT_SUCCESS);
        orderService.auditOrder(params);
        
        // 发送下单成功短信
        FundInfoUpdateEvent event = new FundInfoUpdateEvent();
        event.setOrderAmt(fundOrder.getAmt());
        event.setFundStatus(fundOrder.getFundStatus());
        event.setOrderId(fundOrder.getOrderId());
        event.setPayType(fundOrder.getPayType());
        bizEventBus.post(event);
    }
    
    private void updateFundOrder(FundOrderDto fundOrder, String originalFundStatus, OrderStatusEnum orderStatus){
        // 付款子订单加锁
        QihoFundOrderEntity updateFundOrder = qihoFundOrderDAO.lockByOrderIdAndType(fundOrder.getOrderId(),
                                                                                    fundOrder.getBizType(),
                                                                                    originalFundStatus);
        if (updateFundOrder == null) {
            throw new QihoException("付款子订单加锁失败，fundId="+fundOrder.getFundId());
        }
        // 更新付款子订单
        updateFundOrder.setAccount(fundOrder.getAccount());
        updateFundOrder.setPayType(fundOrder.getPayType());
        updateFundOrder.setFundStatus(fundOrder.getFundStatus());
        updateFundOrder.setOutSeqNo(fundOrder.getOutSeqNo());
        qihoFundOrderDAO.update(BeanUtils.copy(fundOrder, QihoFundOrderEntity.class));
        // 更新订单状态
        qihoOrderDAO.updateOrderStatus(fundOrder.getOrderId(), orderStatus);
        // 更新订单快照信息
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(fundOrder.getOrderId());
        orderSnapshot.setPayType(fundOrder.getPayType());
        orderSnapshot.setFundStatus(fundOrder.getFundStatus());
        orderSnapshot.setOrderStatus(orderStatus.getCode());
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }
    
    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#refund(java.lang.String, java.lang.String, java.lang.Integer)
     */
    @Override
    public void refund(String orderId, String payType, Integer amt) {
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        entity.setOrderId(orderId);
        String bizId = StringUtils.substring(orderId, StringUtils.indexOf(orderId, SeqBizTypeEnum.ORDER.getCode())+1);
        String fundId = builder.createSeqNo(SeqBizTypeEnum.FUND, Long.valueOf(bizId));
        entity.setFundId(fundId);
        entity.setPayType(payType);
        entity.setAmt(amt);
        entity.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        entity.setBizType(FundBizTypeEnum.REFUND.getCode());
        qihoFundOrderDAO.initFundOrder(entity);
        // 发起退款流程
        PayChannelProcessor payChannelProcessor = payChannelDecider.decidePayChannel(PayTypeEnum.getEnumByCode(payType));
        payChannelProcessor.refund(orderId, fundId, amt);
    }

    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#updatePayType(java.lang.String, java.lang.String)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updatePayType(FundOrderDto fundOrder, String originalFundStatus) {
        OrderStatusEnum orderStatus;
        if(StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())){
            orderStatus = OrderStatusEnum.TO_APPROVE;
        }else{
            orderStatus = OrderStatusEnum.TO_PAY;
        }
        updateFundOrder(fundOrder, originalFundStatus, orderStatus);
        if(StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())){
            // 发送付款方式更新成货到付款事件
            UpdateToCodEvent event = new UpdateToCodEvent();
            event.setOrderId(fundOrder.getOrderId());
            bizEventBus.post(event);
        }
    }

    @Override
    public int update(FundOrderDto fundOrder) {
        return qihoFundOrderDAO.update(BeanUtils.copy(fundOrder,QihoFundOrderEntity.class));
    }

    @Override
    public FundOrderDto findByFundId(String fundId) {
        QihoFundOrderEntity entity =qihoFundOrderDAO.findByFundId(fundId);
        return BeanUtils.copy(entity,FundOrderDto.class);
    }


    @Override
    public List<FundOrderDto> queryFundOrderToSync(FundPageParam param,Integer pageSize,Integer pageNo) {
        List<QihoFundOrderEntity> list = qihoFundOrderDAO.queryFundOrderToSync(param,pageSize,pageNo);
        return BeanUtils.copyList(list, FundOrderDto.class);
    }
}
