/**
 * Project Name:qiho-center-biz File Name:OrderServiceImpl.java Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月2日下午9:01:40 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl.order;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.stock.service.api.constant.ConsumeStockTypes;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.dto.order.*;
import com.qiho.center.api.dto.task.BatchOrderTaskDto;
import com.qiho.center.api.enums.*;
import com.qiho.center.api.enums.coupon.CouponStatusEnum;
import com.qiho.center.api.enums.ordertmp.OrderTmpRiskCheckFailEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.CouponConsumerQuery;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.api.params.OrderConsumerParam;
import com.qiho.center.api.params.OrderItemParam;
import com.qiho.center.biz.bo.OrderLogisticBo;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.event.*;
import com.qiho.center.biz.service.BlackListService;
import com.qiho.center.biz.service.ItemDetailRecordService;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.impl.OrderStockService;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.service.ordertmp.OrderTmpService;
import com.qiho.center.biz.task.BatchOrderCancelTask;
import com.qiho.center.biz.task.BatchOrderTask;
import com.qiho.center.biz.task.bo.OrderCancelDo;
import com.qiho.center.biz.task.bo.OrderConfirmDo;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.*;
import com.qiho.center.common.dao.coupon.QihoCouponConsumerDAO;
import com.qiho.center.common.dao.coupon.QihoCouponOrderDAO;
import com.qiho.center.common.daoh.qiho.FundOrderMapper;
import com.qiho.center.common.daoh.qiho.OrderConfirmTaskMapper;
import com.qiho.center.common.daoh.qiho.OrderMapper;
import com.qiho.center.common.entity.coupon.QihoCouponConsumerEntity;
import com.qiho.center.common.entity.coupon.QihoCouponOrderEntity;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.merchant.QihoMerchantEntity;
import com.qiho.center.common.entity.order.*;
import com.qiho.center.common.entityd.qiho.FundOrderEntity;
import com.qiho.center.common.entityd.qiho.OrderConfirmTaskEntity;
import com.qiho.center.common.entityd.qiho.OrderEntity;
import com.qiho.center.common.entityd.qiho.QihoItemDetailRecordEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.enums.CacheConstantseEnum;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;
import com.qiho.center.common.util.AppLogUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ClassName:OrderServiceImpl <br/>
 * Date: 2017年6月2日 下午9:01:40 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Service
@BizEventListener
public class OrderServiceImpl implements OrderService {

    private static final Logger LOG = LoggerFactory.getLogger(OrderServiceImpl.class);

    @Autowired
    private QihoOrderDAO qihoOrderDAO;

    @Autowired
    private QihoOrderItemInfoDAO qihoOrderItemInfoDAO;

    @Autowired
    private QihoItemDAO qihoItemDAO;

    @Autowired
    private QihoItemSkuDAO qihoItemSkuDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoChannelInfoDAO qihoChannelInfoDAO;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private SequenceNoBuilder sequenceNoBuilder;

    @Autowired
    private RemoteStockService remoteStockService;

    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    @Autowired
    private BizEventBus eventBus;

    @Autowired
    private BlackListService expressBlackListService;

    @Autowired
    private QihoFundOrderDAO qihoFundOrderDAO;

    @Autowired
    private ItemService itemService;

    @Autowired
    private ItemDetailRecordService itemDetailRecordService;

    @Autowired
    private QihoMerchantDAO qihoMerchantDAO;

    private static final String LIMIT_NUMBER = "limitNumber";

    private static final Integer DEFAULT_LIMIT_NUMBER = 3;

    @Autowired
    private OrderStockService orderStockService;

    @Resource
    private QihoCouponConsumerDAO qihoCouponConsumerDAO;

    @Resource
    private QihoCouponOrderDAO qihoCouponOrderDAO;

    @Autowired
    private ExecutorService executorService;

    @Autowired
    LogisticsOrderService logisticsOrderService;

    @Autowired
    private LogisticsService logisticsService;

    @Autowired
    private SequenceNoBuilder builder;


    @Autowired
    private ApplicationContext context;

    @Autowired
    OrderMapper orderMapper;


    @Autowired
    FundOrderMapper fundOrderMapper;

    @Autowired
    OrderLogisticBo orderLogisticBo;


    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public String createOrder(OrderDto order, ChannelInfoDto channelInfo, String payType) {
        OrderItemDto orderItemDto = order.getOrderItem();
        // 生成订单编号
        String orderId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.ORDER, orderItemDto.getItemId());
        QihoItemSkuEntity skuEntity = qihoItemSkuDAO.findById(orderItemDto.getSkuId());
        // 记录商品成本价至订单
        orderItemDto.setCostPrice(skuEntity.getCostPrice());
        // 扣减库存
        DubboResult<Boolean> result = remoteStockService
            .consumeStockNum(ConsumeStockTypes.QihoTakeOrder.getType(), orderId, skuEntity.getStockId(),
                orderItemDto.getQuantity());
        if (result.isSuccess() || result.getResult()) {
            // 创建订单
            try {
                insertOrder(orderId, order, channelInfo, payType, order.getCouponId());
            } catch (Exception e) {
                // 创建订单失败，回滚库存
                remoteStockService.rollbackStock(ConsumeStockTypes.QihoTakeOrder.getType(), orderId);
                throw new QihoException("创建订单失败", e);
            }

            WarningEvent warningEvent = new WarningEvent();
            warningEvent.setSkuId(skuEntity.getId());
            eventBus.post(warningEvent);
            return orderId;
        }
        throw new QihoException("创建订单失败-扣减库存失败");
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void createOrderFromTmp(OrderDto order, ChannelInfoDto channelInfo, String payType) {

        String orderId = order.getOrderId();
        OrderItemDto orderItemDto = order.getOrderItem();
        QihoItemSkuEntity skuEntity = qihoItemSkuDAO.findById(orderItemDto.getSkuId());
        // 记录商品成本价至订单
        orderItemDto.setCostPrice(skuEntity.getCostPrice());

        // 扣减库存
        DubboResult<Boolean> result = remoteStockService
                .consumeStockNum(ConsumeStockTypes.QihoTakeOrder.getType(), orderId, skuEntity.getStockId(),
                        orderItemDto.getQuantity());
        if (result.isSuccess() && result.getResult()) {

            try {
                // 创建订单
                insertOrderFromTmp(orderId, order, channelInfo, payType);

            } catch (Exception e) {
                // 创建订单失败，回滚库存
                remoteStockService.rollbackStock(ConsumeStockTypes.QihoTakeOrder.getType(), orderId);
                // 将临时订单变为无效  后续还可以由人工变更为有效
                orderTmpService.updateOrderTmpToInvalid(orderId,"创建正式订单失败", OrderTmpRiskCheckFailEnum.CREATE_FORMAL_FAIL);
                LOG.error("临时订单变更正式订单失败，orderId：{}",orderId, e);
                return;
            }

            WarningEvent warningEvent = new WarningEvent();
            warningEvent.setSkuId(skuEntity.getId());
            eventBus.post(warningEvent);
            return ;
        }
        throw new QihoException("创建正式订单失败-扣减库存失败,orderId: " + orderId + result.getMsg());
    }

    @Autowired
    private OrderTmpService orderTmpService;

    /**
     * 新增订单
     * 数据来自临时订单表
     * @param orderId
     * @param order
     * @param channelInfoDto
     * @param payType
     */
    private void insertOrderFromTmp(String orderId, OrderDto order, ChannelInfoDto channelInfoDto, String payType){
        // 新增订单
        QihoOrderEntity qihoOrder = BeanUtils.copy(order, QihoOrderEntity.class);
        if(StringUtils.isNotBlank(qihoOrder.getConsumerName())){
            qihoOrder.setConsumerName(qihoOrder.getConsumerName().trim());
        }
        qihoOrder.setOrderId(orderId);
        qihoOrder.setConsumerId(order.getMobile());
        qihoOrder.setOrderStatus(OrderStatusEnum .TO_APPROVE.getCode());
        qihoOrder.setOrderApprovalStatus(OrderApprovalStatusEnum.TO_APPROVE.getCode());
        qihoOrder.setRemark(null);
        qihoOrderDAO.insert(qihoOrder);
        // 新增订单商品详情
        QihoOrderItemInfoEntity qihoOrderItemInfo = BeanUtils.copy(order.getOrderItem(), QihoOrderItemInfoEntity.class);
        qihoOrderItemInfo.setOrderId(orderId);
        //获取一下最后版本的商品详情记录ID
        Long detailRecordId = itemDetailRecordService
                .getDetailRecordId(qihoOrderItemInfo.getItemId(), order.getPlanCode());
        qihoOrderItemInfo.setDetailRecordId(detailRecordId);
        qihoOrderItemInfoDAO.insert(qihoOrderItemInfo);
        // 记录渠道信息
        QihoChannelInfoEntity qihoChannelInfo = BeanUtils.copy(channelInfoDto, QihoChannelInfoEntity.class);
        qihoChannelInfo.setOrderId(orderId);
        qihoChannelInfo.setTuiaId(order.getTuiaId());
        qihoChannelInfo.setPlanCode(order.getPlanCode());
        qihoChannelInfo.setTuiaCid(channelInfoDto.getaCid() == null ? "0" : channelInfoDto.getaCid());
        qihoChannelInfoDAO.insert(qihoChannelInfo);
        // 创建订单快照
        QihoItemDetailEntity itemEntity = qihoItemDAO.findById(qihoOrderItemInfo.getItemId());

        QihoOrderSnapshotEntity snapshot = BeanUtils.copy(qihoOrder, QihoOrderSnapshotEntity.class);
        snapshot.setItemName(qihoOrderItemInfo.getItemName());
        snapshot.setItemShortName(qihoOrderItemInfo.getItemShortName());
        snapshot.setItemId(qihoOrderItemInfo.getItemId());
        snapshot.setItemNo(qihoOrderItemInfo.getItemNo());
        snapshot.setDelivery(qihoOrderItemInfo.getDelivery());
        snapshot.setSellingPrice(qihoOrderItemInfo.getSellingPrice());
        snapshot.setOriginalPrice(qihoOrderItemInfo.getOriginalPrice());
        snapshot.setQuantity(qihoOrderItemInfo.getQuantity());
        snapshot.setSkuName(qihoOrderItemInfo.getSkuName());
        snapshot.setSkuNo(qihoOrderItemInfo.getSkuNo());
        snapshot.setChannelId(qihoChannelInfo.getChannelId());
        snapshot.setCountChannelId(qihoChannelInfo.getCountChannelId());
        snapshot.setUrl(qihoChannelInfo.getUrl());
        snapshot.setIp(qihoChannelInfo.getIp());
        snapshot.setUserAgent(qihoChannelInfo.getUserAgent());
        snapshot.setTuiaId(qihoChannelInfo.getTuiaId());
        snapshot.setSendTime(order.getSendTime());
        snapshot.setSid(order.getSid());

        QihoMerchantEntity qihoMerchantEntity = qihoMerchantDAO.getByItemId(qihoOrderItemInfo.getItemId());
        if (qihoMerchantEntity != null) {
            snapshot.setMerchantId(qihoMerchantEntity.getId());
        }
        JSONObject json = JSON.parseObject(itemEntity.getExtParam());
        snapshot.setImage(json.getString("image"));
        snapshot.setOrderApprovalStatus(OrderApprovalStatusEnum.TO_APPROVE.getCode());
        snapshot.setItemCost(qihoOrderItemInfo.getCostPrice());
        snapshot.setPayType(payType);
        snapshot.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        snapshot.setRemark(null);
        qihoOrderSnapshotDAO.insert(snapshot);

        // 创建付款子订单
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        entity.setOrderId(orderId);
        OrderItemDto item = order.getOrderItem();
        String fundId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.FUND, item.getItemId());
        entity.setFundId(fundId);
        entity.setPayType(payType);
        entity.setAmt(order.getOrderAmt());
        entity.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        entity.setBizType(FundBizTypeEnum.PAY.getCode());
        qihoFundOrderDAO.initFundOrder(entity);
        // 发布新增订单事件
        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(snapshot, OrderSnapshotDto.class);
        orderSnapshotDto.setSkinId(order.getSkinId());
        orderSnapshotDto.setOrderStatusEnum(OrderStatusEnum.getByCode(snapshot.getOrderStatus()));

        // 删除order_tmp数据
        orderTmpService.deleteOrderTmpByOrderId(orderId);


        // 事务完成后发布事件
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {

            @Override
            public void afterCompletion(int status) {

                // 事务状态为已提交
                if(TransactionSynchronization.STATUS_COMMITTED == status){

                    OrderCreateFromTmpEvent event = new OrderCreateFromTmpEvent();
                    event.setOrderSnapshot(orderSnapshotDto);
                    event.setTokenId(channelInfoDto.getTokenId());
                    eventBus.post(event);
                }

            }
        });
    }

    /**
     * 新增订单
     *
     * @param orderId
     * @param order
     * @param channelInfoDto
     * @param payType
     *
     * @author zhanglihui
     */
    private void insertOrder(String orderId, OrderDto order, ChannelInfoDto channelInfoDto, String payType,
        Long couponId) {
        // 新增订单
        QihoOrderEntity qihoOrder = BeanUtils.copy(order, QihoOrderEntity.class);
        if(StringUtils.isNotBlank(qihoOrder.getConsumerName())){
            qihoOrder.setConsumerName(qihoOrder.getConsumerName().trim());
        }
        qihoOrder.setOrderId(orderId);
        qihoOrder.setConsumerId(order.getMobile());
        qihoOrder.setOrderStatus(order.getOrderStatus());
        qihoOrder.setOrderApprovalStatus(OrderApprovalStatusEnum.TO_APPROVE.getCode());
        qihoOrder.setGmtCreate(new Date());
        qihoOrder.setGmtModified(new Date());
        qihoOrderDAO.insert(qihoOrder);
        // 新增订单商品详情
        QihoOrderItemInfoEntity qihoOrderItemInfo = BeanUtils.copy(order.getOrderItem(), QihoOrderItemInfoEntity.class);
        qihoOrderItemInfo.setOrderId(orderId);
        //获取一下最后版本的商品详情记录ID
        Long detailRecordId = itemDetailRecordService
            .getDetailRecordId(qihoOrderItemInfo.getItemId(), order.getPlanCode());
        qihoOrderItemInfo.setDetailRecordId(detailRecordId);
        qihoOrderItemInfoDAO.insert(qihoOrderItemInfo);
        // 记录渠道信息
        QihoChannelInfoEntity qihoChannelInfo = BeanUtils.copy(channelInfoDto, QihoChannelInfoEntity.class);
        qihoChannelInfo.setOrderId(orderId);
        qihoChannelInfo.setTuiaId(order.getTuiaId());
        qihoChannelInfo.setPlanCode(order.getPlanCode());
        qihoChannelInfo.setTuiaCid(channelInfoDto.getaCid() == null ? "0" : channelInfoDto.getaCid());
        qihoChannelInfoDAO.insert(qihoChannelInfo);
        // 创建订单快照
        QihoItemDetailEntity itemEntity = qihoItemDAO.findById(qihoOrderItemInfo.getItemId());

        QihoOrderSnapshotEntity snapshot = BeanUtils.copy(qihoOrder, QihoOrderSnapshotEntity.class);
        snapshot.setItemName(qihoOrderItemInfo.getItemName());
        snapshot.setItemShortName(qihoOrderItemInfo.getItemShortName());
        snapshot.setItemId(qihoOrderItemInfo.getItemId());
        snapshot.setItemNo(qihoOrderItemInfo.getItemNo());
        snapshot.setDelivery(qihoOrderItemInfo.getDelivery());
        snapshot.setSellingPrice(qihoOrderItemInfo.getSellingPrice());
        snapshot.setOriginalPrice(qihoOrderItemInfo.getOriginalPrice());
        snapshot.setQuantity(qihoOrderItemInfo.getQuantity());
        snapshot.setSkuName(qihoOrderItemInfo.getSkuName());
        snapshot.setSkuNo(qihoOrderItemInfo.getSkuNo());
        snapshot.setChannelId(qihoChannelInfo.getChannelId());
        snapshot.setCountChannelId(qihoChannelInfo.getCountChannelId());
        snapshot.setUrl(qihoChannelInfo.getUrl());
        snapshot.setIp(qihoChannelInfo.getIp());
        snapshot.setUserAgent(qihoChannelInfo.getUserAgent());
        snapshot.setTuiaId(qihoChannelInfo.getTuiaId());
        snapshot.setSendTime(order.getSendTime());
        snapshot.setSid(order.getSid());

        QihoMerchantEntity qihoMerchantEntity = qihoMerchantDAO.getByItemId(qihoOrderItemInfo.getItemId());
        if (qihoMerchantEntity != null) {
            snapshot.setMerchantId(qihoMerchantEntity.getId());
        }
        JSONObject json = JSON.parseObject(itemEntity.getExtParam());
        snapshot.setImage(json.getString("image"));
        snapshot.setOrderApprovalStatus(OrderApprovalStatusEnum.TO_APPROVE.getCode());
        snapshot.setItemCost(qihoOrderItemInfo.getCostPrice());
        snapshot.setPayType(payType);
        snapshot.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        snapshot.setGmtCreate(new Date());
        snapshot.setGmtModified(new Date());
        qihoOrderSnapshotDAO.insert(snapshot);

        // 创建付款子订单
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        entity.setOrderId(orderId);
        OrderItemDto item = order.getOrderItem();
        String fundId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.FUND, item.getItemId());
        entity.setFundId(fundId);
        entity.setPayType(payType);
        entity.setAmt(order.getOrderAmt());
        entity.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        entity.setBizType(FundBizTypeEnum.PAY.getCode());
        qihoFundOrderDAO.initFundOrder(entity);
        // 发布新增订单事件
        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(snapshot, OrderSnapshotDto.class);
        orderSnapshotDto.setCouponId(couponId);
        orderSnapshotDto.setSkinId(order.getSkinId());
        orderSnapshotDto.setOrderStatusEnum(OrderStatusEnum.getByCode(snapshot.getOrderStatus()));

        //更新优惠券已使用
        // updateCouponBeUse(orderSnapshotDto);


        // 事务完成后发布事件
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {

            @Override
            public void afterCompletion(int status) {

                // 事务状态为已提交
                if(TransactionSynchronization.STATUS_COMMITTED == status){

                    OrderCreateEvent event = new OrderCreateEvent();
                    event.setOrderSnapshot(orderSnapshotDto);
                    event.setTokenId(channelInfoDto.getTokenId());
                    eventBus.post(event);
                }

            }
        });

    }

    private void updateCouponBeUse(OrderSnapshotDto orderSnapshotDto) {
        Long couponId = orderSnapshotDto.getCouponId();
        if (couponId == null) {
            return;
        }
        String orderId = orderSnapshotDto.getOrderId();
        String mobile = orderSnapshotDto.getMobile();
        CouponConsumerQuery query = new CouponConsumerQuery();
        query.setCouponId(couponId);
        query.setMobile(mobile);
        List<QihoCouponConsumerEntity> list = qihoCouponConsumerDAO.findByQuery(query);
        if (CollectionUtils.isEmpty(list)) {
            AppLogUtil.error(LOG, "订单消费优惠券异常,优惠券信息查询不到 orderId={},couponId={}", orderId, couponId);
            return;
        }
        //现在的场景只会使用一张优惠券,所以只去集合的第一个
        QihoCouponConsumerEntity oldEntity = list.get(0);
        // 校验优惠券使用金额和订单金额

        QihoCouponConsumerEntity couponConsumer = new QihoCouponConsumerEntity();
        couponConsumer.setCouponStatus(CouponStatusEnum.BE_USE.getVal());
        couponConsumer.setOrderId(orderId);
        couponConsumer.setEmployTime(new Date());
        couponConsumer.setCouponId(couponId);
        couponConsumer.setMobile(mobile);
        qihoCouponConsumerDAO.update(couponConsumer);
        //更新优惠券与订单关系表
        QihoCouponOrderEntity couponOrderEntity = new QihoCouponOrderEntity();
        couponOrderEntity.setCouponId(couponId);
        couponOrderEntity.setOrderId(orderId);
        couponOrderEntity.setCouponType(oldEntity.getCouponType());
        couponOrderEntity.setCouponVal(oldEntity.getCouponVal());
        couponOrderEntity.setEmployConditType(oldEntity.getEmployConditType());
        couponOrderEntity.setEmployConditThreshold(oldEntity.getEmployConditThreshold());
        qihoCouponOrderDAO.insert(couponOrderEntity);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Boolean auditOrder(OrderAuditParams params) {
        QihoOrderEntity order = qihoOrderDAO.findByOrderId(params.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())) {
            AppLogUtil.warn(LOG, "只能审核待审批状态的订单, orderId={}", order.getOrderId());
            return false;
        }
        Integer result;
        switch (params.getAuditResult()) {
        case OrderAuditParams.AUDIT_SUCCESS:
            result = auditOrderToSuccess(params, order);
            break;
        case OrderAuditParams.AUDIT_FAILED:
            result = auditOrderToFailed(params);
            break;
        default:
            throw new QihoException("订单审核参数异常");
        }
        return result == 1;
    }

    /**
     * 审核拒绝
     *
     * @param params
     *
     * @return
     *
     * @author zhanglihui
     */
    private Integer auditOrderToFailed(OrderAuditParams params) {
        Integer result = qihoOrderDAO.auditOrderToFailed(params);
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(params.getOrderId());
        orderSnapshot.setRemark(params.getRemark());
        orderSnapshot.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        orderSnapshot.setOrderApprovalStatus(OrderApprovalStatusEnum.APPROVE_FAILD.getCode());
        orderSnapshot.setFundStatus(FundStatusEnum.FAILED.getCode());
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
        List<QihoFundOrderEntity> list = qihoFundOrderDAO
            .findByOrderIdAndBizType(params.getOrderId(), FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fund = list.get(0);
            if (StringUtils.equals(PayTypeEnum.COD.getCode(), fund.getPayType())) {
                qihoFundOrderDAO.updateFundOrderStatus(fund.getFundId(), FundStatusEnum.FAILED.getCode());
            }
        }
        if (result == 1)
            orderStockService.returnSkuStockByOrderId(params.getOrderId());//返还库存
        //审核失败,关闭订单,发送订单关闭事件
        OrderClosedEvent orderClosedEvent = new OrderClosedEvent();
        orderClosedEvent.setOrderId(params.getOrderId());
        eventBus.post(orderClosedEvent);
        return result;
    }

    /**
     * 审核通过
     *
     * @param params
     *
     * @author zhanglihui
     */
    private Integer auditOrderToSuccess(OrderAuditParams params, QihoOrderEntity order) {
        Integer result = qihoOrderDAO.auditOrderToSuccess(params);
        qihoOrderSnapshotDAO.updateOrderStatus(params.getOrderId(), params.getRemark(), OrderStatusEnum.TO_SEND,
            OrderApprovalStatusEnum.APPROVE_SUCCESS, null);

        QihoOrderItemInfoEntity itemEntity = qihoOrderItemInfoDAO.findByOrderId(params.getOrderId());

        // 创建ERP子订单
        QihoErpOrderEntity erpOrder = new QihoErpOrderEntity();
        erpOrder.setOrderId(order.getOrderId());
        erpOrder.setDelivery(itemEntity.getDelivery());
        erpOrder.setItemId(itemEntity.getItemId());
        erpOrder.setItemName(itemEntity.getItemName());
        erpOrder.setItemNo(itemEntity.getItemNo());
        erpOrder.setSkuNo(itemEntity.getSkuNo());
        erpOrder.setConsumerName(order.getConsumerName());
        erpOrder.setMobile(order.getMobile());
        erpOrder.setProvince(order.getProvince());
        erpOrder.setCity(order.getCity());
        erpOrder.setDistrict(order.getDistrict());
        erpOrder.setAddress(order.getAddress());
        erpOrder.setErpType(ErpTypeEnum.SEND.getCode());
        erpOrder.setErpId(sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.ERP, itemEntity.getItemId()));
        erpOrder.setErpStatus(ErpStatusEnum.TO_NOTIFY.getCode());
        qihoErpOrderDAO.createErpOrder(erpOrder);
        return result;
    }

    /**
     * @see com.qiho.center.biz.service.order.OrderService#findByOrderId(java.lang.String)
     */
    @Override
    public OrderDto findByOrderId(String orderId) {
        QihoOrderEntity qihoOrderEntity = qihoOrderDAO.findByOrderId(orderId);
        if (qihoOrderEntity == null) {
            return null;
        }
        OrderDto orderDto = BeanUtils.copy(qihoOrderEntity, OrderDto.class);
        QihoOrderItemInfoEntity qihoOrderItemInfoEntity = qihoOrderItemInfoDAO.findByOrderId(orderId);
        OrderItemDto orderItemDto = BeanUtils.copy(qihoOrderItemInfoEntity, OrderItemDto.class);
        orderDto.setOrderItem(orderItemDto);
        return orderDto;
    }

    @Override
    public void updateOrderItem(OrderItemParam param) {

        QihoOrderEntity order = qihoOrderDAO.findByOrderId(param.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())) {
            throw new QihoException(
                "更新订单商品详情失败，订单状态错误，orderId=" + param.getOrderId() + ", status=" + order.getOrderStatus());
        }

        checkLimitNumber(param);

        // 更新库存
        Long decreaseStockId = StringUtils.isBlank(param.getDecreaseStockId()) ?
            null :
            Long.valueOf(param.getDecreaseStockId());
        Integer decreaseStock = param.getDecreaseStock();
        Long increaseStockId = StringUtils.isBlank(param.getIncreaseStockId()) ?
            null :
            Long.valueOf(param.getIncreaseStockId());
        Integer increaseStock = param.getIncreaseStock();
        if (decreaseStockId != null && decreaseStock != null) {
            remoteStockBackendService.decreaseItemStock(decreaseStockId, decreaseStock);
        }
        if (increaseStockId != null && increaseStock != null) {
            remoteStockBackendService.increaseItemStock(increaseStockId, increaseStock);
        }
        try {
            updateOrderItemInTransaction(param);
        } catch (Exception e) {
            // 更新失败反向操作库存
            if (decreaseStockId != null && decreaseStock != null) {
                remoteStockBackendService.increaseItemStock(decreaseStockId, decreaseStock);
            }
            if (increaseStockId != null && increaseStock != null) {
                remoteStockBackendService.decreaseItemStock(increaseStockId, increaseStock);
            }
            throw new QihoException("更新订单商品详情失败，param=" + param, e);
        }
    }

    /**
     * //检验商品购买数量与商品限购件数
     *
     * @param param
     */
    private void checkLimitNumber(OrderItemParam param) {
        QihoOrderItemInfoEntity qihoOrderItemInfoentity = qihoOrderItemInfoDAO.findByOrderId(param.getOrderId());
        ItemDetailDto itemDetailDto = itemService.queryItemDetail(qihoOrderItemInfoentity.getItemId());
        Integer limitNumber;
        if (itemDetailDto.getExtParam().get(LIMIT_NUMBER) == null) {
            limitNumber = DEFAULT_LIMIT_NUMBER;
        } else {
            limitNumber = Integer.valueOf(itemDetailDto.getExtParam().get(LIMIT_NUMBER));
        }
        if (param.getQuantity() > limitNumber) {
            throw new QihoException(
                "更新订单商品详情失败，商品购买数量超过限购件数，orderId=" + param.getOrderId() + ",商品限购" + limitNumber + "件");
        }
    }

    /**
     * 事务内更新订单商品详情
     *
     * @param param
     *
     * @author zhanglihui
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateOrderItemInTransaction(OrderItemParam param) {
        QihoOrderItemInfoEntity qihoOrderItemInfo = new QihoOrderItemInfoEntity();
        qihoOrderItemInfo.setOrderId(param.getOrderId());
        qihoOrderItemInfo.setSkuId(param.getSkuId());
        qihoOrderItemInfo.setSkuName(param.getSkuName());
        qihoOrderItemInfo.setSkuNo(param.getSkuNo());
        qihoOrderItemInfo.setSellingPrice(param.getSellingPrice());
        qihoOrderItemInfo.setOriginalPrice(param.getOriginalPrice());
        qihoOrderItemInfo.setQuantity(param.getQuantity());
        qihoOrderItemInfoDAO.updateByOrderId(qihoOrderItemInfo);

        // 更新订单金额
        Integer orderAmt = param.getSellingPrice() * param.getQuantity();
        qihoOrderDAO.updateOrderAmt(param.getOrderId(), orderAmt);

        QihoOrderSnapshotEntity orderSnapshot = BeanUtils.copy(param, QihoOrderSnapshotEntity.class);
        orderSnapshot.setOrderAmt(orderAmt);
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    /**
     * @see com.qiho.center.biz.service.order.OrderService#updateOrderConsumer(com.qiho.center.api.params.OrderConsumerParam)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateOrderConsumer(OrderConsumerParam param) {
        QihoOrderEntity order = qihoOrderDAO.findByOrderId(param.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())) {
            LOG.warn("更新订单买家信息失败，订单状态错误，orderId={},status={}", param.getOrderId(), order.getOrderStatus());
            throw new QihoException("操作失败");
        }
        Long blackId = expressBlackListService.findOneBlack(ExpressCompanyEnum.YTO, param.getAreaNum());
        if (blackId != null) {
            LOG.warn("更新订单买家信息失败，寄送地址不在货到付款配送区域内，orderId={}, 行政区划代码={}", param.getOrderId(), param.getAreaNum());
            throw new QihoException("下单失败！您的寄送地址不在货到付款配送区域内，暂时无法下单!");
        }
        QihoOrderEntity updateOrder = BeanUtils.copy(param, QihoOrderEntity.class);
        qihoOrderDAO.updateByOrderId(updateOrder);
        QihoOrderSnapshotEntity orderSnapshot = BeanUtils.copy(param, QihoOrderSnapshotEntity.class);
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    @Override
    public void rejectOrder(String orderId, String remark, OrderApprovalStatusEnum orderApprovalStatusEnum) {
        qihoOrderDAO.rejectOrder(orderId, remark, orderApprovalStatusEnum);
    }

    @Override
    public void updateOrderStatus(String orderId, OrderStatusEnum orderStatus) {
        qihoOrderDAO.updateOrderStatus(orderId, orderStatus);
    }

    @Transactional(DsConstants.DATABASE_QIHO)
    @Override
    public boolean closeOrder(String orderId, String remark, String fundId) {
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        updateOrder.setRemark(remark);
        int num = qihoOrderDAO.updateByOrderId(updateOrder);
        qihoOrderSnapshotDAO.updateOrderStatus(orderId, remark, OrderStatusEnum.CLOSED, null, FundStatusEnum.FAILED);
        if (StringUtils.isNotBlank(fundId)) {
            qihoFundOrderDAO.updateFundOrderStatus(fundId, FundStatusEnum.FAILED.getCode());
        }
        boolean result = num == 1;
        if (result) {
            orderStockService.returnSkuStockByOrderId(orderId);//回滚库存
        }
        //订单关闭,发送订单关闭事件
        OrderClosedEvent orderClosedEvent = new OrderClosedEvent();
        orderClosedEvent.setOrderId(orderId);
        eventBus.post(orderClosedEvent);
        return result;
    }

    @Override
    public ItemDetailDto getOrderItemSnapshot(String orderId, Long itemId) {
        ItemDetailDto itemDetailDto = new ItemDetailDto();
        ItemDetailDto itemDetail = itemService.queryItemDetail(itemId);
        //查询商品信息
        QihoOrderItemInfoEntity itemInfoEntity = qihoOrderItemInfoDAO.findByOrderId(orderId);
        itemDetailDto.setItemName(itemInfoEntity.getItemName());
        itemDetailDto.setItemShortName(itemInfoEntity.getItemShortName());
        itemDetailDto.setId(itemId);
        itemDetailDto.setItemNo(itemInfoEntity.getItemNo());
        //设置详情
        if (null == itemInfoEntity.getDetailRecordId()) {
            ItemAdvertDto temAdvertDto = itemService.queryExtParamByItemId(itemId);
            itemDetailDto.setDetail(temAdvertDto.getDetail());
        } else {
            QihoItemDetailRecordEntity detailRecordEntity = itemDetailRecordService
                .selectById(itemInfoEntity.getDetailRecordId());
            itemDetailDto.setDetail(detailRecordEntity.getDetail());
        }
        List<ItemSkuDto> skuDtoList = Lists.newArrayList();
        ItemSkuDto skuDto = new ItemSkuDto();
        skuDto.setSellingPrice(itemInfoEntity.getSellingPrice());
        skuDto.setOriginalPrice(itemInfoEntity.getOriginalPrice());
        skuDto.setCostPrice(itemInfoEntity.getCostPrice());
        skuDto.setSkuName(itemInfoEntity.getSkuName());
        skuDto.setId(itemInfoEntity.getSkuId());
        skuDtoList.add(skuDto);
        itemDetailDto.setSkuList(skuDtoList);
        itemDetailDto.setPriceText(null == itemDetail ? "" : itemDetail.getPriceText());
        return itemDetailDto;
    }

    @Override
    public Long queryMonitorOrderCount(String orderStatus, Date startTime, Date endTime, List<String> provinceList) {
        Long count = qihoOrderDAO.queryMonitorOrderCount(orderStatus, startTime, endTime, provinceList);
        return null == count ? 0 : count;
    }

    @Override
    public OrderDto getLastOrderByMobile(String mobile) {
        if (StringUtils.isBlank(mobile)) {
            throw new QihoException("传入的手机号空白");
        }
        return BeanUtils.copy(qihoOrderDAO.getLastOrderByMobile(mobile), OrderDto.class);
    }

    private static final String NUM_AND_LETTER_REG_EX = "^[A-Za-z0-9]{1,40}$";

    @Override
    public Boolean orderConfirm(OrderConfirmDto orderConfirmDto) throws BizException {
        if(StringUtils.isNotBlank(orderConfirmDto.getExpressCode())){
            orderConfirmDto.setExpressCode(orderConfirmDto.getExpressCode().trim());
        }
        Pattern pat = Pattern.compile(NUM_AND_LETTER_REG_EX, Pattern.CASE_INSENSITIVE);
        Matcher matcher = pat.matcher(orderConfirmDto.getExpressCode());
        // 字符串是否与正则表达式相匹配
        boolean rs = matcher.matches();
        if (!rs) {//数字和字母的检验
            throw new BizException("物流单号格式有误！");
        }

        String orderId = orderConfirmDto.getOrderId();
        BaiqiLogisticsEntity logisticsEntity =
                logisticsService.findByCode(orderConfirmDto.getExpress());//查询物流信息
        if (logisticsEntity == null){
            throw new BizException("不是有效的物流公司！");
        }
        QihoOrderSnapshotEntity snapshotEntity = qihoOrderSnapshotDAO.findByOrderId(orderId);//查询快照表信息

        if(snapshotEntity.getDelivery().equals("ERP")){
            throw new BizException("ERP订单操作失败！");
        }
        //待审核的订单是否可以发货
        if(!OrderStatusEnum.TO_SEND.getCode().equals(snapshotEntity.getOrderStatus())){
            throw new BizException("订单必须处于待发货状态！");
        }
        //初始化物流子订单参数
        //向后兼容 获取erpid
        List<QihoErpOrderEntity> erpOrderEntities = qihoErpOrderDAO.findByOrderId(orderId);
        if(CollectionUtils.isEmpty(erpOrderEntities)){
            throw new BizException("ERP订单订单不存在 确认发货失败！");
        }

        LogisticsOrderEntity logisticsOrderEntity = new LogisticsOrderEntity();
        logisticsOrderEntity.setOrderId(orderId);
        logisticsOrderEntity.setErpId(erpOrderEntities.get(0).getErpId());
        logisticsOrderEntity.setPostId(orderConfirmDto.getExpressCode());
        logisticsOrderEntity.setLogisticsCode(orderConfirmDto.getExpress());
        logisticsOrderEntity.setLogisticsName(logisticsEntity.getLogisticsName());
        logisticsOrderEntity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getDesc());
        // 物流子订单初始化状态为接单
        logisticsOrderEntity.setOrderStatus(LogisticsOrderStatusEnum.ACCEPT.getCode());
        logisticsOrderEntity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, snapshotEntity.getItemId()));

        logisticsOrderEntity.setConsumerName(snapshotEntity.getConsumerName());
        logisticsOrderEntity.setItemId(snapshotEntity.getItemId());
        logisticsOrderEntity.setItemName(snapshotEntity.getItemName());
        logisticsOrderEntity.setItemNo(snapshotEntity.getItemNo());
        logisticsOrderEntity.setMobile(snapshotEntity.getMobile());
        logisticsOrderEntity.setOrderId(snapshotEntity.getOrderId());
        logisticsOrderEntity.setSkuNo(snapshotEntity.getSkuNo());
        logisticsOrderEntity.setProvince(snapshotEntity.getProvince());
        logisticsOrderService.createLogisticsOrder(logisticsOrderEntity);
        return true;
    }


    @Transactional(DsConstants.DATABASE_QIHO)
    @Override
    public Boolean orderCancel(String orderId) throws BizException {
        if(StringUtils.isNotBlank(orderId)){
            orderId = orderId.trim();
        }
        //查询快照表信息
        QihoOrderSnapshotEntity snapshotEntity = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if(snapshotEntity.getDelivery().equals("ERP")){
            throw new BizException("ERP订单操作失败！");
        }
        //待审核的订单是否可以发货
        if(!OrderStatusEnum.TO_SEND.getCode().equals(snapshotEntity.getOrderStatus())&&
                !OrderStatusEnum.TO_APPROVE.getCode().equals(snapshotEntity.getOrderStatus())){
            throw new BizException("订单必须处于待审核、待发货状态！");
        }
        //主订单更新
        OrderEntity orderEntity = new OrderEntity();
        orderEntity.setOrderId(orderId);
        orderEntity.setRemark("手动发货-取消发货");
        orderEntity.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        orderMapper.updateByOrderId(orderEntity);
        //快照表更新
        snapshotEntity.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        snapshotEntity.setFundStatus(FundStatusEnum.FAILED.getCode());
        qihoOrderSnapshotDAO.updateByOrderId(snapshotEntity);
        //资金子订单更新
        FundOrderEntity fundOrderEntity = new FundOrderEntity();
        fundOrderEntity.setOrderId(orderId);
        fundOrderEntity.setFundStatus(FundStatusEnum.FAILED.getCode());
        fundOrderMapper.updateByOrderId(fundOrderEntity);
        orderStockService.returnSkuStockByOrderId(orderId);
        return true;
    }

    @Override
    public Boolean orderUpdate(OrderConfirmDto dto) throws BizException{
        if(StringUtils.isNotBlank(dto.getExpressCode())){
            dto.setExpressCode(dto.getExpressCode().trim());
        }
        Pattern pat = Pattern.compile(NUM_AND_LETTER_REG_EX, Pattern.CASE_INSENSITIVE);
        Matcher matcher = pat.matcher(dto.getExpressCode());
        // 字符串是否与正则表达式相匹配
        boolean rs = matcher.matches();
        if (!rs) {//数字和字母的检验
            throw new BizException("物流单号格式有误！");
        }

        //查询快照表信息
        QihoOrderSnapshotEntity snapshotEntity = qihoOrderSnapshotDAO.findByOrderId(dto.getOrderId());

        if(snapshotEntity.getDelivery().equals("ERP")){
            throw new BizException("ERP订单操作失败！");
        }
        //待审核的订单是否可以发货
        if(!OrderStatusEnum.SEND_OUT.getCode().equals(snapshotEntity.getOrderStatus())){
            throw new BizException("订单必须处于已发货状态！");
        }
        BaiqiLogisticsEntity logisticsEntity =
            logisticsService.findByCode(dto.getExpress());//查询物流信息
        if (logisticsEntity == null){
            throw new BizException("不是有效的物流公司！");
        }
        orderLogisticBo.updateLogistics(logisticsEntity.getLogisticsCode(),logisticsEntity.getLogisticsName(),
            dto.getExpressCode(),dto.getOrderId());
        LogisticsOrderDo logisticsOrderDo = new LogisticsOrderDo();
        logisticsOrderDo.setPostId(dto.getExpressCode());
        logisticsOrderDo.setLogisticsCode(logisticsEntity.getLogisticsCode());
        List<LogisticsOrderDo> logisticsOrderDos = Lists.newArrayList();
        logisticsOrderDos.add(logisticsOrderDo);
        KuaiDi100Event event = new KuaiDi100Event();
        event.setLogisticsOrderDos(logisticsOrderDos);
        eventBus.post(event);
        return true;
    }

    @Override
    public BatchOrderTaskDto batchOrderConfirm(BatchOrderConfirmDto batchOrderConfirmDto) {
        BatchOrderTaskDto dto = new BatchOrderTaskDto();
        Long taskId = new Date().getTime();
        String key = CacheConstantseEnum.DEAL_ORDER_TASK.getCacheKey(taskId);
        dto.setTaskId(key);
        batchOrderConfirmDto.setTaskId(key);
        batchOrderConfirmDto.setTaskType(0);
        OrderConfirmTaskEntity taskEntity = initTaskRedisData(batchOrderConfirmDto);
        //异步读取文件  处理订单
        BatchOrderTask task = context.getBean(BatchOrderTask.class);
        task.init(batchOrderConfirmDto,taskEntity);
        executorService.submit(task);
        return dto;
    }

    @Override
    public BatchOrderTaskDto batchOrderUpdate(BatchOrderConfirmDto batchOrderConfirmDto) {
        BatchOrderTaskDto dto = new BatchOrderTaskDto();
        Long taskId = new Date().getTime();
        String key = CacheConstantseEnum.DEAL_ORDER_TASK.getCacheKey(taskId);
        dto.setTaskId(key);
        batchOrderConfirmDto.setTaskId(key);
        batchOrderConfirmDto.setTaskType(1);
        OrderConfirmTaskEntity taskEntity = initTaskRedisData(batchOrderConfirmDto);
        //异步读取文件  处理订单
        BatchOrderTask task = context.getBean(BatchOrderTask.class);
        task.init(batchOrderConfirmDto,taskEntity);
        executorService.submit(task);
        return dto;
    }

    @Override
    public BatchOrderTaskDto batchOrderCancel(BatchOrderCancelDto batchOrderCancelDto) {
        BatchOrderTaskDto dto = new BatchOrderTaskDto();
        Long taskId = new Date().getTime();
        String key = CacheConstantseEnum.DEAL_ORDER_TASK.getCacheKey(taskId);
        dto.setTaskId(key);
        batchOrderCancelDto.setTaskId(key);
        batchOrderCancelDto.setTaskType(2);
        OrderConfirmTaskEntity taskEntity = initTaskRedisData(batchOrderCancelDto);
        BatchOrderCancelTask task = context.getBean(BatchOrderCancelTask.class);
        task.init(batchOrderCancelDto,taskEntity);
        executorService.submit(task);
        return dto;
    }

    @Override
    public BatchOrderTaskDto openBatchOrderConfirm(OpenBatchOrderConfirmDto confirmDto) {
        BatchOrderTaskDto dto = new BatchOrderTaskDto();
        Long taskId = new Date().getTime();
        String key = CacheConstantseEnum.DEAL_ORDER_TASK.getCacheKey(taskId);
        dto.setTaskId(key);
        confirmDto.setTaskId(key);
        OrderConfirmTaskEntity taskEntity = initTaskRedisData(confirmDto);
        BatchOrderTask task = context.getBean(BatchOrderTask.class);
        BatchOrderConfirmDto batchOrderConfirmDto = BeanUtils.copy(confirmDto,BatchOrderConfirmDto.class);
        batchOrderConfirmDto.setExpress("-");
        task.init(batchOrderConfirmDto,taskEntity,BeanUtils.copyList(confirmDto.getOrderDtos(),
            OrderConfirmDo.class));
        executorService.submit(task);
        return dto;
    }

    @Override
    public BatchOrderTaskDto openBatchOrderCancel(OpenBatchOrderCancelDto cancelDto) {
        BatchOrderTaskDto dto = new BatchOrderTaskDto();
        Long taskId = new Date().getTime();
        String key = CacheConstantseEnum.DEAL_ORDER_TASK.getCacheKey(taskId);
        dto.setTaskId(key);
        cancelDto.setTaskId(key);
        OrderConfirmTaskEntity taskEntity = initTaskRedisData(cancelDto);
        BatchOrderCancelTask task = context.getBean(BatchOrderCancelTask.class);
        task.init(BeanUtils.copy(cancelDto,BatchOrderCancelDto.class),
            taskEntity,BeanUtils.copyList(cancelDto.getOrderDtos(),OrderCancelDo.class));
        executorService.submit(task);
        return dto;
    }

    @Autowired
    StringRedisTemplate redisTemplate;

    @Autowired
    OrderConfirmTaskMapper orderConfirmTaskMapper;

    private OrderConfirmTaskEntity initTaskRedisData(BatchOrderDto dto) {
        OrderConfirmTaskEntity taskEntity = new OrderConfirmTaskEntity();
        taskEntity.setMerchantId(dto.getMerchantId());
        taskEntity.setAllNum(0);
        taskEntity.setConsumeTime(0L);
        taskEntity.setFailNum(0);
        taskEntity.setFileName(dto.getFileName());
        taskEntity.setFileUrl(dto.getFileUrl());
        taskEntity.setTaskStatus(0);
        taskEntity.setTaskId(dto.getTaskId());
        taskEntity.setTaskType(dto.getTaskType());
        orderConfirmTaskMapper.insert(taskEntity);

        Map<String,String> info = Maps.newHashMap();
        info.put("totalCount", "999999999");
        info.put("handleCount", "0");
        info.put("failLines", "");
        info.put("status", "init");
        redisTemplate.opsForHash().putAll(dto.getTaskId(), info);
        redisTemplate.expire(dto.getTaskId(),30, TimeUnit.MINUTES);
        return  taskEntity;
    }

}
