package com.qiho.center.biz.service.impl.order;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.eventbus.AllowConcurrentEvents;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.constant.YZHSmsConstant;
import com.qiho.center.api.dto.ChannelInfoDto;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.embed.EmbedDto;
import com.qiho.center.api.dto.reply.UserReplyDto;
import com.qiho.center.api.enums.*;
import com.qiho.center.api.enums.embed.EmbedResultEnum;
import com.qiho.center.api.enums.ordersms.SmsMobileStatusEnum;
import com.qiho.center.api.params.ForwardToTuiaParams;
import com.qiho.center.api.params.OrderGainParams;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.api.util.BizLog;
import com.qiho.center.biz.bo.SsjxPushOrderBo;
import com.qiho.center.biz.engine.impl.ShotOrderEngine;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.OrderCreateEvent;
import com.qiho.center.biz.event.OrderCreateFromTmpEvent;
import com.qiho.center.biz.event.UpdateToCodEvent;
import com.qiho.center.biz.service.MediaWhiteListService;
import com.qiho.center.biz.service.embed.EmbedService;
import com.qiho.center.biz.service.impl.check.IPMobileCheckService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.biz.service.order.ChannelInfoService;
import com.qiho.center.biz.service.order.OrderSnapshotService;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DomainConstantUtil;
import com.qiho.center.common.dao.*;
import com.qiho.center.common.daoh.qiho.ChuangLanSmsCommitMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderSmsMapper;
import com.qiho.center.common.daoh.qiho.reply.BaiqiUserReplyMapper;
import com.qiho.center.common.entity.QihoShortUrlEntity;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoAnticheateEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.entityd.qiho.ChuangLanSmsCommitEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderSmsEntity;
import com.qiho.center.common.entityd.qiho.reply.BaiqiUserReplyEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.model.tongdun.FraudApiResponse;
import com.qiho.center.common.util.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by liuyao on 2017/6/3.
 */
@Service
@BizEventListener
public class OrderSnapshotServiceImpl implements OrderSnapshotService {

    private static final Logger LOG = LoggerFactory.getLogger(OrderSnapshotServiceImpl.class);

    private static final String CALLBACK_DATA = "部分必要参数为空，未发送";

    private static final String TUIA_URL_POSTFIX = "#buyframe";

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoShortUrlDAO qihoShortUrlDAO;

    @Autowired
    private QihoTemplateDAO qihoTemplateDAO;

    @Autowired
    private SmsService smsService;

    @Resource
    private ShotOrderEngine shotOrderEngine;

    @Autowired
    private QihoAnticheateDAO qihoAnticheateDAO;

    @Autowired
    private ChannelInfoService channelInfoService;

    @Resource
    private MediaWhiteListService mediaWhiteListService;

    @Autowired
    private MerchantService merchantService;

    @Resource
    private SsjxPushOrderBo ssjxPushOrderBo;

    @Resource
    private OrderSnapshotService orderSnapshotService;

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    @Autowired
    private EmbedService embedService;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    IPMobileCheckService ipMobileCheckService;

    @Autowired
    private BaiqiOrderSmsMapper baiqiOrderSmsMapper;

    @Autowired
    private BaiqiUserReplyMapper baiqiUserReplyMapper;



    /**
     * 盛世嘉祥商家id
     */
    @Value("${qiho.order.ssjx.merchantid}")
    private String ssjxMerchartId;

    /**
     * 推啊api转化对接皮肤ID
     */
    @Value("${qiho.order.tuia.skinid}")
    private String skinId;

    /**
     * 推啊api转化对接url
     */
    @Value("${qiho.order.tuia.url}")
    private String tuiaUrl;

    @Override
    public PagenationDto<OrderSnapshotDto> findOrderPage(OrderPageParams params) {
        PagenationDto<OrderSnapshotDto> page = new PagenationDto<>();
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.findOrderPage(params);
        if (CollectionUtils.isEmpty(list)) {
            page.setList(Lists.newArrayList());
            page.setTotal(0);
            return page;
        }
        page.setTotal(qihoOrderSnapshotDAO.findOrderPageCount(params));

        page.setList(this.injectExtraInfo(list, params.getRemoteInvoke()));
        return page;
    }



    /**
     * 添加额外信息 物流、商家、防作弊、用户回复等
     *
     * @param list
     * @return
     */
    private List<OrderSnapshotDto> injectExtraInfo(List<QihoOrderSnapshotEntity> list, RemoteInvokeEnum remoteInvoke) {
        //物流子订单状态
        List<String> orderIds = Lists.newArrayList();
        Set<Long> merchantIds = Sets.newHashSet();
        list.stream().forEach(e -> {
            orderIds.add(e.getOrderId());
            merchantIds.add(e.getMerchantId());
        });
        List<LogisticsOrderEntity> logisticsOrderList = qihoLogisticsOrderDAO.findByOrderIds(orderIds);

        // logisticsMap 用于存放orderId - logisticsStatus
        Map<String, String> logisticsMap = Maps.newHashMap();
        logisticsOrderList.stream().forEach(e->{
            logisticsMap.put(e.getOrderId(), e.getLogisticsStatus());
        });

        // merchantMap 用于存放merchantId - merchantName
        Map<Long, String> merchantMap = merchantService.fetchNamesWithCache(merchantIds);

        list.stream().forEach(e -> {
            e.setLogisticsStatus(logisticsMap.get(e.getOrderId()));
            e.setMerchantName(merchantMap.get(e.getMerchantId()));
        });

        List<OrderSnapshotDto> orderSnapshotDtoList = this.getHandlerOrderSnapshotDto(list);

        // 外部商家后台需要展示用户评论
        if (Objects.equal(remoteInvoke, RemoteInvokeEnum.BAIQI_FERRERI_WEB)) {
            return paddingUserReply(orderSnapshotDtoList);
        }

        return orderSnapshotDtoList;
    }

    /**
     * 封装用户回复
     * 如果每笔订单用户评论数大于5条 则取最新的前5条
     * @param orderSnapshotDtoaList
     */
    private List<OrderSnapshotDto> paddingUserReply(List<OrderSnapshotDto> orderSnapshotDtoaList){
        if (CollectionUtils.isEmpty(orderSnapshotDtoaList)) {
            return orderSnapshotDtoaList;
        }
        List<OrderSnapshotDto> orderSnapshotDtoList = BeanUtils.copyList(orderSnapshotDtoaList, OrderSnapshotDto.class);
        List<String> orderIds = orderSnapshotDtoList.stream().map(OrderSnapshotDto::getOrderId).collect(Collectors.toList());
        // 批量获取订单回复
        List<BaiqiUserReplyEntity> userReplyEntities = baiqiUserReplyMapper.listUserReplyByOrderIds(orderIds);
        if (CollectionUtils.isEmpty(userReplyEntities)) {
            return orderSnapshotDtoList;
        }

        // 将用户回复进行封装
        // userReplyMap key->orderId value->用户回复list
        Map<String, List<BaiqiUserReplyEntity>> userReplyMap = Maps.newHashMap();
        userReplyEntities.forEach(e -> {
            String orderId = e.getOrderId();
            if (userReplyMap.get(orderId) == null) {
                userReplyMap.put(orderId, Lists.newArrayList(e));
            } else {
                List<BaiqiUserReplyEntity> orderReplyList = userReplyMap.get(orderId);
                if (orderReplyList.size() < 5) {
                    orderReplyList.add(e);
                    userReplyMap.put(orderId, orderReplyList);
                }
            }
        });

        // 遍历orderSnapshotList userReplyMap对应相同orderId则set用户回复
        for (OrderSnapshotDto snapshot : orderSnapshotDtoList) {
            String orderId = snapshot.getOrderId();
            if (CollectionUtils.isEmpty(userReplyMap.get(orderId))) {
                snapshot.setUserReplyList(Lists.newArrayList());
            } else {
                List<BaiqiUserReplyEntity> userReplyEntityList = userReplyMap.get(orderId);
                List<UserReplyDto> userReplyDtoList = BeanUtils.copyList(userReplyEntityList, UserReplyDto.class);
                snapshot.setUserReplyList(userReplyDtoList);
            }
        }

        return orderSnapshotDtoList;

    }


    @Override
    public List<OrderSnapshotDto> findOrderPageForExport(OrderPageParams params) {
        List<QihoOrderSnapshotEntity> list = Lists.newArrayList();
        if (!Objects.equal(null, params.getOrderId())) {
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if (order != null) {
                list.add(order);
            }
        } else {
            list.addAll(qihoOrderSnapshotDAO.exportOrderPage(params));
        }
        //再去查询erpid 并且去重
        if(!list.isEmpty()){
            List<String> orderIds = Lists.newArrayList();
            for (QihoOrderSnapshotEntity qihoOrderSnapshotEntity : list) {
                orderIds.add(qihoOrderSnapshotEntity.getOrderId());
            }
            List<QihoErpOrderEntity> list1 = qihoErpOrderDAO.findByOrderIds(orderIds);
            Map<String,QihoErpOrderEntity> map = Maps.newHashMap();
            for (QihoErpOrderEntity erpOrderEntity : list1) {
                map.put(erpOrderEntity.getOrderId(),erpOrderEntity);
            }
            for (QihoOrderSnapshotEntity qihoOrderSnapshotEntity : list) {
                if(map.containsKey(qihoOrderSnapshotEntity.getOrderId())){
                    qihoOrderSnapshotEntity.setErpId(map.get(qihoOrderSnapshotEntity.getOrderId()).getErpId());
                }
            }
        }

        List<OrderSnapshotDto> orderSnapshotDtoList = this.getHandlerOrderSnapshotDto(list);

        return paddingUserReply(orderSnapshotDtoList);

    }

    @Override
    public Integer findAllPageCount(OrderPageParams params) {
        if (!Objects.equal(null, params.getOrderId())) {
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if (order != null) {
                return 1;
            } else {
                return 0;
            }
        } else {
            return qihoOrderSnapshotDAO.findOrderPageCount(params);
        }
    }

    private class DtoTransform implements Function<QihoOrderSnapshotEntity, OrderSnapshotDto> {

        private Map<String, String> anticheatMap;

        /**
         * anticheatMap.
         *
         * @param anticheatMap the anticheatMap to set
         */
        public void setAnticheatMap(Map<String, String> anticheatMap) {
            this.anticheatMap = anticheatMap;
        }

        @Override
        public OrderSnapshotDto apply(QihoOrderSnapshotEntity input) {
            if (input == null) {
                return null;
            }
            OrderSnapshotDto dto = new OrderSnapshotDto();
            BeanUtils.copy(input, dto);
            dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(input.getFundStatus()));
            dto.setOrderStatusEnum(OrderStatusEnum.getByCode(input.getOrderStatus()));
            if (null != LogisticsOrderStatusEnum.getByCode(input.getLogisticsStatus())) {
                dto.setLogisticsStatus(LogisticsOrderStatusEnum.getByCode(input.getLogisticsStatus()).getDesc());
            }

            dto.setDeliveryEnum(DeliveryEnum.getByCode(input.getDelivery()));
            if (anticheatMap != null) {
                dto.setAnticheatRules(anticheatMap.get(input.getOrderId()));
            }
            return dto;
        }
    }

    @Override
    public Integer exportOrderPageCount(OrderPageParams params) {
        if (!Objects.equal(null, params.getOrderId())) {
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if (order != null) {
                return 1;
            } else {
                return 0;
            }
        } else {
            return qihoOrderSnapshotDAO.exportOrderPageCount(params);
        }
    }

    @Override
    public Integer updateOrderStatus(String orderId, String remark, OrderStatusEnum orderStatus,
        OrderApprovalStatusEnum orderApprovalStatusEnum) {
        return qihoOrderSnapshotDAO.updateOrderStatus(orderId, remark, orderStatus, orderApprovalStatusEnum, null);
    }

    @Override
    public OrderSnapshotDto findByOrderId(String orderId) {
        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if (entity == null) {
            return null;
        }
        OrderSnapshotDto dto = BeanUtils.copy(entity, OrderSnapshotDto.class);
        dto.setOrderStatusEnum(OrderStatusEnum.getByCode(entity.getOrderStatus()));
        dto.setDeliveryEnum(DeliveryEnum.getByCode(entity.getDelivery()));
        dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(entity.getFundStatus()));
        return dto;
    }

    /**
     * 监听订单创建事件
     *
     * @param orderCreateEvent
     * @author zhanglihui
     */
    @Subscribe
    @AllowConcurrentEvents
    public void createOrderListener(OrderCreateEvent orderCreateEvent) {
        // 添加IP手机归属地检测
        ipMobileCheckService.checkToSave(orderCreateEvent.getOrderSnapshot());
        // 添加超时打印
        DBTimeProfile.enter("create order");

        OrderSnapshotDto orderSnapshotDto = orderCreateEvent.getOrderSnapshot();
        String orderId = orderSnapshotDto.getOrderId();
        // 保存订单详情页短链接
        String shortUrl = saveShortUrl(orderId);
        if (StringUtils.equals(orderSnapshotDto.getPayType(), PayTypeEnum.COD.getCode())) {
            orderSnapshotDto.setGmtCreate(new Date());

            // 发送下单成功短信
            DBTimeProfile.enter("send message");
            boolean flag = sendMessage(orderSnapshotDto, shortUrl);
            DBTimeProfile.release();

            if(flag){
                // 防作弊规则扫描
                DBTimeProfile.enter("shot order");
                shotOrderEngine.process(orderSnapshotDto);
                DBTimeProfile.release();
            }
        }

        // 获取设备指纹ID
        updateDeviceId(orderId, orderCreateEvent.getTokenId());

        // 推啊api转化对接下单URL、皮肤id是否为空
        if (StringUtils.isBlank(orderSnapshotDto.getUrl()) || orderSnapshotDto.getSkinId() == null) {
            LOG.warn("下单URL为空.{}", orderSnapshotDto);
            DBTimeProfile.release();
            return;
        }

        // 推送埋点至推啊
        LOG.info("推啊埋点推送,orderId：{}.{}", orderSnapshotDto.getOrderId(), orderSnapshotDto);
        DBTimeProfile.enter("forward to tuia");
        forwardToTuia(orderSnapshotDto);
        DBTimeProfile.release();

        DBTimeProfile.release();
    }

    /**
     * 监听 订单从临时订单写入事件
     * @param orderCreateEvent
     */
    @Subscribe
    @AllowConcurrentEvents
    private void createOrderFromTmpListener(OrderCreateFromTmpEvent orderCreateEvent){

        OrderSnapshotDto orderSnapshotDto = orderCreateEvent.getOrderSnapshot();
        if (orderSnapshotDto == null ) {
            return;
        }

        String orderId = orderCreateEvent.getOrderSnapshot().getOrderId();

        // 获取设备指纹ID
        updateDeviceId(orderId, orderCreateEvent.getTokenId());


        // 推啊api转化对接下单URL、皮肤id是否为空
        if (StringUtils.isBlank(orderSnapshotDto.getUrl()) || orderSnapshotDto.getSkinId() == null) {
            LOG.warn("下单URL为空.{}", orderSnapshotDto);
            return;
        }

        // 推送埋点至推啊
        LOG.info("推啊埋点推送,orderId：{}.{}", orderSnapshotDto.getOrderId(), orderSnapshotDto);
        forwardToTuia(orderSnapshotDto);


        BaiqiOrderSmsEntity orderSms = baiqiOrderSmsMapper.findByOrderId(orderId);
        if (null == orderSms) {
            LOG.warn("获取不到下单短信记录，orderId：{}", orderId);
            return;
        }

        // 非创蓝短信的直接跑踢单规则 或者是创蓝的且有了返回消息 或者通过定时任务有了结果
        if (orderSms.getSmsType() != SmsTypeEnum.CHUANG_LAN.getNum() ||
                (orderSms.getSmsType() == SmsTypeEnum.CHUANG_LAN.getNum()
                        && orderSms.getMobileStatus() != SmsMobileStatusEnum.DEFAULT.getNum())) {
            orderSnapshotDto.setSmsRetCode(orderSms.getSyncResult());
            shotOrderEngine.process(orderSnapshotDto);
        }

        // 发送给盛世嘉祥
        toSsjx(orderSnapshotDto, orderId, orderSms );
    }





    /**
     * 监听付款方式更新成货到付款事件
     *
     * @param event
     * @author zhanglihui
     */
    @Subscribe
    public void updateToCodListener(UpdateToCodEvent event) {

        // 添加超时打印
        DBTimeProfile.enter("updateToCodListener");

        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(event.getOrderId());
        // 防作弊规则扫描
        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(order, OrderSnapshotDto.class);
        shotOrderEngine.process(orderSnapshotDto);
        String shortUrl = ShortUrlUtil.getShortUrlByOrderId(event.getOrderId());
        // 发送下单成功短信
        sendMessage(orderSnapshotDto, shortUrl);

        DBTimeProfile.release();

    }

    /**
     * @param orderId
     * @return
     * @author zhanglihui
     */
    private String saveShortUrl(String orderId) {

        String qihoWebUrl = DomainConstantUtil.getQihoWebUrl();
        // 手机端域名
        String domain = StringUtils.substring(qihoWebUrl, StringUtils.indexOf(qihoWebUrl, "//") + 2);
        // 订单详情页长链接完整地址
        String longUrl = qihoWebUrl + "/order/detail?orderId=" + orderId;
        // 短链接地址
        int section = Integer.parseInt(StringUtils.substring(orderId, 15, 19));
        String shortUrl = ShortUrlUtil.generateShortUrl(longUrl, section);
        // 保存长链接和短链接的映射关系
        QihoShortUrlEntity qihoShortUrlEntity = new QihoShortUrlEntity();
        qihoShortUrlEntity.setShortUrl(shortUrl);
        qihoShortUrlEntity.setLongUrl(longUrl);
        qihoShortUrlDAO.insert(qihoShortUrlEntity);
        return domain + "/" + shortUrl;
    }

    /**
     * @param tokenId
     * @param orderId
     * @author zhanglihui
     */
    private void updateDeviceId(String orderId, String tokenId) {
        FraudApiResponse fraudApiResponse = TongdunUtil.getRiskInfo(tokenId);
        if(fraudApiResponse != null){
            String tongdunId = fraudApiResponse.getDeviceId();
            ChannelInfoDto channelInfo = new ChannelInfoDto();
            channelInfo.addDeviceId(ChannelInfoDto.TOKEN_ID, tokenId);
            channelInfo.setOrderId(orderId);
            channelInfo.addDeviceId(ChannelInfoDto.TONGDUN_DEVICE_ID, tongdunId);

            String maxentId = MaxentUtil.getDeviceId(tokenId);
            channelInfo.addDeviceId(ChannelInfoDto.MAXENT_DEVICE_ID, maxentId);
            channelInfoService.updateByOrderId(channelInfo);
        }
    }


    @Autowired
    ChuangLanSmsCommitMapper chuangLanSmsCommitMapper;

    /**
     * @param
     * @author zhanglihui
     */
    private boolean sendMessage(OrderSnapshotDto orderSnapshotDto, String orderDetailUrl) {

        String orderId = orderSnapshotDto.getOrderId();

        String mobile = orderSnapshotDto.getMobile();

        String itemShortName = orderSnapshotDto.getItemShortName();

        // 用户下单成功短信通知
        QihoTemplateEntity template = null;
        if (mediaWhiteListService.isInWhiteList(orderId)) {
            template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCC_400.getCode());
        } else {
            template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCCESS.getCode());
        }

        OrderSnapshotDto entity = orderSnapshotService.findByOrderId(orderId);

        String smsRetCode = YZHSmsConstant.YZHCODE.UNKNOWN;
        DBTimeProfile.enter("send sms");
        boolean smsFlag = true;
        if (null != template && template.getTemplateEnable()) {
            Map<String, String> context = Maps.newHashMap();
            context.put("orderDetailUrl", orderDetailUrl);
            context.put("itemName", itemShortName);
            try {
                template.setReport(true);
                smsRetCode = smsService.singleSendRetCode(template, context, mobile);
                smsFlag = chuangLanHandler(orderSnapshotDto, template, smsRetCode, smsFlag);
            } catch (Exception e) {
                LOG.warn("发送短信错误", e);
                smsRetCode = YZHSmsConstant.YZHCODE.TIME_OUT;
            }
        }
        DBTimeProfile.release();
        toSsjx(orderSnapshotDto, orderId, template, entity, smsRetCode);

        // 发送下单后温馨提示
        QihoTemplateEntity template1 = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_NOTIFY.getCode());
        if (template1.getTemplateEnable()) {
            smsService.singleSend(template1, mobile);
        }

        if(template.getSmsType().equals(SmsTypeEnum.CHUANG_LAN.getCode())&& template.getTemplateEnable()&&smsFlag){
            return false;
        }else {
            return true;
        }

    }

    private boolean chuangLanHandler(OrderSnapshotDto orderSnapshotDto, QihoTemplateEntity template, String smsRetCode,
        boolean smsFlag) {
        if(template.getSmsType().equals(SmsTypeEnum.CHUANG_LAN.getCode())){
            //记录短信与订单的关系
            JSONObject jsonObject = JSONObject.parseObject(smsRetCode);
            //判断失败的情况
            if(StringUtils.isBlank(smsRetCode)||StringUtils.isBlank(jsonObject.getString("msgId"))){
                smsFlag = false;
            }else{
                ChuangLanSmsCommitEntity smsCommitEntity = new ChuangLanSmsCommitEntity();
                smsCommitEntity.setOrderId(orderSnapshotDto.getOrderId());
                smsCommitEntity.setMsgId(jsonObject.getString("msgId"));
                smsCommitEntity.setHandlerStatus(0);
                chuangLanSmsCommitMapper.insert(smsCommitEntity);
            }
        }
        return smsFlag;
    }

    private void toSsjx(OrderSnapshotDto orderSnapshotDto, String orderId, QihoTemplateEntity template,
        OrderSnapshotDto entity, String smsRetCode) {
        //如果不是创蓝 reCode码正常推送  否则推送一个空的
        orderSnapshotDto.setSmsRetCode(smsRetCode);

        List<Long> ssjxMerchartIds = StringUtil.stringToListOrderByIds(ssjxMerchartId);

        BizLog.log("用户下单成功短信已发送, orderId:{}, merchantId:{}, smsRetCode:{}, ssjxMerchartId:{}, OrderSnapshotDto:{}",
            orderId, entity.getMerchantId(), smsRetCode, ssjxMerchartId, entity);

        if (StringUtils.isNotBlank(smsRetCode) && !CollectionUtils.isEmpty(ssjxMerchartIds) && ssjxMerchartIds
            .contains(entity.getMerchantId())) {

            // 盛世嘉祥返回值
            String retCode = this.getSsjxSmsRetCode(smsRetCode);

            BizLog.log("盛世嘉祥下单成功发短信, orderId:{}, merchantId:{}, retCode:{}, ssjxMerchartId:{}", orderId,
                entity.getMerchantId(), retCode, ssjxMerchartId);
            if(template.getSmsType().equals(SmsTypeEnum.CHUANG_LAN.getCode())){
                ssjxPushOrderBo.createOrderSnapshotListener(entity, "");
            }else{
                ssjxPushOrderBo.createOrderSnapshotListener(entity, retCode);
            }
        }
    }

    /**
     * 推送给盛世嘉祥
     * @param orderSnapshotDto
     * @param orderId
     */
    private void toSsjx(OrderSnapshotDto orderSnapshotDto, String orderId, BaiqiOrderSmsEntity orderSms) {

        List<Long> ssjxMerchartIds = StringUtil.stringToListOrderByIds(ssjxMerchartId);
        if (CollectionUtils.isNotEmpty(ssjxMerchartIds) && ssjxMerchartIds
                .contains(orderSnapshotDto.getMerchantId())) {
            if (orderSms.getSmsType().intValue() == SmsTypeEnum.CHUANG_LAN.getNum()) {
                // 如果是创蓝的 推送空的返回码
                ssjxPushOrderBo.createOrderSnapshotListener(orderSnapshotDto, "");
            } else {
                ssjxPushOrderBo.createOrderSnapshotListener(orderSnapshotDto, this.getSsjxSmsRetCode(orderSms.getSyncResult()));
            }
        }
    }


    private String getSsjxSmsRetCode(String smsCode) {
        if (YZHSmsConstant.YZHCODE.SUCCESS_CODE.equals(smsCode)) {
            return "DELIVRD";
        } else if (YZHSmsConstant.YZHCODE.TIME_OUT.equals(smsCode)) {
            return "TIMEOUT";
        } else if (YZHSmsConstant.YZHCODE.UNKNOWN.equals(smsCode) || YZHSmsConstant.YZHCODE.EMPTY_NUM.equals(smsCode)) {
            return "UNDELIV";
        } else {
            return smsCode;
        }
    }

    /**
     * 物流子订单更新事件
     *
     * @param event
     */
    @Subscribe
    public void logisticsInfoUpdateListener(LogisticsInfoUpdateEvent event) {

        // 添加超时打印
        DBTimeProfile.enter("logisticsInfoUpdateListener");

        // 发送发货短信
        String postId = event.getPostId();
        String mobile = event.getMobile();

        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SEND.getCode());
        if (template.getTemplateEnable()) {
            Map<String, String> context = Maps.newHashMap();
            context.put("postId", postId);
            context.put("orderDetailUrl", ShortUrlUtil.getShortUrlByOrderId(event.getOrderId()));
            smsService.singleSend(template, context, mobile);
        }

        DBTimeProfile.release();
    }

    /**
     *
     */
    @Override
    public List<OrderSnapshotDto> findByMobile(String mobile, List<String> statusList) {
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.findByMobile(mobile, statusList);
        List<OrderSnapshotDto> orderList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return orderList;
        }
        for (QihoOrderSnapshotEntity input : list) {
            OrderSnapshotDto dto = new OrderSnapshotDto();
            BeanUtils.copy(input, dto);
            dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(input.getFundStatus()));
            dto.setOrderStatusEnum(OrderStatusEnum.getByCode(input.getOrderStatus()));
            dto.setDeliveryEnum(DeliveryEnum.getByCode(input.getDelivery()));
            orderList.add(dto);
        }
        return orderList;
    }

    @Override
    public Integer queryItemSaleCount(Long itemId, Date startTime, Date endTime) {
        Integer saleNum = qihoOrderSnapshotDAO.queryItemSaleCount(itemId, startTime, endTime);
        return null == saleNum ? 0 : saleNum;
    }

    @Override
    public Integer updateOrderProcessWayByOrderId(OrderGainParams orderGainParams) {
        return qihoOrderSnapshotDAO.updateOrderProcessWayByOrderId(orderGainParams);
    }

    @Override
    public List<OrderSnapshotDto> queryOrderByOrderIds(List<String> orderIds) {
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.queryOrderByOrderIds(orderIds);
        List<OrderSnapshotDto> orderList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return orderList;
        }
        return list.stream().map(this::entityToDto).collect(Collectors.toList());
    }

    @Override
    public OrderSnapshotDto queryByMobileAsNewMost(String mobile, String orderStatus) {
        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.queryByMobileAsNewMost(mobile, orderStatus);
        return entityToDto(entity);
    }

    private OrderSnapshotDto entityToDto(QihoOrderSnapshotEntity entity) {
        if (null == entity) {
            return new OrderSnapshotDto();
        }
        OrderSnapshotDto dto = BeanUtils.copy(entity, OrderSnapshotDto.class);
        dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(entity.getFundStatus()));
        dto.setOrderStatusEnum(OrderStatusEnum.getByCode(entity.getOrderStatus()));
        dto.setDeliveryEnum(DeliveryEnum.getByCode(entity.getDelivery()));
        return dto;
    }

    @Override
    public Map<Long, Integer> queryItemSaleCountBatch(List<Long> itemIdList, Date startTime, Date endTime) {
        Map<Long, Integer> map = Maps.newHashMap();
        //现将所有的出事数量设置为0
        itemIdList.stream().forEach(itemId -> map.put(itemId, 0));
        List<Map<String, Object>> maps = qihoOrderSnapshotDAO.queryItemSaleCountBatch(itemIdList, startTime, endTime);
        maps.stream().forEach(e -> {
            Long itemId = null;
            Integer saleNum = 0;
            for (Map.Entry<String, Object> ma : e.entrySet()) {
                if (StringUtils.equals("itemId", ma.getKey())) {
                    itemId = (Long) ma.getValue();
                } else if (StringUtils.equals("saleNum", ma.getKey())) {
                    saleNum = ((BigDecimal) ma.getValue()).intValue();
                }
            }
            map.put(itemId, saleNum);
        });
        return map;
    }

    @Override
    public List<OrderSnapshotDto> queryOrderAsNewMost(OrderPageParams pageParams) {
        List<QihoOrderSnapshotEntity> orderSnapshotEntities = qihoOrderSnapshotDAO.queryOrderAsNewMost(pageParams);
        return BeanUtils.copyList(orderSnapshotEntities, OrderSnapshotDto.class);
    }

    @Override
    public List<OrderSnapshotDto> getHandlerOrderSnapshotDto(List<QihoOrderSnapshotEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        List<String> orderIds = Lists.newArrayList();
        for (QihoOrderSnapshotEntity order : list) {
            orderIds.add(order.getOrderId());
        }
        List<QihoAnticheateEntity> anticheatList = qihoAnticheateDAO.findByOrderIds(orderIds);
        Map<String, String> map = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(anticheatList)) {
            for (QihoAnticheateEntity anticheat : anticheatList) {
                String rules = map.get(anticheat.getOrderId());
                if (StringUtils.isNotBlank(rules)) {
                    map.put(anticheat.getOrderId(), rules + "," + anticheat.getRule());
                } else {
                    map.put(anticheat.getOrderId(), anticheat.getRule());
                }
            }
        }
        DtoTransform transform = new DtoTransform();
        transform.setAnticheatMap(map);
        return Lists.transform(list, transform);
    }

    @Override
    public List<QihoOrderSnapshotEntity> findByMobileAndItem(String mobile, Long itemId) {
        return qihoOrderSnapshotDAO.findByMobileAndItem(mobile, itemId);
    }

    @Override
    public OrderSnapshotDto findByOderIdAndMobile(String orderId, String mobile) {
        return BeanUtils.copy(qihoOrderSnapshotDAO.findByOderIdAndMobile(orderId, mobile),OrderSnapshotDto.class);
    }

    /**
     * 推啊api转化对接
     *
     * @param orderSnapshotDto
     */
    private void forwardToTuia(OrderSnapshotDto orderSnapshotDto) {
        BizLog.log(orderSnapshotDto.getUrl());
        String paramUrl = "";
        try {
            paramUrl = new URL(orderSnapshotDto.getUrl()).getQuery();
        } catch (MalformedURLException e) {
            LOG.info("前端传入URL解析获取参数失败：{}", orderSnapshotDto.getUrl());
        }
        ForwardToTuiaParams params = getParamsByUrl(paramUrl);
        String url = tuiaUrl + "?a_oId=" + params.getA_oId() + "&a_cid=" + params.getA_cid() + "&a_timeStamp=" + params
            .getA_timeStamp() + "&type=" + params.getType();

        String callbackData;

        // 发送url到推啊
        if (StringUtils.isNotBlank(params.getA_oId()) && params.getA_cid() != null) {
            callbackData = HttpClientUtil.sendGet(url);
            LOG.info("发送推啊转换日志，orderId：{}, URL：{}, result:{}", orderSnapshotDto.getOrderId(), url, callbackData);
        } else {
            LOG.info("参数不完整, 不发送埋点, orderId：{}.{}", orderSnapshotDto.getOrderId(), orderSnapshotDto);
            callbackData = CALLBACK_DATA;
        }

        EmbedDto embedDto = new EmbedDto();
        embedDto.setOrderId(orderSnapshotDto.getOrderId());
        embedDto.setResult(getEmbedResult(callbackData));
        embedDto.setUrl(orderSnapshotDto.getUrl());
        embedDto.setCallbackData(callbackData);
        embedDto.setSkinId(orderSnapshotDto.getSkinId());
        embedDto.setGmtCreate(new Date());
        embedDto.setGmtModified(new Date());
        embedService.insert(embedDto);
    }

    /**
     * 从下单url中解析参数
     *
     * @param url
     *
     * @return
     */
    private ForwardToTuiaParams getParamsByUrl(String url) {
        String[] params;
        if (url.contains(TUIA_URL_POSTFIX)) {
            params = url.substring(url.indexOf("?") + 1, url.indexOf(TUIA_URL_POSTFIX)).split("&");
        } else {
            params = url.substring(url.indexOf("?") + 1).split("&");
        }
        Map<String, String> map = new HashMap<>();
        for (String param : params) {
            map.put(param.substring(0, param.indexOf("=")), param.substring(param.indexOf("=") + 1));
        }
        ForwardToTuiaParams forwardToTuiaParams = new ForwardToTuiaParams();
        forwardToTuiaParams.setA_oId(map.get("a_oId"));
        if (StringUtils.isBlank(forwardToTuiaParams.getA_oId())) {
            forwardToTuiaParams.setA_oId(map.get("a_tuiaId"));
        }

        if (StringUtils.isNotBlank(map.get("a_cid"))) {
            forwardToTuiaParams.setA_cid(map.get("a_cid"));
        }
        forwardToTuiaParams.setChannelId(map.get("channelId"));
        forwardToTuiaParams.setA_timeStamp(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
        return forwardToTuiaParams;
    }

    /**
     * 根据推啊返回错误码返回对应的埋点结果
     */
    private EmbedResultEnum getEmbedResult(String callbackData) {
        if (CALLBACK_DATA.equals(callbackData)) {
            return EmbedResultEnum.UNSEND;
        } else if (callbackData.contains("0000000")) {
            return EmbedResultEnum.SUCCESS;
        } else {
            return EmbedResultEnum.FAILED;
        }
    }

    private List<String> getSkinIdList(String skinId) {
        if (StringUtils.isBlank(skinId)) {
            return Collections.emptyList();
        }
        return Arrays.asList(skinId.split(","));
    }

}
