package com.qiho.center.biz.service.impl.order;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSONObject;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.dto.ordertmp.OrderTmpDetailDto;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.ShotOrder.ShotOrderFieldEnum;
import com.qiho.center.biz.engine.action.OrderMobileAction;
import com.qiho.center.biz.service.ordertmp.FilterRuleHitService;
import com.qiho.center.common.daoh.qiho.order.StrategyMerchantMapper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.qiho.center.api.enums.ShotOrder.StrategyType;
import com.qiho.center.api.enums.StrategyUseType;
import com.qiho.center.api.params.OrderStrategyParams;
import com.qiho.center.biz.service.order.OrderStrategyService;
import com.qiho.center.biz.service.order.StrategyMerchantService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoOrderStrategyDAO;
import com.qiho.center.common.entity.order.QihoOrderStrategyEntity;
import com.qiho.center.common.entityd.qiho.order.StrategyMerchantEntity;

import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * Created by danke on 2017/6/21.
 */
@Service
public class OrderStrategyServiceImpl implements OrderStrategyService {

    /**
     * 日志
     */
    Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private QihoOrderStrategyDAO qihoOrderStrategyDAO;

    @Resource
    private StrategyMerchantService strategyMerchantService;

    @Resource
    private StrategyMerchantMapper strategyMerchantMapper;

    @Autowired
    private OrderMobileAction orderMobileAction;

    @Autowired
    private FilterRuleHitService filterRuleHitService;


    @Override
    public OrderStrategyDto findStrategyById(Long id) {
        QihoOrderStrategyEntity entity = qihoOrderStrategyDAO.findStrategyById(id);
        if (null == entity)
            return null;
        return BeanUtils.copy(entity, OrderStrategyDto.class);
    }

    @Override
    public PagenationDto<OrderStrategyDto> findStrategyByQuery(OrderStrategyParams params, Integer pageNum,
        Integer pageSize) {
        List<OrderStrategyDto> channelDtos = new ArrayList<>();
        PagenationDto<OrderStrategyDto> resultBase = new PagenationDto<>();
        List<QihoOrderStrategyEntity> entities = qihoOrderStrategyDAO.findStrategyByQuery(params, pageNum, pageSize);
        if (CollectionUtils.isEmpty(entities))
            return resultBase;
        //遍历集合,封装参数
        entities.stream().forEach(e -> channelDtos.add(BeanUtils.copy(e, OrderStrategyDto.class)));
        resultBase.setList(channelDtos);
        //查询total
        Integer channelCount = qihoOrderStrategyDAO.countByQuery(params);
        resultBase.setTotal(channelCount);
        return resultBase;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer updateStrategy(OrderStrategyDto orderStrategyDto) {

        qihoOrderStrategyDAO.updateByEntity(orderStrategyDto);

        //适用部分商家处理
        StrategyUseType useType = StrategyUseType.findByValue(orderStrategyDto.getUseType());

        //所有商家
        if (StrategyUseType.ALL.equals(useType)) {

            //删除策略与商家关联
            strategyMerchantService.delMerchantStrategy(null, orderStrategyDto.getId());

            //部分商家
        } else if (StrategyUseType.PART.equals(useType)) {

            // 更新关联(先删除后插入)
            strategyMerchantService.delMerchantStrategy(null, orderStrategyDto.getId());

            handleStrategyMerchant(orderStrategyDto.getId(), orderStrategyDto);
        }

        return 1;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer deleteStrategyById(Long id, String updator) {
        return qihoOrderStrategyDAO.deleteById(id, updator);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer insertStrategy(OrderStrategyDto orderStrategyDto) {

        Long result = qihoOrderStrategyDAO.insertStrategy(orderStrategyDto);

        handleStrategyMerchant(result, orderStrategyDto);

        return result == null ? null : Integer.valueOf(result.toString());
    }

    private void handleStrategyMerchant(Long strategyId, OrderStrategyDto orderStrategyDto) {
        //适用部分商家处理
        StrategyUseType useType = StrategyUseType.findByValue(orderStrategyDto.getUseType());

        if (StrategyUseType.PART.equals(useType)) {
            List<SimpleIdAndNameDto> merchants = orderStrategyDto.getMerchants();

            List<StrategyMerchantEntity> insertList = Lists.newArrayListWithExpectedSize(merchants.size());
            merchants.forEach(merchant -> {

                StrategyMerchantEntity merchantEntity = new StrategyMerchantEntity();
                merchantEntity.setMerchantId(merchant.getId());
                merchantEntity.setStrategyId(strategyId);

                insertList.add(merchantEntity);
            });

            strategyMerchantService.batchInsert(insertList);
        }
    }

    @Override
    public Integer countStrategyByQuery(OrderStrategyParams params) {
        return qihoOrderStrategyDAO.countByQuery(params);
    }

    @Override
    public OrderStrategyDto findStrategyByStrategyType(StrategyType type) {
        QihoOrderStrategyEntity entity = qihoOrderStrategyDAO.findStrategyByStrategyType(type.getCode());
        if (null == entity)
            return null;
        return BeanUtils.copy(entity, OrderStrategyDto.class);
    }

    @Override
    public boolean needCheckPhoneStatues(Long merchantId, StrategyType strategyType) {
        List<QihoOrderStrategyEntity> strategyEntities = getStrategy(merchantId, strategyType);

        if (CollectionUtils.isEmpty(strategyEntities)) {
            return false;
        }
        // 循环比对策略字段 存在手机号校验的策略即返回true
        for (QihoOrderStrategyEntity orderStrategy : strategyEntities) {
            String ruleValue = orderStrategy.getRulevalue();
            if (StringUtils.isBlank(ruleValue)) {
                continue;
            }
            List<StrategyRuleDto> strategyRuleDtos = JSONObject.parseArray(ruleValue,StrategyRuleDto.class);
            if (CollectionUtils.isEmpty(strategyRuleDtos)) {
                continue;
            }
            for (StrategyRuleDto strategyRuleDto : strategyRuleDtos) {
                if (StringUtils.equals(strategyRuleDto.getFieldName(), ShotOrderFieldEnum.PHONE_STATUS.getVal())) {
                    // 存在这种策略
                    return true;
                }
            }
        }

        return false;
    }


    /**
     * 获取适用于当前商家的规则
     * @param merchantId
     * @param strategyType
     * @return
     */
    private List<QihoOrderStrategyEntity> getStrategy(Long merchantId, StrategyType strategyType){
        // 获取适用全部商家的规则
        OrderStrategyParams param = new OrderStrategyParams();
        param.setStrategyType(strategyType.getCode());
        param.setUseType(StrategyUseType.ALL.getValue());
        List<QihoOrderStrategyEntity> strategyEntities = qihoOrderStrategyDAO.findByParam(param);

        // 获取适用部分商家的规则
        List<StrategyMerchantEntity> strategyMerchantEntities = strategyMerchantMapper.findByMerchantId(merchantId);
        if (CollectionUtils.isNotEmpty(strategyMerchantEntities)) {
            List<Long> strategyIds = strategyMerchantEntities.stream().map(StrategyMerchantEntity::getStrategyId).collect(Collectors.toList());
            param.setIdList(strategyIds);
            param.setUseType(StrategyUseType.PART.getValue());
            List<QihoOrderStrategyEntity> partStrategyEntities = qihoOrderStrategyDAO.findByParam(param);
            strategyEntities.addAll(partStrategyEntities);
        }
        return strategyEntities;
    }


    @Override
    public boolean checkPhoneStatusForFilterRule(OrderTmpDetailDto orderTmpDetailDto) {
        boolean needCheck = this.needCheckPhoneStatues(orderTmpDetailDto.getMerchantId(),StrategyType.FILTER_ORDER_RULE);
        if (!needCheck) {
            // 不需要校验 返回true
            return true;
        }

        List<QihoOrderStrategyEntity> strategyEntities = getStrategy(orderTmpDetailDto.getMerchantId(), StrategyType.FILTER_ORDER_RULE);

        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(orderTmpDetailDto, OrderSnapshotDto.class);

        for (QihoOrderStrategyEntity orderStrategyEntity : strategyEntities) {
            List<StrategyRuleDto> strategyRuleDtos = JSONObject.parseArray(orderStrategyEntity.getRulevalue(),StrategyRuleDto.class);
            if (CollectionUtils.isEmpty(strategyRuleDtos)) {
                continue;
            }

            for (StrategyRuleDto strategyRuleDto : strategyRuleDtos) {
                if (!StringUtils.equals(strategyRuleDto.getFieldName(), ShotOrderFieldEnum.PHONE_STATUS.getVal())) {
                    continue;
                }

                // 进行校验
                ResultBase<Boolean> resultBase = orderMobileAction.valuteCostomByQuery(orderSnapshotDto, strategyRuleDto);
                if (resultBase.hasSuccessValue() && resultBase.getValue()) {
                    // 命中了规则
                    filterRuleHitService.insert(orderSnapshotDto.getOrderId(), orderStrategyEntity.getStrategyName());
                    return false;
                }
            }
        }

        return true;
    }
}
