/**
 * Project Name:qiho-center-biz File Name:SmsServiceImpl.java Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月3日下午12:32:06 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl.order;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Resource;

import com.qiho.center.common.constant.DomainConstantUtil;
import com.qiho.center.common.dao.QihoShortUrlDAO;
import com.qiho.center.common.entity.QihoShortUrlEntity;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.LogisticsNameEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.enums.SmsTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.YunPianPushParams;
import com.qiho.center.biz.event.FundInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.process.note.SmsProcess;
import com.qiho.center.biz.process.note.SmsProcessFactory;
import com.qiho.center.biz.service.MediaWhiteListService;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.dao.QihoTemplateDAO;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.ShortUrlUtil;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duibaboot.ext.autoconfigure.accesslog.MD5;

/**
 * ClassName:SmsServiceImpl <br/>
 * 短信服务实现类. <br/>
 * Date: 2017年6月3日 下午12:32:06 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Service
@BizEventListener
public class SmsServiceImpl implements SmsService {

    private static final Logger LOG = LoggerFactory.getLogger(SmsServiceImpl.class);

    @Value("${yunpian.sms.url}")
    private String smsUrl;

    @Value("${yunpian.apikey}")
    private String apikey;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoTemplateDAO qihoTemplateDAO;

    @Autowired
    private MediaWhiteListService mediaWhiteListService;

    @Autowired
    private QihoShortUrlDAO qihoShortUrlDAO;

    @Resource
    private SmsProcessFactory smsProcessFactory;

    /**
     * @see com.qiho.center.biz.service.order.SmsService#sendSmsCode(java.lang.String, java.lang.String)
     */
    @Override
    public boolean sendSmsCode(String mobile, String smsCode) {
        Map<String, String> context = Maps.newHashMap();
        context.put("code", smsCode);
        QihoTemplateEntity entity = qihoTemplateDAO.findByCode(SmsTemplateEnum.SMS_CODE.getCode());
        return null == entity || !entity.getTemplateEnable() || singleSend(entity, context, mobile);
    }

    @Override
    public boolean singleSend(QihoTemplateEntity templateEntity, Map<String, String> context, String mobile) {
        SmsProcess smsProcess = getSmsProcessByType(templateEntity);
        if (null == smsProcess)
            throw new QihoException("短信模板配置有误,找不到对应的短信渠道处理器");
        if (null == context)
            return smsProcess.smsSend(templateEntity, mobile);
        else
            return smsProcess.smsSendByParam(templateEntity, context, mobile);
    }

    @Override
    public String singleSendRetCode(QihoTemplateEntity templateEntity, Map<String, String> context, String mobile) {
        SmsProcess smsProcess = getSmsProcessByType(templateEntity);
        if (null == smsProcess) {
            throw new QihoException("短信模板配置有误,找不到对应的短信渠道处理器");
        }

        if (null == context) {
            return smsProcess.smsSendRetCode(templateEntity, mobile);
        } else {
            return smsProcess.smsSendByParamRetCode(templateEntity, context, mobile);
        }
    }

    @Override
    public boolean singleSend(QihoTemplateEntity entity, String mobile) {
        return singleSend(entity, null, mobile);
    }

    @Override
    public boolean checkSign(YunPianPushParams params) {
        String signStr = this.getCheckSignFromParams(params);
        try {
            String sign = MD5.md5(signStr);
            return sign.equals(params.getSign());
        } catch (Exception e) {
            AppLogUtil.error(LOG, "MD5加密异常", e);
            return false;
        }
    }

    private String getCheckSignFromParams(YunPianPushParams params) {
        StringBuilder sb = new StringBuilder();
        sb.append(params.getBaseExtend()).append(",").append(params.getExtend()).append(",").append(params.getId())
            .append(",").append(params.getMobile()).append(",").append(params.getReplyTime()).append(",")
            .append(params.getText()).append(",").append(apikey);
        return sb.toString().trim();
    }

    /**
     * 监听物流状态更新事件
     *
     * @param event
     *
     * @author zhanglihui
     */
    @Subscribe
    public void updateLogisticsListener(LogisticsUpdateEvent event) {
        // 添加超时打印
        DBTimeProfile.enter("updateLogisticsListener");

        // 获取要发送短信的手机号码
        String orderId = event.getOrderId();
        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(orderId);

        if (Objects.equal(null, order) || StringUtils.isBlank(order.getMobile())) {
            LOG.info("订单不存在，或者手机号为空，无法发送短信.orderId={}", orderId);

            DBTimeProfile.release();
            return;
        }
        //具体发送短信逻辑
        sendSms(event, order);

        DBTimeProfile.release();
    }

    /**
     * 发送短信
     *
     * @param event
     * @param order
     */
    private void sendSms(LogisticsUpdateEvent event, QihoOrderSnapshotEntity order) {
        String mobile = order.getMobile();
        //将之前根据圆通物流状态判断改成通用的奇货物流订单状态
        LogisticsOrderStatusEnum status = event.getLogisticsOrderStatusEnum();
        switch (status) {
        case SENT_SCAN:
            //区分圆通和快递100的短信模板
            SmsTemplateEnum sendTemplate = getTemplate(event, order);
            QihoTemplateEntity entity = qihoTemplateDAO.findByCode(sendTemplate.getCode());
            if (entity != null && entity.getTemplateEnable() && null != getSmsProcessByType(entity))
                getSmsProcessByType(entity).smsSendByParam(entity, event.getDeliveryMsg(), mobile);
            break;
        case SUCCESS:
            QihoTemplateEntity signedEntity = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_RECIVED.getCode());
            if (signedEntity.getTemplateEnable() && null != getSmsProcessByType(signedEntity)){
                Map<String,String> params = new HashMap<>(16);
                params.put("userEvaluateUrl",saveShortUrl(order.getOrderId(),order.getMobile()));
                getSmsProcessByType(signedEntity).smsSendByParam(signedEntity,params,mobile);
            }
            break;
        default:
            break;
        }
    }

    /**
     * 生成短链
     * @param orderId
     * @return
     */
    private String saveShortUrl(String orderId,String mobile) {

        String qihoWebUrl = DomainConstantUtil.getQihoWebUrl();
        // 手机端域名
        String domain = StringUtils.substring(qihoWebUrl, StringUtils.indexOf(qihoWebUrl, "//") + 2);
        // 订单详情页长链接完整地址
        String longUrl = qihoWebUrl + "/order/evaluate?orderId=" + orderId + "&mobile="+mobile;
        // 短链接地址
        int section = Integer.parseInt(StringUtils.substring(orderId, 15, 19));
        String shortUrl = ShortUrlUtil.generateShortUrl(longUrl, section);
        // 保存长链接和短链接的映射关系
        QihoShortUrlEntity qihoShortUrlEntity = new QihoShortUrlEntity();
        qihoShortUrlEntity.setShortUrl(shortUrl);
        qihoShortUrlEntity.setLongUrl(longUrl);
        qihoShortUrlDAO.insert(qihoShortUrlEntity);
        return domain + "/" + shortUrl;
    }

    /**
     * 获取派件中短信模板
     *
     * @param event
     * @param order
     *
     * @return
     */
    private SmsTemplateEnum getTemplate(LogisticsUpdateEvent event, QihoOrderSnapshotEntity order) {
        if (DeliveryEnum.ERP.getCode().equals(order.getDelivery()) && LogisticsNameEnum.YTO.getCode()
            .equals(event.getLogisticsCode())) {
            return StringUtils.equals(order.getPayType(), PayTypeEnum.COD.getCode()) ?
                SmsTemplateEnum.ORDER_DELIVERY :
                SmsTemplateEnum.ORDER_DELIVERY_PAID;
        } else {
            return StringUtils.equals(order.getPayType(), PayTypeEnum.COD.getCode()) ?
                SmsTemplateEnum.ORDER_DELIVERY_100 :
                SmsTemplateEnum.ORDER_DELIVERY_PAID_100;
        }
    }

    /**
     * @param event
     *
     * @author zhanglihui
     */
    @Subscribe
    public void paySuccessListener(FundInfoUpdateEvent event) {

        // 添加超时打印
        DBTimeProfile.enter("paySuccessListener");

        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(event.getOrderId());
        
        // 用户下单成功短信通知
        QihoTemplateEntity template = null;
        if (mediaWhiteListService.isInWhiteList(event.getOrderId())) {
            template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCC_400.getCode());
        } else {
            template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCCESS.getCode());
        }
        String mobile = null;
        if (null != template && template.getTemplateEnable()) {
            mobile = order.getMobile();
            Map<String, String> context = Maps.newHashMap();
            context.put("orderDetailUrl", ShortUrlUtil.getShortUrlByOrderId(event.getOrderId()));
            context.put("itemName", order.getItemShortName());
            SmsProcess smsProcess = getSmsProcessByType(template);
            if (null != smsProcess){
                smsProcess.smsSendByParam(template, context, mobile);
            }
        }
        //如果是在线支付的订单,还需要发送短信
        if (null != event.getPayType() && !StringUtils.equals(event.getPayType(), PayTypeEnum.COD.getCode())) {
            QihoTemplateEntity paySuccessTemp = qihoTemplateDAO.findByCode(SmsTemplateEnum.PAY_SUCCESS.getCode());
            if (paySuccessTemp.getTemplateEnable() && null != getSmsProcessByType(paySuccessTemp)) {
                getSmsProcessByType(paySuccessTemp).smsSend(paySuccessTemp, mobile);
            }
        }

        DBTimeProfile.release();
    }

    /**
     * 获取对应的短信渠道处理类
     *
     * @param templateEntity
     *
     * @return
     */
    private SmsProcess getSmsProcessByType(QihoTemplateEntity templateEntity) {
        String smsType = templateEntity.getSmsType();
        SmsTypeEnum smsTypeEnum = SmsTypeEnum.fromCode(smsType);
        return smsProcessFactory.decidePayChannel(smsTypeEnum);
    }
}
