package com.qiho.center.biz.service.impl.tag;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.tag.TagDto;
import com.qiho.center.biz.service.tag.ItemTagService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.daoh.qiho.tag.QihoItemTagMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoTagMapper;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoItemTagEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoTagEntity;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author chensong
 * @create 2018-03-26 17:19
 **/
@Service
public class ItemTagServiceImpl implements ItemTagService{

    @Autowired
    private QihoItemTagMapper qihoItemTagMapper;

    @Autowired
    private QihoTagMapper qihoTagMapper;

    @Autowired
    private QihoItemDAO qihoItemDAO;



    @Override
    public List<TagDto> findByItemId(Long itemId) {
        List<QihoItemTagEntity> itemTags = qihoItemTagMapper.findByItemId(itemId);
        if (CollectionUtils.isEmpty(itemTags)) {
            return Lists.newArrayList();
        }
        List<Long> tagIds = itemTags.stream().map(QihoItemTagEntity::getTagId).collect(Collectors.toList());
        List<QihoTagEntity> tagList = qihoTagMapper.findByIds(tagIds);
        return BeanUtils.copyList(tagList, TagDto.class);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> saveItemTag(Long itemId, List<Long> newTagIds) {
        // 商品id校验
        if (Objects.equals(null, itemId) || itemId.longValue() <= 0) {
            return ResultDto.failResult("参数异常，itemId不合法");
        }
        QihoItemEntity itemEntity = qihoItemDAO.findSimpleById(itemId);
        if (Objects.equals(null, itemEntity)) {
            return ResultDto.failResult("商品不存在或已删除");
        }
        // 若新的tagIds为空  则删除原有的
        if (CollectionUtils.isEmpty(newTagIds)) {
            qihoItemTagMapper.deleteByItemId(itemId);
            return ResultDto.successResult(Boolean.TRUE);
        }
        // 过滤掉无效的tagId
        List<QihoTagEntity> tagEntities = qihoTagMapper.findByIds(newTagIds);
        newTagIds = tagEntities.stream().map(e -> e.getId()).collect(Collectors.toList());
        // 原有的tagIds和新的tagIds求差集  表示需要变动的对应关系
        List<Long> copyTagIds = newTagIds.stream().collect(Collectors.toList());
        List<QihoItemTagEntity> itemTagEntities = qihoItemTagMapper.findByItemId(itemId);  // 原有的对应关系
        List<Long> originTagIds = itemTagEntities.stream().map(QihoItemTagEntity::getTagId).collect(Collectors.toList());
        // 求交集 copyTagIds剩下不需要变动的tagIds
        copyTagIds.retainAll(originTagIds);
        newTagIds.removeAll(copyTagIds);     //新增的tagIds
        originTagIds.removeAll(copyTagIds);  //需要删除的tagIds
        // 删除不需要的数据
        if (CollectionUtils.isNotEmpty(originTagIds)) {
            Map<String, Object> param = Maps.newHashMap();
            param.put("itemId", itemId);
            param.put("tagIds",originTagIds);
            qihoItemTagMapper.deleteByParam(param);
        }
        // 新增关系
        if (CollectionUtils.isNotEmpty(newTagIds)) {
            List<QihoItemTagEntity> itemTagParam = newTagIds.stream().map(e -> {
                QihoItemTagEntity qihoItemTagEntity = new QihoItemTagEntity();
                qihoItemTagEntity.setItemId(itemId);
                qihoItemTagEntity.setTagId(e);
                return qihoItemTagEntity;
            }).collect(Collectors.toList());
            qihoItemTagMapper.insertBatch(itemTagParam);
        }
        return ResultDto.successResult(Boolean.TRUE);
    }

    @Override
    public Boolean copyItemTag(Long originItemId, Long newItemId) {
        List<QihoItemTagEntity> itemTagEntities = qihoItemTagMapper.findByItemId(originItemId);
        if (CollectionUtils.isEmpty(itemTagEntities)) {
            // 如果原商品没有关联标签 直接返回
            return Boolean.TRUE;
        }
        itemTagEntities.stream().forEach(e-> e.setItemId(newItemId));
        qihoItemTagMapper.insertBatch(itemTagEntities);
        return Boolean.TRUE;
    }

    @Override
    public Boolean deleteByItemId(Long itemId) {
        qihoItemTagMapper.deleteByItemId(itemId);
        return Boolean.TRUE;
    }

    @Override
    public Boolean batchDeleteByItemId(List<Long> itemIdList) {
        if (CollectionUtils.isEmpty(itemIdList)) {
            return Boolean.TRUE;
        }
        qihoItemTagMapper.deleteByItemIdList(itemIdList);
        return Boolean.TRUE;
    }

}
