package com.qiho.center.biz.task;

import com.qiho.center.api.enums.ErpStatusEnum;
import com.qiho.center.api.enums.ErpTypeEnum;
import com.qiho.center.api.enums.OrderApprovalStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.task.bo.OrderConfirmDo;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entityd.qiho.OrderConfirmTaskDetailEntity;
import com.qiho.center.common.entityd.qiho.OrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.SequenceNoBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import java.util.concurrent.CountDownLatch;

/**
 * Created by weny.cai on 2018/3/29.
 */
@Component
@Scope("prototype")
public abstract class OrderBoRunnable implements Runnable {

    private final Logger logger = LoggerFactory.getLogger(OrderBoRunnable.class);

    @Autowired
    LogisticsOrderService logisticsOrderService;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoOrderDAO qihoOrderDAO;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private SequenceNoBuilder sequenceNoBuilder;

    /**
     * 处理的任务
     */
    BatchOrderTask task;

    /**
     * 工作统计 CountDownLatch
     */
    final CountDownLatch countDownLatch;

    /**
     * 处理详情结果
     */
    OrderConfirmTaskDetailEntity detailEntity;

    public OrderBoRunnable(BatchOrderTask batchOrderTask, CountDownLatch countDownLatch) {
        this.task = batchOrderTask;
        this.countDownLatch = countDownLatch;
    }

    /**
     * 处理
     * 单个订单线程
     */
    @Override
    public void run() {
        while (true) {
            OrderConfirmDo orderConfirmDo = task.getBlockingQueue().poll();
            if (orderConfirmDo == null) {
                break;
            }
            Integer lineNum = orderConfirmDo.getLineNum();
            try {
                OrderSnapshotEntity orderSnapshotEntity = task.getEntityMap().get(orderConfirmDo.getOrderId());
                QihoErpOrderEntity erpOrderEntity = task.getErpOrderEntityMap().get(orderConfirmDo.getOrderId());
                boolean checkFlag = this.checkParam(orderConfirmDo, orderSnapshotEntity);
                if (!checkFlag) {
                    task.addDetailEntity(detailEntity);
                    task.failLines.add(lineNum);
                    task.failNum.addAndGet(1);
                    continue;
                }
                this.auditOrder(orderConfirmDo, orderSnapshotEntity, erpOrderEntity);
            } catch (Exception e) {
                logger.error("任务编号:" + task.dto.getTaskId() + "数据库操作失败！");
                e.printStackTrace();
            } finally {
                task.ops.increment("handleCount", 1);
            }
        }
        countDownLatch.countDown();
    }

    abstract boolean checkParam(OrderConfirmDo confirmDo, OrderSnapshotEntity orderSnapshotEntity);

    abstract void auditOrder(OrderConfirmDo bo, OrderSnapshotEntity orderSnapshotEntity, QihoErpOrderEntity erpOrderEntity);

    /**
     * 订单待审核到待发货
     *
     * @param order
     * @return
     */
    @Transactional(value = DsConstants.DATABASE_QIHO, rollbackFor = Exception.class)
    public QihoErpOrderEntity auditOrderToSuccess(OrderSnapshotEntity order) {


        OrderAuditParams params = new OrderAuditParams();
        params.setOrderId(order.getOrderId());
        qihoOrderDAO.auditOrderToSuccess(params);
        qihoOrderSnapshotDAO.updateOrderStatus(params.getOrderId(), params.getRemark(), OrderStatusEnum.TO_SEND,
                OrderApprovalStatusEnum.APPROVE_SUCCESS, null);

        // 创建ERP子订单
        QihoErpOrderEntity erpOrder = new QihoErpOrderEntity();
        erpOrder.setOrderId(order.getOrderId());
        erpOrder.setDelivery(order.getDelivery());
        erpOrder.setItemId(order.getItemId());
        erpOrder.setItemName(order.getItemName());
        erpOrder.setItemNo(order.getItemNo());
        erpOrder.setSkuNo(order.getSkuNo());
        erpOrder.setConsumerName(order.getConsumerName());
        erpOrder.setMobile(order.getMobile());
        erpOrder.setProvince(order.getProvince());
        erpOrder.setCity(order.getCity());
        erpOrder.setDistrict(order.getDistrict());
        erpOrder.setAddress(order.getAddress());
        erpOrder.setErpType(ErpTypeEnum.SEND.getCode());
        erpOrder.setErpId(sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.ERP, order.getItemId()));
        erpOrder.setErpStatus(ErpStatusEnum.TO_NOTIFY.getCode());
        qihoErpOrderDAO.createErpOrder(erpOrder);
        return erpOrder;
    }
}
