package com.qiho.center.biz.bo;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncAckDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.enums.ErpErrorCodeReturnEnum;
import com.qiho.center.api.enums.LogisticsNameEnum;
import com.qiho.center.biz.runnable.LogisticsOrderSignRunnable;
import com.qiho.center.biz.service.EdbApiService;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.ERPClientUtil;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

/**
 * ClassName: LogisticsOrderBo <br/>
 * date: 2017年6月8日 下午6:04:21 <br/>
 *
 * @author chencheng
 * @version
 */
@Service
public class LogisticsOrderBo implements ApplicationContextAware {

    private static final Logger   LOG                            = LoggerFactory.getLogger(LogisticsOrderBo.class);

    /** ERP系统返回物流信息条数，ERP系统最多支持100条 */
    private static final String   QUERY_ERP_LOGISTICS_SYNC_LIMIT = "100";

    /** 是否支持拆单发货 0 否 1 是 */
    private static final String   IS_PART_SYNC_ABLE              = "0";

    private ApplicationContext    context;

    @Autowired
    private LogisticsOrderService logisticsOrderService;

    @Resource
    private ExecutorService       executorService;

    @Resource
    private EdbApiService edbApiService;

    /**
     * 定时任务，同步订单发货物流信息
     *
     * @author chencheng
     */
    public void erpLogisticsSyncQuery() {
        while (true) {
            // ERP系统同步物流信息
            Map<String, String> params = Maps.newHashMap();
            params.put("shop_no", ERPClientUtil.getShopNo());
            params.put("limit", QUERY_ERP_LOGISTICS_SYNC_LIMIT);
            params.put("is_part_sync_able", IS_PART_SYNC_ABLE);
            String logisticsJson = null;
            //设置全局参数
            Boolean brokenStatus = Boolean.FALSE;
            List<ErpLogisticsSyncDto> list = null;
            try {
                logisticsJson = ERPClientUtil.execute(ERPClientUtil.getLogisticsSyncUrl(), params);
                list = getLogisticsOrderDtoList(logisticsJson);
                if (CollectionUtils.isEmpty(list)) {
                    LOG.info("ERP系统同步物流信息完成。无待同步物流信息");
                    brokenStatus = Boolean.TRUE;
                }
            } catch (Exception e) {
                LOG.error("ERP系统同步物流信息失败，发送请求失败", e);
                brokenStatus = Boolean.TRUE;
            }
            if(brokenStatus){
                break;
            }
            // 创建物流子订单
            List<ErpLogisticsSyncAckDto> listReurn = logisticsOrderService.createLogisticsOrderBatch(list);
            if (CollectionUtils.isNotEmpty(listReurn)) {
                Map<String, String> returnParams = Maps.newHashMap();
                returnParams.put("logistics_list", JSON.toJSONString(listReurn));
                try {
                    ERPClientUtil.execute(ERPClientUtil.getLogisticsAckUrl(), returnParams);
                    Thread.sleep(2000L);
                } catch (Exception e) {
                    LOG.error("返回ERP系统同步物流信息处理结果，请求失败", e);
                }
            }
        }
    }

    /**
     * 定时任务，从e店宝同步订单发货物流信息
     *
     */
    public void edbErpLogisticsSyncQuery(){
        int pageNo = 1;
        SimpleDateFormat fd = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Calendar calendar = Calendar.getInstance();
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        String endTime = fd.format(calendar.getTime());
        if (hour == 9){
            calendar.add(Calendar.HOUR,-14);
        }else {
            calendar.add(Calendar.HOUR,-1);
            calendar.add(Calendar.MINUTE,-30);
        }
        String beginTime = fd.format(calendar.getTime());
        while (true){
            //循环获取e店宝订单物流信息
            List<ErpLogisticsSyncDto> items = edbApiService.edbTradeGet(pageNo,QUERY_ERP_LOGISTICS_SYNC_LIMIT,beginTime,endTime);
            if (CollectionUtils.isEmpty(items)){
                break;
            }
            // 创建物流子订单
            logisticsOrderService.createLogisticsOrderBatch(items);
            pageNo ++;
        }
    }

    /**
     * 解析erp系统返回物流信息json
     *
     * @author chencheng
     * @param logisticsJson
     * @return List<ErpLogisticsSyncDto>
     */
    @SuppressWarnings("unchecked")
    public List<ErpLogisticsSyncDto> getLogisticsOrderDtoList(String logisticsJson) {
        List<ErpLogisticsSyncDto> list = Lists.newArrayList();
        try {
            if (StringUtils.isBlank(logisticsJson)) {
                return ListUtils.EMPTY_LIST;
            }
            JSONObject json = JSON.parseObject(logisticsJson);
            String code = json.getString("code");
            // 错误码为失败
            if (!StringUtils.equals(code, ErpErrorCodeReturnEnum.SUCCESS.getCode())) {
                return ListUtils.EMPTY_LIST;
            }
            JSONArray temp = JSON.parseArray(json.get("trades").toString());
            // 无待同步物流订单信息
            if (CollectionUtils.isEmpty(temp)) {
                return ListUtils.EMPTY_LIST;
            }
            for (int j = 0; j < temp.size(); j++) {
                JSONObject t = temp.getJSONObject(j);
                ErpLogisticsSyncDto queryLogisticsReturnDto = JSON.toJavaObject(t, ErpLogisticsSyncDto.class);
                LogisticsNameEnum logisticsName = LogisticsNameEnum.getByName(queryLogisticsReturnDto.getLogisticsName());
                if(logisticsName != null){
                    queryLogisticsReturnDto.setLogisticsCode(logisticsName.getCode());
                }
                list.add(queryLogisticsReturnDto);
            }
        } catch (Exception e) {
            AppLogUtil.warn(LOG, "同步物流订单失败", e);
            return ListUtils.EMPTY_LIST;
        }
        return list;
    }

    public Integer batchSignOrder(String progressKey, List<LogisticsOrderDto> paramList) {
        LogisticsOrderSignRunnable runnable = context.getBean(LogisticsOrderSignRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramList);
        int enableCount = runnable.enableFilter();
        executorService.submit(runnable);
        return enableCount;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.context = applicationContext;
    }

}
