package com.qiho.center.biz.bo;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ConfigConstant;
import com.qiho.center.api.dto.media.MediaWarningDto;
import com.qiho.center.api.enums.ordertmp.OrderTmpStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.ordertmp.OrderTmpPageParam;
import com.qiho.center.common.dao.QihoChannelInfoDAO;
import com.qiho.center.common.dao.QihoConfigDAO;
import com.qiho.center.common.daoh.qiho.media.BaiqiMediaWarningMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderTmpExtMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderTmpMapper;
import com.qiho.center.common.daoh.qihostatistics.BaiqiTuiaAppStatisMapper;
import com.qiho.center.common.entity.QihoConfigEntity;
import com.qiho.center.common.entity.order.QihoChannelInfoEntity;
import com.qiho.center.common.entityd.qiho.media.BaiqiMediaWarningEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderTmpExtEntity;
import com.qiho.center.common.entityd.qihostatistics.BaiqiTuiaAppStatisEntity;
import com.qiho.center.common.util.DingTalkUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Description:媒体订单监控任务
 *
 * @author chensong
 * @create 2018-06-14
 */
@Component
public class MediaOrderBo {

    private static final Logger LOGGER = LoggerFactory.getLogger(MediaOrderBo.class);

    /** 默认告警阈值 */
    private static final BigDecimal DEAULT_THRESHOLD = new BigDecimal("0.1");

    /** 最小告警阈值 1% */
    private static final BigDecimal MIN_THRESHOLD = new BigDecimal("0.01");

    /** 最大告警阈值 100% */
    private static final BigDecimal MAX_THRESHOLD = new BigDecimal("1");

    /** 默认的订单数告警阈值 **/
    private static final BigDecimal DEFAULT_ORDER_NUM_THRESHOLD = new BigDecimal("10");

    /** 数值格式化 */
    private static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat("#.##");

    /** 钉钉告警机器人地址 */
    @Value("${dingtalk.mediaorder.url}")
    private String dingTalkRobot;

    @Autowired
    private BaiqiOrderTmpMapper baiqiOrderTmpMapper;

    @Autowired
    private BaiqiOrderTmpExtMapper baiqiOrderTmpExtMapper;

    @Autowired
    private QihoChannelInfoDAO qihoChannelInfoDAO;

    @Autowired
    private QihoConfigDAO qihoConfigDAO;

    @Autowired
    private BaiqiTuiaAppStatisMapper baiqiTuiaAppStatisMapper;

    @Autowired
    private BaiqiMediaWarningMapper baiqiMediaWarningMapper;


    /**
     * 媒体订单监控任务主执行方法
     * ->从临时表获取最近一小时的无效订单，去除没有appid的订单，获得appId无效的订单量
     * ->从tb_qiho_channel_info表获取最近一小时的appId 订单量 获取告警阈值 比对告警阈值
     * ->保存结果 发送钉钉告警
     */
    public void doTask(){

        Date startTime = new DateTime().minusHours(1).toDate();
        Date endTime = new Date();
        Map<Long, Integer> invalidAppIdMap = Maps.newHashMap();
        List<Long> invalidAppIdList = Lists.newArrayList();

        // 1、获取无效订单的 过滤掉appId为null的订单 以及统计每个appId的无效订单量
        getInvalidAppId(invalidAppIdMap, invalidAppIdList, startTime, endTime);
        if (CollectionUtils.isEmpty(invalidAppIdList)) {
            return ;
        }

        // 2、获取正式AppId 以及统计其订单量
        Map<Long, Integer> normalAppIdMap = getNormalAppId(invalidAppIdList, startTime, endTime);

        // 3、获取告警阈值
        BigDecimal threshold = getWarningThreshold();

        // 4、对比无效订单和正式订单 获超过到告警阈值的appId
        List<MediaWarningDto> mediaWarningDtoList = getWarningAppId(threshold, invalidAppIdMap, normalAppIdMap, startTime, endTime);
        if (CollectionUtils.isEmpty(mediaWarningDtoList)) {
            return;
        }

        // 5、获取app名称
        obtainAppName(mediaWarningDtoList);

        // 6、将告警数据保存到数据库
        saveWarningData(mediaWarningDtoList);

        // 7、发送钉钉告警
        sendDingTalkMsg(mediaWarningDtoList);

    }


    /**
     * 获取规定时间段内的无效订单
     * 过滤掉无appId的订单 并统计每个appId的无效订单数量
     * @param invalidAppIdMap
     */
    private void getInvalidAppId(Map<Long, Integer> invalidAppIdMap, List<Long> invalidAppIdList, Date startTime, Date endTime){
        // 获取时间段内无效的订单
        OrderTmpPageParam orderTmpParam = new OrderTmpPageParam();
        orderTmpParam.setStartTime(startTime);
        orderTmpParam.setEndTime(endTime);
        orderTmpParam.setOrderStatus(OrderTmpStatusEnum.INVALID.getNum());
        List<String> orderIds = baiqiOrderTmpMapper.findOrderIdsByParam(orderTmpParam);
        if (CollectionUtils.isEmpty(orderIds)) {
            return;
        }

        // 过滤掉appId为null的无效订单 并统计appId 的无效订单数量
        List<BaiqiOrderTmpExtEntity> orderTmpExtEntityList = baiqiOrderTmpExtMapper.findBatchByOrderIds(orderIds);
        orderTmpExtEntityList.stream().filter(e -> e.getAppId() != null).forEach(e->{
            invalidAppIdList.add(e.getAppId());

            if (invalidAppIdMap.get(e.getAppId()) != null) {
                Integer num = invalidAppIdMap.get(e.getAppId());
                invalidAppIdMap.put(e.getAppId(), num + 1);
            } else {
                invalidAppIdMap.put(e.getAppId(), 1);
            }
        });
    }

    /**
     * 查询规定时间段内的正式订单appId 的数量
     * @param appIdList
     * @return key->appId value->正式订单数量
     */
    private Map<Long, Integer> getNormalAppId(List<Long> appIdList, Date startTime, Date endTime){
        Map<Long, Integer> normalAppIdMap = Maps.newHashMap();

        // 查询时间段内的订单 从渠道表查 根据appIdList获取
        Map<String, Object> channelInfoParam = Maps.newHashMap();
        channelInfoParam.put("startTime", startTime);
        channelInfoParam.put("endTime", endTime);
        channelInfoParam.put("appIdList", appIdList);
        List<QihoChannelInfoEntity> channelInfoEntities = qihoChannelInfoDAO.findByAppIdList(channelInfoParam);
        if (CollectionUtils.isEmpty(channelInfoEntities)) {
            return normalAppIdMap;
        }

        // 统计appId 的订单量
        channelInfoEntities.forEach(e->{
            if (normalAppIdMap.get(e.getAppId()) != null) {
                Integer num = normalAppIdMap.get(e.getAppId());
                normalAppIdMap.put(e.getAppId(), num+1);
            } else {
                normalAppIdMap.put(e.getAppId(), 1);
            }
        });

        return normalAppIdMap;
    }

    /**
     * 获取告警阈值 从tb_qiho_config表获取
     * 最低1% 最高100% 默认10%
     * @return 告警阈值 小数形式
     */
    private BigDecimal getWarningThreshold(){
        QihoConfigEntity configEntity = qihoConfigDAO.findConfigByName(ConfigConstant.MEDIA_MONITOR_THRESHOLD);
        if (configEntity == null) {
            // 查不到配置 返回默认的告警阈值
            LOGGER.warn("查询不到媒体告警阈值，返回默认阈值");
            return DEAULT_THRESHOLD;
        }

        String configValue = configEntity.getConfigValue();
        if (StringUtils.isBlank(configValue)) {
            LOGGER.error("配置媒体告警阈值为空");
            return DEAULT_THRESHOLD;
        }

        try {
            BigDecimal warningThreshold = new BigDecimal(configValue);
            if (warningThreshold.compareTo(MIN_THRESHOLD) < 0 || warningThreshold.compareTo(MAX_THRESHOLD) > 0) {
                throw new QihoException("阈值配置错误");
            }
            return warningThreshold;
        } catch (Exception e) {
            LOGGER.error("解析媒体告警阈值错误，返回默认阈值，配置的value是：{}", configValue, e);
            return DEAULT_THRESHOLD;
        }
    }

    /**
     * 对比无效订单和正式订单 获取超过告警阈值的appId
     * @param threshold
     * @param invalidAppIdMap
     * @param normalAppIdMap
     * @return 需要告警的媒体名单
     */
    private List<MediaWarningDto> getWarningAppId(BigDecimal threshold, Map<Long, Integer> invalidAppIdMap,
                                                  Map<Long, Integer> normalAppIdMap, Date startTime, Date endTime){
        // 订单数告警阈值
        BigDecimal orderNumThreshold = getOrderNumThreshold();
        List<MediaWarningDto> warningDtoList = Lists.newArrayList();

        // 遍历无效订单的appId 获取需要告警的appId
        invalidAppIdMap.forEach((key, value) -> {
            Integer normalOrderNum = normalAppIdMap.get(key);
            if (normalOrderNum == null) {
                normalOrderNum = 0;
            }

            // 无效订单数
            BigDecimal invalidNum = new BigDecimal(value);
            // 正式订单数
            BigDecimal normalNum = new BigDecimal(normalOrderNum);

            // 当 无效订单+正式订单>=订单数告警阈值 才加入告警列表
            if (invalidNum.add(normalNum).compareTo(orderNumThreshold) < 0) {
                return ;
            }

            // 无效率 = 无效订单/(无效订单+有效订单) 保留4位小数
            BigDecimal invalidRate = invalidNum.divide(invalidNum.add(normalNum), 4, BigDecimal.ROUND_HALF_UP);
            if (invalidRate.compareTo(threshold) >= 0){
                // 无效率超过了阈值 将该媒体数据加入告警列表
                MediaWarningDto mediaWarningDto = new MediaWarningDto();
                mediaWarningDto.setAppId(key);
                mediaWarningDto.setInvalidOrderNum(value);
                mediaWarningDto.setThreshold(threshold);
                mediaWarningDto.setStartTime(startTime);
                mediaWarningDto.setEndTime(endTime);
                mediaWarningDto.setNormalOrderNum(normalOrderNum);
                mediaWarningDto.setInvalidRate(invalidRate);
                warningDtoList.add(mediaWarningDto);
            }
        });

        return warningDtoList;
    }

    /**
     * 获取订单数量告警阈值
     * 当媒体订单 无效+正式的数量大于等于此数值时 才会加入告警列表
     * 从tb_qiho_config 查 configname = MEDIA_ORDER_NUM_MONITOR_THRESHOLD 的配置项， 配置的configValue为整数形式
     * @return
     */
    private BigDecimal getOrderNumThreshold(){

        try{
            QihoConfigEntity configEntity = qihoConfigDAO.findConfigByName(ConfigConstant.MEDIA_ORDER_NUM_MONITOR_THRESHOLD);
            if (configEntity == null) {
                // 查不到配置 返回默认的告警阈值
                LOGGER.warn("查询不到媒体订单数量告警阈值，返回默认阈值");
                return DEFAULT_ORDER_NUM_THRESHOLD;
            }

            String configValue = configEntity.getConfigValue();
            if (!StringUtils.isNumeric(configValue)) {
                throw new QihoException("媒体订单数量告警阈值配置错误， 当前数据是" + configValue);
            }

            return new BigDecimal(configValue);
        } catch (Exception e){
            // 出异常返回默认的告警阈值
            LOGGER.error("获取订单数量告警阈值异常 ", e);
            return DEFAULT_ORDER_NUM_THRESHOLD;
        }
    }


    /**
     * 获取app媒体名称
     * @param mediaWarningDtoList
     */
    private void obtainAppName(List<MediaWarningDto> mediaWarningDtoList){

        // 从推啊媒体名单批量获取媒体名单数据
        List<Long> appIdList = mediaWarningDtoList.stream().map(MediaWarningDto::getAppId).collect(Collectors.toList());
        List<BaiqiTuiaAppStatisEntity> tuiaAppEntityList = baiqiTuiaAppStatisMapper.listMediaByAppIds(appIdList);

        // 将媒体信息封装在Map中 key->appId value->appName
        Map<Long, String> appNameMap = Maps.newHashMap();
        tuiaAppEntityList.forEach(e -> appNameMap.put(e.getAppId(), e.getAppName()));

        // 遍历告警名单 补充appName
        mediaWarningDtoList.forEach(e -> {
            String appName = appNameMap.get(e.getAppId());
            if (StringUtils.isBlank(appName)) {
                e.setAppName(StringUtils.EMPTY);
            } else {
                e.setAppName(appName);
            }
        });
    }

    /**
     * 保存告警数据到数据库
     * @param mediaWarningDtoList
     */
    private void saveWarningData(List<MediaWarningDto> mediaWarningDtoList){
        // 将MediaWarningDto转成Entity
        List<BaiqiMediaWarningEntity> mediaWarningEntities = mediaWarningDtoList.stream().map(e -> {
            BaiqiMediaWarningEntity mediaWarningEntity = BeanUtils.copy(e, BaiqiMediaWarningEntity.class);
            mediaWarningEntity.setThreshold(e.getThreshold().multiply(new BigDecimal("10000")).intValue());
            mediaWarningEntity.setInvalidRate(e.getInvalidRate().multiply(new BigDecimal("10000")).intValue());
            return mediaWarningEntity;
        }).collect(Collectors.toList());

        // 批量保存
        baiqiMediaWarningMapper.insertBatch(mediaWarningEntities);
    }


    /**
     * 发送钉钉告警
     * 拼接 媒体id、媒体名称、无效订单数量、有效订单数量、无效占比 等信息
     * @param mediaWarningDtoList
     */
    private void sendDingTalkMsg(List<MediaWarningDto> mediaWarningDtoList){
        StringBuilder msg = new StringBuilder("媒体防作弊监控告警：\n");
        for (MediaWarningDto mediaWarning : mediaWarningDtoList) {
            msg.append("媒体ID：").append(mediaWarning.getAppId()).append("，媒体名称：").append(mediaWarning.getAppName())
                    .append("，无效订单数：").append(mediaWarning.getInvalidOrderNum()).append("，有效订单数：").append(mediaWarning.getNormalOrderNum())
                    .append("，无效订单占比达").append(convertRateToPercent(mediaWarning.getInvalidRate())).append("\n");
        }
        msg.append("请及时关注");

        // 调用钉钉发送消息工具
        DingTalkUtil.sendTextMessageWith(msg.toString(), dingTalkRobot,true);
    }


    /**
     * 将无效订单率转化成 % 形式
     * @param invalidRate
     * @return
     */
    private String convertRateToPercent(BigDecimal invalidRate){
        BigDecimal b = invalidRate.multiply(new BigDecimal("100"));
        return DECIMAL_FORMAT.format(b) + "%";
    }




}
