package com.qiho.center.biz.engine.action;

import cn.com.duiba.wolf.utils.DateUtils;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.StrategyRuleDto;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.ShotOrder.ShotOrderRuleEnum;
import com.qiho.center.api.enums.ShotOrder.StrategyType;
import com.qiho.center.common.dao.QihoChannelInfoDAO;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderTmpExtMapper;
import com.qiho.center.common.entity.order.QihoChannelInfoEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderTmpExtEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Resource;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 24小时同一推啊用户id下单量校验 action
 *
 * @author jiafeng.geng
 * @create 2018-06-06 19:50:24
 **/
public class RepeatOrderTuiacidAction extends ShotOrderAbstractAction {

    private static final Logger LOGGER = LoggerFactory.getLogger(RepeatOrderTuiacidAction.class);

    @Resource
    private BaiqiOrderTmpExtMapper orderTmpExtMapper;

    @Resource
    private QihoChannelInfoDAO qihoChannelInfoDAO;

    @Override
    public ResultBase<Boolean> valuteCostomByQuery(OrderSnapshotDto orderSnapshotDto, StrategyRuleDto strategyRuleDto) {

        /*
         *  判断当前的规则类型：踢单规则、下单规则、过滤规则，并根据不同的规则使用不同的方法获取 推啊用户ID(tuiaCid)
         *  判断tuiaCid是否为空，为空则不进行规则校验
         *  获取 24小时下单量
         *  根据阈值和下单量进行校验
         */
        Map<String, String> map = getTuiaCid(orderSnapshotDto);
        String tuiaCid = map.get("tuiaCid");

        // 当推啊用户ID为空时，不命中 24小时下单量(推啊用户id统计) 规则
        if (StringUtils.isEmpty(tuiaCid) || StringUtils.equals(tuiaCid, Integer.toString(0))) {
            LOGGER.info("推啊用户ID(a_cid)为空，不校验 24小时下单量(推啊用户id统计) 规则!");
            return ResultBase.rightReturn(false);
        }

        // 获取 24小时下单量
        int count = getCount(tuiaCid);
        // 进行规则校验
        return ShotOrderRuleEnum.fromVal(strategyRuleDto.getRuleName())
                .valuteOrderByQuery(strategyRuleDto.getThreshold(), count);
    }

    private int getCount(String tuiaCid) {
        // 封装参数
        Date startTime = DateUtils.daysAddOrSub(new Date(), -1);
        QihoChannelInfoEntity params = new QihoChannelInfoEntity();
        params.setStartTime(startTime);
        params.setTuiaCid(tuiaCid);

        List<QihoChannelInfoEntity> channelInfoEntityList = qihoChannelInfoDAO.countByParams(params);
        if (CollectionUtils.isEmpty(channelInfoEntityList)) {
            return -1;
        }

        return channelInfoEntityList.size();
    }

    private Map<String, String> getTuiaCid(OrderSnapshotDto orderSnapshotDto) {
        Map<String, String> map = new HashMap<>();
        String tuiaCid;

        // 判断orderId是否存在：若存在，则为踢单规则或过滤规则；若不存在，则为下单规则
        if (StringUtils.isNotEmpty(orderSnapshotDto.getOrderId())) {
            // 根据orderId查询临时订单表，判断是否有返回结果：若有则说明当前订单为待过滤状态，即过滤规则；若没有则为踢单规则
            BaiqiOrderTmpExtEntity orderTmpExtEntity = orderTmpExtMapper.findByOrderId(orderSnapshotDto.getOrderId());

            if (null != orderTmpExtEntity) {
                // 过滤规则，直接从临时订单对象中获取tuiaCid
                tuiaCid = orderTmpExtEntity.getTuiaCid();
                map.put("strategyType", StrategyType.FILTER_ORDER_RULE.getValue());
            } else {
                // 踢单规则，根据orderId查询渠道信息表，从渠道信息表中获取tuiaCid
                tuiaCid = qihoChannelInfoDAO.findByOrderId(orderSnapshotDto.getOrderId()).getTuiaCid();
                map.put("strategyType", StrategyType.SHOT_ORDER.getValue());
            }
        } else {
            // 下单规则，从orderSnapshotDto中获取tuiaCid
            tuiaCid = orderSnapshotDto.getTuiaCid();
            map.put("strategyType", StrategyType.AHEAD_PREVENT_ORDER.getValue());
        }
        map.put("tuiaCid", tuiaCid);
        return map;
    }

}
