package com.qiho.center.biz.engine.impl;

import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.OrderStrategyDto;
import com.qiho.center.api.dto.StrategyRuleDto;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.dto.strategy.FilterOrderCheckDto;
import com.qiho.center.api.enums.ShotOrder.ShotOrderFieldEnum;
import com.qiho.center.api.enums.ShotOrder.StrategyType;
import com.qiho.center.api.enums.StrategyUseType;
import com.qiho.center.biz.engine.BaseEngine;
import com.qiho.center.biz.engine.BaseHandler;
import com.qiho.center.biz.engine.Factory.ProcessFaceory;
import com.qiho.center.biz.engine.ProcessContext;
import com.qiho.center.biz.engine.chain.BusEntiyHandler;
import com.qiho.center.biz.engine.task.filter.FilterOrderTask;
import com.qiho.center.biz.service.order.StrategyMerchantService;
import com.qiho.center.biz.service.ordertmp.FilterRuleHitService;
import com.qiho.center.common.entityd.qiho.order.StrategyMerchantEntity;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/** 过滤规则执行器
 * @author chensong
 * @create 2018-05-21 09:44
 **/
public class FilterOrderEngine extends BaseOrderEngine implements BaseEngine<FilterOrderCheckDto> {


    private static final Logger LOGGER = LoggerFactory.getLogger(FilterOrderEngine.class);

    /**
     * 校验链路入口类
     * {@link BusEntiyHandler}
     */
    private BaseHandler baseHandler;

    @Autowired
    private StrategyMerchantService strategyMerchantService;

    @Autowired
    private FilterRuleHitService filterRuleHitService;

    @Autowired
    private FilterOrderTask filterOrderTask;

    @Override
    public FilterOrderCheckDto process(FilterOrderCheckDto valuteData) {

        if (null == valuteData || null == valuteData.getOrderSnapshotDto()) {
            // 数据为空直接返回
            return valuteData;
        }

        try {
            List<OrderStrategyDto> orderStrategyDtos = getOrderStrategyList(StrategyType.FILTER_ORDER_RULE);
            if (CollectionUtils.isEmpty(orderStrategyDtos)) {
                // 规则为空直接返回
                return valuteData;
            }

            return this.filterRuleCheck(valuteData, orderStrategyDtos);

        } catch (Exception e){
            LOGGER.error("过滤规则失败，orderId,{}",valuteData.getOrderSnapshotDto().getOrderId(), e);
        }
        return valuteData;
    }

    /**
     * 过滤规则校验
     * @param result
     * @param orderStrategyDtos
     * @return
     */
    private FilterOrderCheckDto filterRuleCheck(FilterOrderCheckDto result, List<OrderStrategyDto> orderStrategyDtos){

        // 标记这条规则是否命中 默认false-不命中
        boolean flag = false;

        Set<String> handleResult = new HashSet<>(8);

        // 循环遍历每一条规则
        for (OrderStrategyDto orderStrategyDto : orderStrategyDtos) {
            Boolean match = isOrderMatchStrategy(orderStrategyDto, result.getOrderSnapshotDto());
            if(!match){
                continue;
            }

            ResultBase<List<StrategyRuleDto>> resultBase = super.validateorderStrategyDto(orderStrategyDto);
            if(!resultBase.hasSuccessValue()) {
                LOGGER.warn("过滤规则解析错误，过滤规则：{}，错误原因：{}", orderStrategyDto, resultBase.getErrorMsg());
                continue;
            }

            List<StrategyRuleDto> strategyRuleDtos =resultBase.getValue();
            OrderSnapshotDto snapshotDto = result.getOrderSnapshotDto();

            ResultBase<Boolean> checkResult = this.checkStrategyDetail(orderStrategyDto, strategyRuleDtos, snapshotDto);

            // 运行过滤规则中就算命中一条规则之后，剩余的规则也需要继续跑完
            if (checkResult.getValue() != null && checkResult.getValue()) {
                // 表示命中该规则
                flag = true;
                handleResult.add(orderStrategyDto.getHandlerName());
                // 命中结果保存在数据库
                filterRuleHitService.insert(snapshotDto.getOrderId(), orderStrategyDto.getStrategyName(), snapshotDto.getMobile());
            }

        }

        //  当有命中过滤规则时 根据策略的handlerName进行处理
        if (flag && CollectionUtils.isNotEmpty(handleResult)) {
            filterOrderTask.postFilterRuleHandle(result.getOrderSnapshotDto(), handleResult);
        }

        result.setCheckResult(flag);
        return result;
    }



    /**
     * 校验每一条策略
     * @param orderStrategyDto
     * @param strategyRuleDtos
     * @param orderSnapshotDto
     * @return
     */
    private ResultBase<Boolean> checkStrategyDetail(OrderStrategyDto orderStrategyDto, List<StrategyRuleDto> strategyRuleDtos,
                                                    OrderSnapshotDto orderSnapshotDto){

        Boolean flag = Boolean.TRUE;
        ResultBase<Boolean> checkResult = ResultBase.defaultReturn(false);


        // 同步校验规则时，不校验短信状态
        List<StrategyRuleDto> mobileStrategyList = strategyRuleDtos.stream()
                                                                    .filter(e -> ShotOrderFieldEnum.PHONE_STATUS.getVal().equals(e.getFieldName()))
                                                                    .collect(Collectors.toList());

        strategyRuleDtos.removeAll(mobileStrategyList);
        if (CollectionUtils.isEmpty(strategyRuleDtos)) {
            checkResult.setValue(false);
            return checkResult;
        }

        for (StrategyRuleDto strategyRuleDto : strategyRuleDtos){
            // 遍历每一条策略

            // 创建链路上下文对象
            ProcessContext processContext = this.createProcessContext(orderSnapshotDto, orderStrategyDto, strategyRuleDto);

            // 校验每一条策略 真正校验执行
            // 策略入口链路开始
            checkResult = baseHandler.handleRequest(processContext);

            if (checkResult.hasSuccessValue()){
                flag = flag && checkResult.getValue();
            } else {
                // 如果其中一条策略校验异常，本条规则不需要再校验
                flag = Boolean.FALSE;
                LOGGER.error("过滤规则校验失败 msg={} strategyName={} ruleName={} fieldName={}",
                        checkResult.getErrorMsg(),orderStrategyDto.getStrategyName(),strategyRuleDto.getRuleName(),strategyRuleDto.getFieldName());
                break;
            }
        }
        checkResult.setValue(flag);
        return checkResult;
    }




    /**
     *  创建链路上下文对象
     * @param orderSnapshotDto
     * @param orderStrategyDto
     * @param strategyRuleDto
     * @return
     */
    private ProcessContext createProcessContext(OrderSnapshotDto orderSnapshotDto, OrderStrategyDto orderStrategyDto, StrategyRuleDto strategyRuleDto) {
        // 适用部分商家
        if(StrategyUseType.PART.equals(StrategyUseType.findByValue(orderStrategyDto.getUseType()))) {
            strategyRuleDto.setPartMerchant(true);
        }else {
            strategyRuleDto.setPartMerchant(false);
        }
        return ProcessFaceory.createProcessContext(orderSnapshotDto,strategyRuleDto,orderStrategyDto);
    }


    /**
     * 订单是否匹配策略
     *
     * @param orderStrategyDto
     * @param orderSnapshotDto
     * @return
     */
    private boolean isOrderMatchStrategy(OrderStrategyDto orderStrategyDto, OrderSnapshotDto orderSnapshotDto) {

        StrategyUseType useType = StrategyUseType.findByValue(orderStrategyDto.getUseType());

        //适用部分商家的策略
        if(StrategyUseType.PART.equals(useType)){

            // 商家id
            Long merchantId = orderSnapshotDto.getMerchantId();

            // 策略id
            Long strategyId = orderStrategyDto.getId();

            //find
            StrategyMerchantEntity strategyMerchantEntity = strategyMerchantService.find(merchantId, strategyId);

            return strategyMerchantEntity != null;
        }

        return true;
    }

    public BaseHandler getBaseHandler() {
        return baseHandler;
    }

    public void setBaseHandler(BaseHandler baseHandler) {
        this.baseHandler = baseHandler;
    }
}
