package com.qiho.center.biz.job;

import java.io.InputStream;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import com.alibaba.dubbo.common.utils.ConcurrentHashSet;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.order.DeliveryRecordDto;
import com.qiho.center.api.enums.DeliveryRecordStateEnum;
import com.qiho.center.api.enums.DeliveryRecordTypeEnum;
import com.qiho.center.api.params.OrderFileDeliveryParam;
import com.qiho.center.api.util.BizLog;
import com.qiho.center.api.util.FileDeliveryUtil;
import com.qiho.center.biz.runnable.OrderFileCancelDeliveryRunnable;
import com.qiho.center.biz.runnable.OrderFileConfirmDeliveryRunnable;
import com.qiho.center.biz.service.OSSFileService;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.biz.service.order.OrderFileDeliveryService;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.entityd.qiho.order.DeliveryRecordEntity;

/**
 * 批量上传发物流单号-批量发货
 *
 * @Created by zhangshun on 2018-03-08 13:45:49
 */
@Component
public class OrderFileDeliveryJob extends AbstractQihoSimpleElasticJob implements ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrderFileDeliveryJob.class);

    /**
     * 核心线程20个，最大线程20个，队列200个
     */
    private static final ExecutorService EXECUTOR_ORDER_DELIVERY = new ThreadPoolExecutor(20, 30, 0L,
        TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(210));

    private final ConcurrentHashMap<String, Integer> COUNT_MAP_CONFIRM = new ConcurrentHashMap<>();

    private final ConcurrentHashSet<Integer> FAIL_ID_SET_CONFIRM = new ConcurrentHashSet<>();

    private volatile boolean PROCESSING_CONFIRM = false;

    private volatile boolean PROCESSING_CANCEL = false;

    private ApplicationContext context;

    @Autowired
    private OSSFileService ossFileService;

    @Resource
    private LogisticsService logisticsService;

    @Resource
    private OrderFileDeliveryService orderFileDeliveryService;

    private String failedCountKey = "FailedCount";

    private String succeedCountKey = "SucceedCount";

    private String updatedCountKey = "UpdatedCount";

    @Override
    protected void doProcess(JobExecutionMultipleShardingContext shardingContext) {
        LOGGER.info("start: 批量上传发物流单号批量发货");
        try {
            doStart();
        } catch (Exception e) {
            LOGGER.error("批量上传发物流单号批量发货错误", e);
        }
        LOGGER.info("end: 批量上传发物流单号批量发货");
    }

    /**
     * 归档任务处理，再执行一次消耗任务处理
     */
    public void doStart() {

        DeliveryRecordDto recordDto = orderFileDeliveryService.getJob();

        if (null == recordDto) {
            LOGGER.info("批量上传发物流单号，没有要执行的任务");
            return;
        }

        // 确认发货
        if (DeliveryRecordTypeEnum.CONFIRM.getCode().equals(recordDto.getRecordType())) {
            this.confirmDeliveryStart(recordDto);
        }

        // 取消发货
        else if (DeliveryRecordTypeEnum.CANCEL.getCode().equals(recordDto.getRecordType())) {
            this.cancelDeliveryStart(recordDto);
        }

    }

    public void confirmDeliveryStart(DeliveryRecordDto recordDto) {

        if (PROCESSING_CONFIRM) {
            LOGGER.warn("[确认发货]任务正在运行中。{}", recordDto);
            return;
        }

        // 设置为运行中...
        setConfirmRunning();

        InputStream input = null;

        //加载文件
        try {

            input = ossFileService.getOssFileInputStream(recordDto.getFileUrl());
            if (Objects.equal(null, input)) {
                BizLog.log("[确认发货]OSS文件不存在。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "OSS文件不存在");
                return;
            }

            List<OrderFileDeliveryParam> list = FileDeliveryUtil.getExcelContent(recordDto, input);

            if (CollectionUtils.isEmpty(list)) {
                BizLog.log("[确认发货]Excel文件内容错误。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "Excel文件内容错误。");
                return;
            }

            int totalCount = list.size();

            // 总条数
            if (10000 < totalCount) {
                BizLog.log("[确认发货]物流数量大于1万条。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "数量大于1万条。");
                return;
            }

            // 更新信息
            if (!orderFileDeliveryService.setStateRunning(recordDto.getId(), totalCount)) {
                LOGGER.error("[确认发货]更新任务为处理中失败。id:{}", recordDto.getId());
            }

            // 获取物流名称
            String logisticsName = this.getLogisticsNameByCode(recordDto.getLogisticsCode());

            // 分批提交
            List<List<OrderFileDeliveryParam>> parts = Lists.partition(list, 100);

            CountDownLatch countDownLatch = new CountDownLatch(parts.size());

            for (List<OrderFileDeliveryParam> data : parts) {
                this.addConfirmDeliveryJob(countDownLatch, recordDto, logisticsName, data);
            }

            countDownLatch.await();

            BizLog.log("[确认发货]所有工作线程都已完成, id:{}", recordDto.getId());

            DeliveryRecordEntity entity = new DeliveryRecordEntity();
            entity.setId(recordDto.getId());
            entity.setFailedCount(COUNT_MAP_CONFIRM.get(failedCountKey));
            entity.setSucceedCount(COUNT_MAP_CONFIRM.get(succeedCountKey));
            entity.setUpdatedCount(COUNT_MAP_CONFIRM.get(updatedCountKey));
            entity.setRemark(getConfirmFailMsg());

            // 任务完成
            entity.setState(DeliveryRecordStateEnum.COMPLETE.getCode());

            // 更新数据库
            if (!orderFileDeliveryService.updateByIdSelective(entity)) {
                LOGGER.error("[确认发货]批量上传发物流单号-更新数据失败, id:{}", entity.getId());
            }

        } catch (Exception e) {
            LOGGER.error("[确认发货]OSS文件Input错误。{}", recordDto, e);
            orderFileDeliveryService.setStateError(recordDto.getId(), "导入文件错误");
        } finally {
            // 设置任务状态
            resetConfirmState();

            // 继续执行任务
            doStart();

            if (null != input) {
                ossFileService.closeInputStream(input);
            }
        }
    }

    public void cancelDeliveryStart(DeliveryRecordDto recordDto) {

        if (PROCESSING_CANCEL) {
            LOGGER.warn("[取消发货]任务正在运行中。{}", recordDto);
            return;
        }

        // 设置为运行中...
        setCancelRunning();

        InputStream input = null;

        //加载文件
        try {

            input = ossFileService.getOssFileInputStream(recordDto.getFileUrl());
            if (Objects.equal(null, input)) {
                BizLog.log("[取消发货]OSS文件不存在。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "OSS文件不存在");
                return;
            }

            List<OrderFileDeliveryParam> list = FileDeliveryUtil.getExcelContent(recordDto, input);

            if (CollectionUtils.isEmpty(list)) {
                BizLog.log("[取消发货]Excel文件内容错误。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "Excel文件内容错误");
                return;
            }

            int totalCount = list.size();

            // 总条数
            if (10000 < totalCount) {
                BizLog.log("[取消发货]物流数量大于1万条。ossUrl:{}", recordDto.getFileUrl());
                orderFileDeliveryService.setStateError(recordDto.getId(), "数量大于1万条。");
                return;
            }

            // 更新信息
            if (!orderFileDeliveryService.setStateRunning(recordDto.getId(), totalCount)) {
                LOGGER.error("[取消发货]更新任务为处理中失败。id:{}", recordDto.getId());
            }

            // 提交任务
            this.addCancelDeliveryJob(recordDto, list);

        } catch (Exception e) {
            LOGGER.error("[取消发货]OSS文件Input错误。{}", recordDto, e);
            orderFileDeliveryService.setStateError(recordDto.getId(), "导入文件错误");
        } finally {
            // 设置任务状态
            resetCancelState();
            // 继续执行任务
            doStart();
            if (null != input) {
                ossFileService.closeInputStream(input);
            }
        }
    }

    /**
     * 添加任务
     *
     * @param countDownLatch
     * @param recordDto      任务ID
     * @param logisticsName  物流名称
     * @param data           数据
     */
    private void addConfirmDeliveryJob(CountDownLatch countDownLatch, DeliveryRecordDto recordDto, String logisticsName,
        List<OrderFileDeliveryParam> data) {
        // 获取工作线程
        OrderFileConfirmDeliveryRunnable runnable = context
            .getBean(OrderFileConfirmDeliveryRunnable.class, this, recordDto, logisticsName, countDownLatch, data);
        // 提交工作线程
        EXECUTOR_ORDER_DELIVERY.submit(runnable);
    }

    /**
     * 添加任务
     *
     * @param recordDto 任务ID
     * @param data      数据
     */

    private void addCancelDeliveryJob(DeliveryRecordDto recordDto, List<OrderFileDeliveryParam> data) {
        // 获取工作线程
        OrderFileCancelDeliveryRunnable runnable = context
            .getBean(OrderFileCancelDeliveryRunnable.class, recordDto, data);
        // 提交工作线程
        EXECUTOR_ORDER_DELIVERY.submit(runnable);
    }

    /**
     * @param value
     */
    public void setConfirmFailedCount(int value) {
        if (COUNT_MAP_CONFIRM.containsKey(failedCountKey)) {
            COUNT_MAP_CONFIRM.put(failedCountKey, value + COUNT_MAP_CONFIRM.get(failedCountKey));
        } else {
            COUNT_MAP_CONFIRM.put(failedCountKey, value);
        }
    }

    public void setConfirmSucceedCount(int value) {
        if (COUNT_MAP_CONFIRM.containsKey(succeedCountKey)) {
            COUNT_MAP_CONFIRM.put(succeedCountKey, value + COUNT_MAP_CONFIRM.get(succeedCountKey));
        } else {
            COUNT_MAP_CONFIRM.put(succeedCountKey, value);
        }
    }

    public void setConfirmUpdatedCount(int value) {
        if (COUNT_MAP_CONFIRM.containsKey(updatedCountKey)) {
            COUNT_MAP_CONFIRM.put(updatedCountKey, value + COUNT_MAP_CONFIRM.get(updatedCountKey));
        } else {
            COUNT_MAP_CONFIRM.put(updatedCountKey, value);
        }
    }

    public void setConfirmFailId(Integer rowNumber) {
        FAIL_ID_SET_CONFIRM.add(rowNumber);
    }

    private String getConfirmFailMsg() {
        String failMsg = StringUtils.join(FAIL_ID_SET_CONFIRM.iterator(), ",");
        if (2040 < failMsg.length()) {
            failMsg = failMsg.substring(0, 2040) + "......";
        }
        return failMsg;
    }

    private void setConfirmRunning() {
        // 设置运行中状态
        PROCESSING_CONFIRM = true;
    }

    private void resetConfirmState() {
        // 重置运行状态
        PROCESSING_CONFIRM = false;

        COUNT_MAP_CONFIRM.put(failedCountKey, 0);
        COUNT_MAP_CONFIRM.put(succeedCountKey, 0);
        COUNT_MAP_CONFIRM.put(updatedCountKey, 0);
        FAIL_ID_SET_CONFIRM.clear();
    }

    private void setCancelRunning() {
        // 设置运行中状态
        PROCESSING_CANCEL = true;
    }

    private void resetCancelState() {
        // 重置运行状态
        PROCESSING_CANCEL = false;
    }

    /**
     * 根据物流编码获取物流名称
     *
     * @param code
     * @return
     */
    private String getLogisticsNameByCode(String code) {
        BaiqiLogisticsEntity entity = logisticsService.findByCode(code);
        if (null == entity) {
            return StringUtils.EMPTY;
        }
        return entity.getLogisticsName();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }
}
