package com.qiho.center.biz.job;

import cn.com.duiba.tuia.core.api.dto.rsp.baiqi.BaiQiAdvertDetailDto;
import cn.com.duiba.tuia.core.api.remoteservice.baiqi.RemoteBaiQiService;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.enums.finance.*;
import com.qiho.center.api.enums.merchant.BaiqiMerchantLinkTypeEnum;
import com.qiho.center.biz.bo.MerchantAdvertBO;
import com.qiho.center.biz.service.advert.BaiqiTuiaAdvertConsumeService;
import com.qiho.center.biz.service.advert.BaiqiTuiaPlanService;
import com.qiho.center.biz.service.finance.FinanceService;
import com.qiho.center.biz.service.impl.finance.bean.CashRebateAmountBean;
import com.qiho.center.biz.service.merchant.BaiqiMerchantLinkService;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceDetailMapper;
import com.qiho.center.common.entityd.qiho.advert.BaiqiTuiaAdvertConsumeEntity;
import com.qiho.center.common.entityd.qiho.advert.BaiqiTuiaPlanEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceDetailEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceEntity;
import com.qiho.center.common.entityd.qiho.merchant.BaiqiMerchantLinkEntity;
import com.qiho.center.common.util.BaiqiSerialUtil;
import org.apache.commons.collections.CollectionUtils;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 推啊广告消耗拉取任务
 *
 * @author peanut.huang
 * @date 2017/12/27.
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class TuiaAdvertConsumeJob extends AbstractQihoSimpleElasticJob {

    private static final Logger LOGGER = LoggerFactory.getLogger(TuiaAdvertConsumeJob.class);

    @Resource
    private MerchantAdvertBO merchantAdvertBO;
    @Resource
    private BaiqiTuiaAdvertConsumeService tuiaAdvertConsumeService;
    @Resource
    private BaiqiMerchantLinkService merchantLinkService;
    @Resource
    private BaiqiFinanceDetailMapper financeDetailMapper;
    @Resource
    private FinanceService financeService;
    @Resource
    private RemoteBaiQiService remoteBaiQiService;
    @Resource
    private BaiqiTuiaPlanService tuiaPlanService;

    /**
     * 默认不是归档
     */
    private Boolean isArchive = false;

    public void setArchive(Boolean archive) {
        isArchive = archive;
    }

    @Override
    protected void doProcess(JobExecutionMultipleShardingContext shardingContext) {

        LOGGER.info("fetch tuia advert consume start...");

        doStart(LocalDate.now(), LocalDate.now());

        LOGGER.info("fetch tuia advert consume end...");

    }

    /**
     *
     * 此任务将从推啊中拉取百奇商家在推啊中的广告消耗明细，对于明细有以下处理
     *
     *
     * a、百奇将以天为单位存储明细数据（后续可做报表），根据明细保存广告主信息及广告计划信息
     *
     * b、广告的消耗将影响百奇商家的账户金额（商家与推啊广告主做关联，商家账户做扣款动作）
     *
     */
    public void doStart(LocalDate start, LocalDate end) {

        //1、 获取消耗数据
        List<BaiqiTuiaAdvertConsumeEntity> consumeList = fetchConsumeList(start, end);
        if(CollectionUtils.isEmpty(consumeList)){
            return;
        }

        //2、消耗数据入百奇库(insert / update)：保存数据
        saveConsumeList(consumeList);

        //3、消耗数据影响百奇商家账户: 扣款
        affectMerchantFinance(consumeList);
    }

    /**
     * 消耗数据影响百奇商家账户: 扣款
     *
     * @param consumeList
     */
    private void affectMerchantFinance(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        //1、消数据依据广告主分组
        Map<Long, List<BaiqiTuiaAdvertConsumeEntity>> advertConsumeListMap = consumeList.stream().collect(Collectors.groupingBy(BaiqiTuiaAdvertConsumeEntity::getAdvertId));

        //2、 广告主逐个处理
        advertConsumeListMap.forEach(this::executeAdvertConsume);
    }

    /**
     * 执行广告消耗
     *
     * @param advertId
     * @param eachConsumeList
     */
    private void executeAdvertConsume(Long advertId, List<BaiqiTuiaAdvertConsumeEntity> eachConsumeList){

        // 广告主是否与商家关联，未关联则跳过
        Long merchantId = findMerchantId(advertId);
        if(merchantId == null){
            return;
        }

        //商家账户id
        Long financeId = findFinanceId(merchantId);
        if(financeId == null){
            return;
        }

        //当前广告主各广告计划消耗总和
        Long totalConsume = eachConsumeList.stream().mapToLong(BaiqiTuiaAdvertConsumeEntity::getConsumeTotal).sum();


        // 1、流水号拿到流水
        Date date = LocalDate.now().toDate();
        if(isArchive){
            //归档，取昨天
            date = LocalDate.now().minusDays(1).toDate();
        }

        String serialNo = BaiqiSerialUtil.getAdCostSerialNo(FinanceOptTypeEnum.AD_COST, FinanceTypeEnum.MAIN_ACCOUNT, merchantId, date);
        BaiqiFinanceDetailEntity detailEntity = financeDetailMapper.selectBySerialNo(serialNo);

        //2、处理流水,返回扣款金额对象
        Map<Long ,CashRebateAmountBean> mapBean = Maps.newHashMap();
        BaiqiFinanceDetailEntity resultEntity = processDetail(advertId, merchantId, financeId, totalConsume, detailEntity, mapBean);
        resultEntity.setSerialNo(serialNo);

        //3、保存流水 + 商家账户扣款
        merchantAdvertBO.saveDetailAndReduceAmount(financeId, resultEntity, mapBean.get(merchantId));

    }

    /**
     * 处理流水实体
     *
     * @param advertId      广告主id
     * @param merchantId    商家id
     * @param financeId     商家账户id
     * @param totalConsume  总消耗
     * @param detailEntity  流水实体
     * @return CashRebateAmountBean  扣款金额对象
     */
    private BaiqiFinanceDetailEntity processDetail(Long advertId, Long merchantId, Long financeId, Long totalConsume, BaiqiFinanceDetailEntity detailEntity, Map<Long ,CashRebateAmountBean> mapBean) {
        BaiqiFinanceDetailEntity detailEntityTemp =  detailEntity;
        //有流水实体
        if(detailEntityTemp != null){

            //处理流水实体不为null
            handleFinanceDetailNotNull(advertId, merchantId, financeId, totalConsume, detailEntityTemp, mapBean);

        //无流水
        }else{
            //处理无流水
            detailEntityTemp = handleNotFinanceDetail(merchantId, financeId, totalConsume, mapBean);
        }

        //定时任务：管理员
        detailEntityTemp.setOperator(0L);
        detailEntityTemp.setOperatorType(FinanceOptTypeEnum.AD_COST.getNum());
        detailEntityTemp.setState(isArchive ? FinanceDetailStatusEnum.SUCCESS.getNum() : FinanceDetailStatusEnum.PROCESSING.getNum());
        detailEntityTemp.setPartnerType(ParterTypeEnum.TUIA.getCode());

        return detailEntityTemp;
    }

    /**
     * 处理无流水
     *
     * @param merchantId
     * @param financeId
     * @param totalConsume
     * @param mapBean
     * @return
     */
    private BaiqiFinanceDetailEntity handleNotFinanceDetail(Long merchantId, Long financeId, Long totalConsume, Map<Long, CashRebateAmountBean> mapBean) {
        BaiqiFinanceDetailEntity detailEntity = new BaiqiFinanceDetailEntity();

        detailEntity.setFinanceId(financeId);
        detailEntity.setRelationType(FinanceTypeEnum.MAIN_ACCOUNT.getCode());
        detailEntity.setRelationId(merchantId);

        detailEntity.setExpenditureTotal(totalConsume);

        //计算现金与返点金额 totalConsume * 基数比
        CashRebateAmountBean amountBean = financeService.calcCashAndRebateAmount(financeId, totalConsume, CashRebateAmountEnum.EXPENDITURE);
        if(amountBean != null){

            //总支出
            detailEntity.setExpenditureTotal(totalConsume);

            //现金支出
            detailEntity.setExpenditureCash(amountBean.getCashAmount());

            //返点支出
            detailEntity.setExpenditureRebate(amountBean.getRebateAmount());

            mapBean.put(merchantId, amountBean);

        }

        return detailEntity;
    }

    /**
     * 处理流水实体不为null
     *
     * @param advertId
     * @param merchantId
     * @param financeId
     * @param totalConsume
     * @param detailEntity
     * @param mapBean
     */
    private void handleFinanceDetailNotNull(Long advertId, Long merchantId, Long financeId, Long totalConsume, BaiqiFinanceDetailEntity detailEntity, Map<Long, CashRebateAmountBean> mapBean) {

        //上一次流水总消耗
        Long lastTotalConsume =  detailEntity.getExpenditureTotal();


        //本次消耗与上次消耗差
        Long rest = totalConsume - lastTotalConsume;

        //消耗未发生变化
        if(rest == 0L){
            return;
        }

        if(rest < 0 && !isArchive){
            LOGGER.error("本次总消耗小于上次总消耗，广告主[{}], 商家[{}]", advertId, merchantId);
            return;
        }

        CashRebateAmountEnum amountEnum;
        if(rest > 0){
            amountEnum = CashRebateAmountEnum.EXPENDITURE;
        }else{
            amountEnum = CashRebateAmountEnum.INCOME;
        }

        //消耗较上一次增加，商家账户余额扣款


        //计算现金与返点金额 rest * 基数比
        CashRebateAmountBean amountBean = financeService.calcCashAndRebateAmount(financeId, rest, amountEnum);
        if(amountBean != null){

            Long originTotal = detailEntity.getExpenditureTotal();

            //多扣，补款流水
            if(CashRebateAmountEnum.INCOME.equals(amountEnum)){

                //总支出 - 多扣
                detailEntity.setExpenditureTotal(originTotal - amountBean.getTotalAmount());
                //现金支出 - 多扣
                detailEntity.setExpenditureCash(detailEntity.getExpenditureCash() - amountBean.getCashAmount());
                //返点支出 - 多扣
                detailEntity.setExpenditureRebate(detailEntity.getExpenditureRebate() - amountBean.getRebateAmount());

                //异常处理
                handleException(detailEntity);
            }

            //正常扣款
            if(CashRebateAmountEnum.EXPENDITURE.equals(amountEnum)){

                //总支出 +
                detailEntity.setExpenditureTotal(originTotal + amountBean.getTotalAmount());
                //总现金支出 +
                detailEntity.setExpenditureCash(detailEntity.getExpenditureCash() + amountBean.getCashAmount());
                //总返点支出 +
                detailEntity.setExpenditureRebate(detailEntity.getExpenditureRebate() + amountBean.getRebateAmount());
            }

            //设置归档备注
            if(isArchive){

                StringBuilder remarkStr = new StringBuilder("预计扣款: ");

                remarkStr.append(new BigDecimal(originTotal).movePointLeft(2))
                         .append(" 元, 归档扣款: ")
                         .append(new BigDecimal(detailEntity.getExpenditureTotal()).movePointLeft(2))
                         .append(" 元");

                detailEntity.setRemark(remarkStr.toString());

            }

            mapBean.put(merchantId, amountBean);
        }
    }

    /**
     * 当多扣，给流水补款时，现金或返点不足于补款额时的异常情况处理：
     *
     * @param detailEntity
     */
    private void handleException(BaiqiFinanceDetailEntity detailEntity) {

        Long cash = detailEntity.getExpenditureCash();
        Long rebate = detailEntity.getExpenditureRebate();

        //现金不足，补返点
        if(cash < 0){

            detailEntity.setExpenditureCash(0L);

            detailEntity.setExpenditureRebate(rebate - Math.abs(cash));
        }

        //返点不足补现金
        if(rebate < 0){

            detailEntity.setExpenditureRebate(0L);

            detailEntity.setExpenditureCash(cash - Math.abs(rebate));
        }
    }

    /**
     * 根据商家id 获取账户id
     *
     * @param merchantId
     * @return
     */
    private Long findFinanceId(Long merchantId) {

        BaiqiFinanceEntity financeEntity = financeService.findByRelation(FinanceTypeEnum.MAIN_ACCOUNT, merchantId);

        return financeEntity != null ? financeEntity.getId() : null;
    }

    /**
     * 广告主id获取商家id
     *
     * @param advertId   广告主id
     * @return           关联后的商家id
     */
    private Long findMerchantId(Long advertId) {

        BaiqiMerchantLinkEntity linkEntity = merchantLinkService.findByRelation(BaiqiMerchantLinkTypeEnum.TUI_A.getType(), advertId);

        return  linkEntity != null ? linkEntity.getMerchantId() : null;
    }

    /**
     * 从推啊获取数据
     *
     * @return
     */
    private List<BaiqiTuiaAdvertConsumeEntity> fetchConsumeList(LocalDate start, LocalDate end) {
        String startDate;
        if(start == null){
            startDate = LocalDate.now().toString();
        }else {
            startDate = start.toString();
        }

        String endDate;
        if(end == null){
            endDate = LocalDate.now().toString();
        }else {
            endDate = end.toString();
        }

        //fetch data from tuia
        List<BaiQiAdvertDetailDto> detailDtoList;
        try {
            detailDtoList = remoteBaiQiService.findBaiQiDetail(startDate, endDate);
        }catch (Exception e){
            LOGGER.error("find tuia advert detail failed", e);
            return Collections.emptyList();
        }

        List<BaiqiTuiaAdvertConsumeEntity> result = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(detailDtoList)){

            detailDtoList.forEach(e -> dto2Entity(e, result));
        }

        return result;
    }

    /**
     * dto转换为entity
     *
     * @param e
     * @param result
     */
    private void dto2Entity(BaiQiAdvertDetailDto e, List<BaiqiTuiaAdvertConsumeEntity> result) {

        BaiqiTuiaAdvertConsumeEntity entity = BeanUtils.copy(e, BaiqiTuiaAdvertConsumeEntity.class);

        entity.setState(e.getStatus());
        entity.setBelongDate(isArchive ? LocalDate.now().toDate() : LocalDate.now().minusDays(1).toDate());
        entity.setBudgetPerday(e.getBudgetPerDay() == null ? 0 : e.getBudgetPerDay());
        entity.setConsumeTotal(e.getConsumeTotal() == null ? 0 : e.getConsumeTotal());
        entity.setExposureCount(e.getExposureCount() == null ? 0 : e.getExposureCount());
        entity.setClickCount(e.getClickCount() == null ? 0 : e.getClickCount());

        result.add(entity);
    }


    /**
     * 保存消耗数据
     *
     * a、保存广告主数据
     *
     * b、保存消耗明细
     *
     * @param consumeList   消耗数据
     */
    private void saveConsumeList(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        //1、保存广告主
        saveAdvert(consumeList);

        //2、保存广告计划
        savePlan(consumeList);

        //3、保存消耗 insert or update
        saveConsume(consumeList);
    }

    /**
     * 保存广告计划
     *
     * @param consumeList
     */
    private void savePlan(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        //trans to plan list
        List<BaiqiTuiaPlanEntity> insertList = transferPlan(consumeList);

        //batch insert list
        tuiaPlanService.batchInsert(insertList);
    }

    /**
     * trans to plan list
     *
     * @param consumeList
     * @return
     */
    private List<BaiqiTuiaPlanEntity> transferPlan(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        return BeanUtils.copyList(consumeList, BaiqiTuiaPlanEntity.class);
    }


    /**
     * 保存广告主
     *
     * @param consumeList
     */
    private void saveAdvert(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        //广告主id与名称map
        Map<Long, String> advertIdAndNameMap = consumeList.stream().collect(Collectors.toMap(
                                                                                             // map's key
                                                                                            BaiqiTuiaAdvertConsumeEntity::getAdvertId,
                                                                                            // map's value
                                                                                            BaiqiTuiaAdvertConsumeEntity::getAdvertName,
                                                                                            // if key is exist, use the new one
                                                                                            (oldValue, newValue) -> newValue,
                                                                                            // map's type
                                                                                            HashMap::new
                                                                                            )
                                                                           );

        List<BaiqiMerchantLinkEntity> saveEntityList = Lists.newArrayListWithExpectedSize(advertIdAndNameMap.size());

        //构建实体list
        advertIdAndNameMap.forEach( (advertId, advertName) -> {

            BaiqiMerchantLinkEntity linkEntity = new BaiqiMerchantLinkEntity();
            linkEntity.setRelationId(advertId);
            linkEntity.setRelationName(advertName);
            linkEntity.setRelationType(BaiqiMerchantLinkTypeEnum.TUI_A.getType());

            saveEntityList.add(linkEntity);
        });

        //batch insert
        if(CollectionUtils.isNotEmpty(saveEntityList)){
            merchantLinkService.batchInsertTuiaLink(saveEntityList);
        }
    }


    /**
     * 保存消耗
     *
     * @param consumeList
     */
    private void saveConsume(List<BaiqiTuiaAdvertConsumeEntity> consumeList) {

        Date today = isArchive ? LocalDate.now().minusDays(1).toDate() : LocalDate.now().toDate();

        consumeList.forEach(e -> {
            e.setBelongDate(today);

            tuiaAdvertConsumeService.insert(e);
        });
    }
}
