package com.qiho.center.biz.service.advert.impl;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.merchant.MerchantAdvertDto;
import com.qiho.center.api.dto.merchant.MerchantSimpleDto;
import com.qiho.center.api.enums.merchant.BaiqiMerchantLinkTypeEnum;
import com.qiho.center.biz.service.advert.BaiqiTuiaAdvertConsumeService;
import com.qiho.center.biz.service.merchant.BaiqiMerchantLinkService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.daoh.qiho.advert.BaiqiTuiaAdvertConsumeMapper;
import com.qiho.center.common.entityd.qiho.advert.BaiqiTuiaAdvertConsumeEntity;
import com.qiho.center.common.entityd.qiho.merchant.BaiqiMerchantLinkEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.map.HashedMap;
import org.joda.time.LocalDate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 推啊广告消耗service
 *
 * @author peanut.huang
 * @date 2017/12/27.
 */
@Service("baiqiTuiaAdvertConsumeService")
public class BaiqiTuiaAdvertConsumeServiceImpl implements BaiqiTuiaAdvertConsumeService {

    @Resource
    private BaiqiTuiaAdvertConsumeMapper baiqiTuiaAdvertConsumeMapper;
    @Resource
    private BaiqiMerchantLinkService baiqiMerchantLinkService;
    @Resource
    private MerchantService merchantService;

    @Override
    public void insert(BaiqiTuiaAdvertConsumeEntity consumeEntity) {

        baiqiTuiaAdvertConsumeMapper.insert(consumeEntity);
    }

    @Override
    public List<MerchantAdvertDto> findAdvertConsume(Long merchantId, Date startDate, Date endDate) {

        Date start = startDate;
        Date end = endDate;

        //日期为空，默认今天
        if (start == null) {
            start = LocalDate.now().toDate();
        }
        if (end == null) {
            end = LocalDate.now().toDate();
        }

        /**
         * 商家 ---> 广告主 ---> 广告计划
         *
         * 广告计划总消耗统计
         *
         */


        //查询商家
        List<MerchantAdvertDto> dtoList = fetchMerchant(merchantId);
        if(CollectionUtils.isEmpty(dtoList)){
            return Collections.emptyList();
        }

        //与广告主相关联
        List<BaiqiMerchantLinkEntity> linkEntityList = baiqiMerchantLinkService.findByTypeWithLinked(BaiqiMerchantLinkTypeEnum.TUI_A);
        if(CollectionUtils.isEmpty(linkEntityList)){
            return dtoList;
        }

        //合并广告主
        mergeAdvert(dtoList, linkEntityList);

        //广告id集
        List<Long> advertIdList = linkEntityList.stream().map(BaiqiMerchantLinkEntity::getRelationId).collect(Collectors.toList());

        //广告消耗数据统计合并
        mergeAdvertConsume(dtoList, advertIdList, start, end);

        //sort
        return sortDtoList(dtoList);
    }

    private List<MerchantAdvertDto> sortDtoList(List<MerchantAdvertDto> dtoList) {


        List<MerchantAdvertDto> frontList = dtoList.stream().filter(e -> e.getAdvertConsume() != null).collect(Collectors.toList());

        List<MerchantAdvertDto> backList = dtoList.stream().filter(e -> e.getAdvertConsume() == null).collect(Collectors.toList());

        frontList.addAll(backList);
        return frontList;
    }


    private void mergeAdvert(List<MerchantAdvertDto> dtoList, List<BaiqiMerchantLinkEntity> linkEntityList) {

        Map<Long, BaiqiMerchantLinkEntity> merchantIdEntityMap = linkEntityList.stream()
                                                                               .collect(Collectors.toMap(
                                                                                                        BaiqiMerchantLinkEntity::getMerchantId,
                                                                                                        Function.identity(),
                                                                                                       (n,o) -> o,
                                                                                                       HashedMap::new
                                                                                                        )
                                                                                        );

        dtoList.forEach(e -> {

            BaiqiMerchantLinkEntity linkEntity = merchantIdEntityMap.get(e.getMerchantId());

            if( linkEntity != null){
                e.setAdvertId(linkEntity.getRelationId());
                e.setAdvertName(linkEntity.getRelationName());
            }
        });
    }

    private List<MerchantAdvertDto> fetchMerchant(Long merchantId) {

        //1、查询所有商家
        List<MerchantSimpleDto> allMerchantList;
        if(merchantId != null){
            allMerchantList = merchantService.findIdAndNamesByIds(Lists.newArrayList(merchantId));
        }else {
            allMerchantList = merchantService.findAllIdAndNames();
        }

        if (CollectionUtils.isEmpty(allMerchantList)) {
            return Collections.emptyList();
        }


        List<MerchantAdvertDto> result = Lists.newArrayListWithExpectedSize(allMerchantList.size());

        allMerchantList.forEach( e ->{
            MerchantAdvertDto dto = new MerchantAdvertDto();
            dto.setMerchantName(e.getMerchantName());
            dto.setMerchantId(e.getId());

            result.add(dto);
        });


        return result;
    }

    /**
     * 合并消耗数据
     *
     * @param result
     * @param advertIdList
     * @param startDate
     * @param endDate
     */
    private void mergeAdvertConsume(List<MerchantAdvertDto> result, List<Long> advertIdList, Date startDate, Date endDate) {

        //根据条件查询广告消耗
        List<BaiqiTuiaAdvertConsumeEntity> advertConsumeList = baiqiTuiaAdvertConsumeMapper.findAdvertConsume(advertIdList, startDate, endDate);
        if(CollectionUtils.isEmpty(advertConsumeList)){
            return;
        }

        //广告主与消耗数据map
        Map<Long, List<BaiqiTuiaAdvertConsumeEntity>> advertIdMap = advertConsumeList.stream()
                                                                                     .collect(Collectors.groupingBy(BaiqiTuiaAdvertConsumeEntity::getAdvertId));

        //广告主为维度设置总消耗量
        result.forEach(e -> {

            List<BaiqiTuiaAdvertConsumeEntity> consumeList = advertIdMap.get(e.getAdvertId());

            if (CollectionUtils.isNotEmpty(consumeList)) {
                //总消耗量
                Long sum = consumeList.stream().mapToLong(BaiqiTuiaAdvertConsumeEntity::getConsumeTotal).sum();

                e.setAdvertConsume(sum);
            }
        });
    }
}
