/**
 * Project Name:qiho-center-biz File Name:ItemServiceImpl.java Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月18日上午10:23:17 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.javatuples.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.dto.ItemAdvertDto;
import com.qiho.center.api.dto.ItemDetailDto;
import com.qiho.center.api.dto.ItemDto;
import com.qiho.center.api.dto.ItemSimpleDto;
import com.qiho.center.api.dto.ItemSkuDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.SkuPropertyDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.item.ItemPageParams;
import com.qiho.center.api.util.StringRandUtil;
import com.qiho.center.biz.event.ItemDeleteEvent;
import com.qiho.center.biz.service.ItemDetailRecordService;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.ItemSkuService;
import com.qiho.center.biz.service.SkuPropertyService;
import com.qiho.center.biz.service.item.ItemEvaluateService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.biz.service.tag.ItemTagService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.dao.QihoItemExtDAO;
import com.qiho.center.common.dao.QihoItemMerchantDAO;
import com.qiho.center.common.dao.QihoItemRecommendDAO;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entity.item.QihoItemExtEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.entity.merchant.QihoItemMerchantEntity;
import com.qiho.center.common.entityd.qiho.QihoItemDetailRecordEntity;
import com.qiho.center.common.support.BizEventBus;

import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * ClassName:ItemServiceImpl <br/>
 * 商品服务service <br/>
 * Date: 2017年5月18日 上午10:23:17 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Service
public class ItemServiceImpl implements ItemService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ItemServiceImpl.class);

    @Autowired
    private QihoItemDAO qihoItemDAO;

    @Autowired
    private QihoItemSkuDAO qihoItemSkuDAO;

    @Autowired
    private QihoItemExtDAO qihoItemExtDAO;

    @Autowired
    private QihoItemRecommendDAO qihoItemRecommendDAO;

    @Autowired
    private SkuPropertyService skuPropertyService;

    @Autowired
    private ItemSkuService itemSkuService;

    @Autowired
    private RemoteStockService remoteStockService;

    @Autowired
    private ItemDetailRecordService itemDetailRecordService;

    @Autowired
    private BizEventBus eventBus;

    @Autowired
    private QihoItemMerchantDAO qihoItemMerchantDAO;

    @Autowired
    private ItemTagService itemTagService;


    @Autowired
    private MerchantService merchantService;

    @Autowired
    private QihoItemExtDAO itemExtDAO;

    @Autowired
    private ItemEvaluateService itemEvaluateService;

    @Override
    public PagenationDto<ItemDto> queryItemPage(ItemDto queryItem, Integer pageNum, Integer pageSize) {
        QihoItemEntity param = BeanUtils.copy(queryItem, QihoItemEntity.class);
        Long merchantId = queryItem.getMerchantId();

        PagenationDto<ItemDto> pagenation = new PagenationDto<>();
        if (null == merchantId || merchantId <= 0) {
            List<QihoItemDetailEntity> list = qihoItemDAO.queryByParam(param, pageNum, pageSize);
            int total = qihoItemDAO.countByParam(param);
            pagenation.setList(convertToItemDto(list));
            pagenation.setTotal(total);
        } else {
            pagenation = queryItemPageByMerchant(param, pageNum, pageSize);
        }
        return pagenation;
    }

    public PagenationDto<ItemDto> queryItemPageByMerchant(QihoItemEntity queryItem,
                                                              Integer pageNum, Integer pageSize) {
        PagenationDto<ItemDto> pagenation = new PagenationDto<>();

        // 查询当前商家下所有的商品
        List<QihoItemMerchantEntity> itemMerchantEntityList = qihoItemMerchantDAO
                .findByMerchantId(queryItem.getMerchantId());
        if (CollectionUtils.isEmpty(itemMerchantEntityList)) {
            LOGGER.info("当前商家下没有查询到商品, queryItem={}", queryItem);
            return pagenation;
        }

        // 提取商品id
        List<Long> itemIdList = itemMerchantEntityList.stream().map(QihoItemMerchantEntity::getItemId)
                .collect(Collectors.toList());

        // 条件查询中有商品id
        Long itemId = queryItem.getId();
        if (null != itemId) {
            // 若该商品id属于当前商家, 则仅查询当前商品
            if (itemIdList.contains(itemId)) {
                itemIdList.clear();
                itemIdList.add(itemId);
            } else {
                LOGGER.info("当前商家下没有查询到该商品, queryItem={}", queryItem);
                return pagenation;
            }
        }

        // 在商品表中, 根据商品id集合和查询条件查询对应的商品集合
        List<QihoItemDetailEntity> detailEntityList = qihoItemDAO
                .queryByParamAndItemIds(queryItem, itemIdList, pageNum, pageSize);
        if (CollectionUtils.isEmpty(detailEntityList)) {
            LOGGER.info("根据商品id集合和指定条件没有查询到商品, itemIdList={}, queryItem={}", itemIdList, queryItem);
            return pagenation;
        }

        // 在商品扩展信息表中, 根据商品id集合和查询条件查询对应的商品集合
        List<QihoItemExtEntity> extEntityList = itemExtDAO.queryAllItemExt(itemIdList);
        if (CollectionUtils.isNotEmpty(extEntityList)) {
            Map<Long, QihoItemExtEntity> map = new HashMap<>();
            for (QihoItemExtEntity entity : extEntityList) {
                if (null != entity) {
                    map.put(entity.getItemId(), entity);
                }
            }

            // 根据itemId将两次查询到的数据合并为一个对象
            detailEntityList.forEach(itemDetailEntity -> {
                QihoItemExtEntity itemExtEntity = map.get(itemDetailEntity.getId());
                if (null != itemExtEntity) {
                    itemDetailEntity.setExtParam(itemExtEntity.getExtParam());
                    itemDetailEntity.setDetail(itemExtEntity.getDetail());
                }
            });
        }

        // 查询条数
        int count = qihoItemDAO.countByParamAndItemIds(queryItem, itemIdList);

        pagenation.setList(convertToItemDto(detailEntityList));
        pagenation.setTotal(count);

        return pagenation;
    }

    /**
     * @param list
     * @return
     * @author zhanglihui
     */
    private List<ItemDto> convertToItemDto(List<QihoItemDetailEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        List<Long> idList = Lists.newArrayList();
        for (QihoItemDetailEntity entity : list) {
            idList.add(entity.getId());
        }
        List<QihoItemSkuEntity> skuList = qihoItemSkuDAO.getSkuByItemIds(idList);

        return setItemDtoInfo(list, skuList);
    }

    private List<ItemDto> setItemDtoInfo(List<QihoItemDetailEntity> list, List<QihoItemSkuEntity> skuList) {
        List<ItemDto> itemList = Lists.newArrayList();

        List<Long> stockIds = Lists.newArrayList();
        for (QihoItemSkuEntity skuEntity : skuList) {
            stockIds.add(skuEntity.getStockId());
        }
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
        Pair<Map, Map> stockPair = getSkuStockInfo(stockMap, skuList);
        // itemId : 库存 键值对
        Map<Long, Long> itemStockMap = stockPair.getValue0();
        //有效库存
        Map<Long, Long> itemEnableStockMap = stockPair.getValue1();

        //价格
        Pair<Map, Map> pricePair = getSkuPriceInfo(skuList);
        // itemId: 最小价格
        Map<Long, Integer> minPriceMap = pricePair.getValue0();
        // itemId: 最小原价
        Map<Long, Integer> minOriginalPriceMap = pricePair.getValue1();

        // 最小价格sku
        Pair<Map, Map> minPriceSkuPair = getMinPriceSkuInfo(skuList);
        // itemId：最小价格时的sku的原价
        Map<Long, Integer> minSkuOriginalPriceMap = minPriceSkuPair.getValue0();
        // itemId：最小价格时的sku的差价
        Map<Long, Integer> minSkuPriceDiffMap = minPriceSkuPair.getValue1();

        // 最大差价sku
        Pair<Map, Map> maxPriceDiffPair = getMaxPriceDiffSkuInfo(skuList);
        // itemId：最大差价时的sku的价格
        Map<Long, Integer> maxPriceDiffPriceMap = maxPriceDiffPair.getValue0();
        // itemId：最大差价时的sku的原价
        Map<Long, Integer> maxPriceDiffOrigPriceMap = maxPriceDiffPair.getValue1();
        // itemId：最大差价
        Map<Long, Integer> maxPriceDiffMap = getMaxPriceDiff(skuList).getValue0();

        Set<Long> recommendIds = qihoItemRecommendDAO.findAllRecommendItemId();
        for (QihoItemDetailEntity entity : list) {
            ItemDto dto = new ItemDto();
            dto.setDeleted(entity.getDeleted());
            dto.setId(entity.getId());
            dto.setItemName(entity.getItemName());
            dto.setItemShortName(entity.getItemShortName());
            dto.setItemNo(entity.getItemNo());
            dto.setItemStatus(entity.getItemStatus());
            dto.setTopCategoryId(entity.getTopCategoryId());
            dto.setDelivery(entity.getDelivery());
            dto.setStock(itemStockMap.get(entity.getId()));
            Long enableStock = itemEnableStockMap.get(entity.getId());
            dto.setEnableStock(enableStock == null ? 0L : enableStock);

            // 所有sku中的最小价格
            dto.setMinPrice(minPriceMap.get(entity.getId()));
            // 所有sku中的最小原价
            dto.setMinOriginalPrice(minOriginalPriceMap.get(entity.getId()));
            // 最小价格时的sku的原价
            dto.setMinSkuOriginalPrice(minSkuOriginalPriceMap.get(entity.getId()));
            // 最小价格时的sku的差价
            dto.setMinSkuPriceDiff(minSkuPriceDiffMap.get(entity.getId()));
            // 最大差价sku
            dto.setMaxPriceDiff(maxPriceDiffMap.get(entity.getId()));
            // 最大差价时的sku的价格
            dto.setMaxPriceDiffPrice(maxPriceDiffPriceMap.get(entity.getId()));
            // 最大差价时的sku的原价
            dto.setMaxPriceDiffOriginalPrice(maxPriceDiffOrigPriceMap.get(entity.getId()));

            dto.setIsRecommend(recommendIds.contains(entity.getId()));
            Map<String, String> extParam = JSON
                .parseObject(entity.getExtParam(), new TypeReference<Map<String, String>>() {
                });
            dto.setExtParam(extParam);
            itemList.add(dto);
        }
        return itemList;
    }

    private Pair<Map,Map> getMaxPriceDiff(List<QihoItemSkuEntity> skuList) {
        // itemId: 最大差价sku
        Map<Long, Integer> maxPriceDiffMap = Maps.newHashMap();

        for (QihoItemSkuEntity skuEntity : skuList) {
            // 过滤无效sku
            if (!skuEntity.getSkuEnable() || skuEntity.getStock() == null || skuEntity.getStock() <= 0){
                continue;
            }

            Long itemId = skuEntity.getItemId();

            // 当前sku的原价
            Integer originalPrice = skuEntity.getOriginalPrice();
            originalPrice = originalPrice == null ? 0 : originalPrice;

            // 当前sku的价格
            Integer sellingPrice = skuEntity.getSellingPrice();
            sellingPrice = sellingPrice == null ? 0 : sellingPrice;

            // 当前sku的差价
            Integer priceDiff = originalPrice - sellingPrice;

            // 最大差价sku
            Integer maxPriceDiff = maxPriceDiffMap.get(itemId);
            maxPriceDiffMap.put(itemId, maxPriceDiff == null ? priceDiff : Math.max(maxPriceDiff, priceDiff));
        }
        return Pair.with(maxPriceDiffMap, null);
    }

    private Pair<Map,Map> getMaxPriceDiffSkuInfo(List<QihoItemSkuEntity> skuList) {
        // itemId: 最大差价sku
        Map<Long, Integer> maxPriceDiffMap = Maps.newHashMap();
        // itemId: 最大差价时的sku的价格
        Map<Long, Integer> maxPriceDiffPriceMap = Maps.newHashMap();
        // itemId: 最大差价时的sku的原价
        Map<Long, Integer> maxPriceDiffOrigPriceMap = Maps.newHashMap();

        for (QihoItemSkuEntity skuEntity : skuList) {
            // 过滤无效sku
            if (!skuEntity.getSkuEnable() || skuEntity.getStock() == null || skuEntity.getStock() <= 0){
                continue;
            }

            Long itemId = skuEntity.getItemId();

            // 当前sku的原价
            Integer originalPrice = skuEntity.getOriginalPrice();
            originalPrice = originalPrice == null ? 0 : originalPrice;

            // 当前sku的价格
            Integer sellingPrice = skuEntity.getSellingPrice();
            sellingPrice = sellingPrice == null ? 0 : sellingPrice;

            // 当前sku的差价
            Integer priceDiff = originalPrice - sellingPrice;

            // 最大差价sku
            Integer maxPriceDiff = maxPriceDiffMap.get(itemId);
            maxPriceDiffMap.put(itemId, maxPriceDiff == null ? priceDiff : Math.max(maxPriceDiff, priceDiff));

            // 若当前没有最大差价 或 最大差价比当前sku的差价小, 则表示当前sku为差价最大的sku, 故保存当前sku的价格和原价
            if (maxPriceDiff == null || (maxPriceDiff.intValue() < priceDiff.intValue())) {
                maxPriceDiffPriceMap.put(itemId, sellingPrice);
                maxPriceDiffOrigPriceMap.put(itemId, originalPrice);
            }
        }
        return Pair.with(maxPriceDiffPriceMap, maxPriceDiffOrigPriceMap);
    }

    private Pair<Map,Map> getMinPriceSkuInfo(List<QihoItemSkuEntity> skuList) {
        // itemId: 最小价格
        Map<Long, Integer> minPriceMap = Maps.newHashMap();
        // itemId: 最小价格时的sku的原价
        Map<Long, Integer> minSkuOriginalPriceMap = Maps.newHashMap();
        // itemId: 最小价格时的sku的差价
        Map<Long, Integer> minSkuPriceDiffMap = Maps.newHashMap();

        for (QihoItemSkuEntity skuEntity : skuList) {
            // 过滤无效sku
            if (!skuEntity.getSkuEnable() || skuEntity.getStock() == null || skuEntity.getStock() <= 0){
                continue;
            }

            Long itemId = skuEntity.getItemId();

            // 当前sku的原价
            Integer minSkuOriginalPrice = skuEntity.getOriginalPrice();
            minSkuOriginalPrice = minSkuOriginalPrice == null ? 0 : minSkuOriginalPrice;

            // 当前sku的价格
            Integer sellingPrice = skuEntity.getSellingPrice();
            sellingPrice = sellingPrice == null ? 0 : sellingPrice;

            // 最小价格
            Integer minPrice = minPriceMap.get(itemId);
            minPriceMap.put(itemId, minPrice == null ? sellingPrice : Math.min(minPrice, sellingPrice));

            // 若当前没有最小价格 或 最小价格比当前sku的价格大, 则当前sku的价格将替换原来最小价格
            // 同时保存当前sku的差价
            if (minPrice == null || (minPrice.intValue() > sellingPrice.intValue())) {
                minSkuOriginalPriceMap.put(itemId, minSkuOriginalPrice);
                minSkuPriceDiffMap.put(itemId, minSkuOriginalPrice - sellingPrice);
            }
        }
        return Pair.with(minSkuOriginalPriceMap, minSkuPriceDiffMap);
    }

    private Pair<Map, Map> getSkuPriceInfo(List<QihoItemSkuEntity> skuList) {
        // itemId: 最小价格
        Map<Long, Integer> minPriceMap = Maps.newHashMap();
        // itemId: 最小原价
        Map<Long, Integer> minOriginalPriceMap = Maps.newHashMap();
        for (QihoItemSkuEntity skuEntity : skuList) {
            // 过滤无效sku
            if (!skuEntity.getSkuEnable() || skuEntity.getStock() == null || skuEntity.getStock() <= 0){
                continue;
            }

            Long itemId = skuEntity.getItemId();
            // 商品最小价格
            Integer minPrice = minPriceMap.get(itemId);
            Integer sellingPrice = skuEntity.getSellingPrice() == null ? 0 : skuEntity.getSellingPrice();
            minPriceMap.put(itemId, minPrice == null ? sellingPrice : Math.min(minPrice, sellingPrice));
            // 商品最小原价
            Integer minOriginalPrice = minOriginalPriceMap.get(itemId);
            Integer originalPrice = skuEntity.getOriginalPrice() == null ? 0 : skuEntity.getOriginalPrice();
            minOriginalPriceMap
                .put(itemId, minOriginalPrice == null ? originalPrice : Math.min(minOriginalPrice, originalPrice));
        }
        return Pair.with(minPriceMap, minOriginalPriceMap);
    }

    @Override
    public Map<Long, QihoItemEntity> selectByIds(List<Long> itemIds) {
        List<QihoItemEntity> entityList = qihoItemDAO.selectByIds(itemIds);
        if (CollectionUtils.isEmpty(entityList)) {
            return Maps.newHashMap();
        }
        return entityList.stream().collect(Collectors.toMap(QihoItemEntity::getId, e -> e));
    }

    @Override
    public Map<Long, QihoItemEntity> selectByIdList(List<Long> itemIdList) {
        List<QihoItemEntity> entityList = qihoItemDAO.selectByIdList(itemIdList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Maps.newHashMap();
        }
        return entityList.stream().collect(Collectors.toMap(QihoItemEntity::getId, e -> e));
    }

    private Pair<Map, Map> getSkuStockInfo(Map<Long, Long> stockMap, List<QihoItemSkuEntity> skuList) {
        // itemId : 库存 键值对
        Map<Long, Long> itemStockMap = Maps.newHashMap();
        //有效库存
        Map<Long, Long> itemEnableStockMap = Maps.newHashMap();
        for (QihoItemSkuEntity skuEntity : skuList) {
            // sku库存
            Long stock = stockMap.get(skuEntity.getStockId());
            if (stock == null || stock <= 0) {
                stock = 0L;
            }
            Long itemId = skuEntity.getItemId();
            // 商品总库存
            Long itemStock = itemStockMap.get(itemId);
            itemStockMap.put(itemId, itemStock == null ? stock : itemStock + stock);
            if (skuEntity.getSkuEnable()) {
                Long itemEnableStock = itemEnableStockMap.get(itemId);
                if(itemEnableStock != null){
                    stock = itemEnableStock + stock;
                }

                itemEnableStockMap.put(itemId, stock);
                skuEntity.setStock(stock);
            }
        }
        return Pair.with(itemStockMap, itemEnableStockMap);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#createItem(com.qiho.center.api.dto.ItemDetailDto)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Long createItem(ItemDetailDto itemDetail) {
        QihoItemEntity item = BeanUtils.copy(itemDetail, QihoItemEntity.class);
        item.setDeleted(false);
        qihoItemDAO.insert(item);

        // 插入商品-商家对应关系
        QihoItemMerchantEntity qihoItemMerchantEntity = new QihoItemMerchantEntity();
        qihoItemMerchantEntity.setMerchantId(itemDetail.getMerchantDto().getId());
        qihoItemMerchantEntity.setItemId(item.getId());
        qihoItemMerchantEntity.setGmtModifier(itemDetail.getGmtModifier());
        qihoItemMerchantEntity.setGmtCreator(itemDetail.getGmtCreator());
        qihoItemMerchantDAO.insert(qihoItemMerchantEntity);
        Long itemId = item.getId();

        QihoItemExtEntity itemExt = new QihoItemExtEntity();
        itemExt.setDetail(itemDetail.getDetail());
        //更新虚拟库存
        Map<String, String> extParam = itemDetail.getExtParam();
        Long virStock = processVirtailStock(itemDetail, Boolean.TRUE);
        extParam.put(ItemConstants.ItemExtConstans.VIRTUALSTOCK, String.valueOf(virStock));
        itemExt.setExtParam(JSON.toJSONString(extParam));
        itemExt.setItemId(itemId);
        qihoItemExtDAO.insert(itemExt);

        //插入之后,同时插入一条详情操作记录
        addRecord(itemExt);
        // 关联商品属性
        List<SkuPropertyDto> propertyList = itemDetail.getSkuPropertyList();
        skuPropertyService.updateItemId(propertyList, itemId);
        // 生效属性
        skuPropertyService.enableProperty(propertyList, itemId);

        // 创建SKU
        itemSkuService.updateItemSku(itemDetail.getSkuList(), itemId);

        // 如果是复制的商品 且和原商品的一级类目相同 则连同原商品的标签一并复制
        if (itemDetail.getOriginItemId() != null && itemDetail.getOriginItemId().longValue() > 0L) {
            QihoItemEntity originItem = qihoItemDAO.findSimpleById(itemDetail.getOriginItemId());
            if (originItem.getTopCategoryId().longValue() == itemDetail.getTopCategoryId().longValue()) {
                itemTagService.copyItemTag(itemDetail.getOriginItemId(), itemId);
            }
        }
        return itemId;
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#updateItem(com.qiho.center.api.dto.ItemDetailDto)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateItem(ItemDetailDto itemDetail) {
        QihoItemEntity oldItem = qihoItemDAO.findSimpleById(itemDetail.getId());
        if (oldItem.getTopCategoryId().longValue() != itemDetail.getTopCategoryId().longValue()) {
            // 如果商品的一级类目发生变更 则删除商品所关联的标签
            itemTagService.deleteByItemId(itemDetail.getId());
        }

        // 更新商品详情
        QihoItemEntity item = BeanUtils.copy(itemDetail, QihoItemEntity.class);
        qihoItemDAO.update(item);
        QihoItemMerchantEntity qihoItemMerchantEntity = qihoItemMerchantDAO.findByItemId(item.getId());

        if (qihoItemMerchantEntity == null || !Objects.equals(qihoItemMerchantEntity.getMerchantId(), itemDetail.getMerchantDto().getId())) {

            QihoItemMerchantEntity itemMerchantEntity = new QihoItemMerchantEntity();
            itemMerchantEntity.setItemId(item.getId());
            itemMerchantEntity.setMerchantId(itemDetail.getMerchantDto().getId());
            itemMerchantEntity.setGmtModifier(itemDetail.getGmtModifier());
            itemMerchantEntity.setGmtCreator(itemDetail.getGmtModifier());
            qihoItemMerchantDAO.insert(itemMerchantEntity);
        }

        Long itemId = itemDetail.getId();
        QihoItemExtEntity itemExt = new QihoItemExtEntity();
        itemExt.setDetail(itemDetail.getDetail());
        //更新虚拟库存
        Map<String, String> extParam = itemDetail.getExtParam();
        Long virStock = processVirtailStock(itemDetail, Boolean.FALSE);
        extParam.put(ItemConstants.ItemExtConstans.VIRTUALSTOCK, String.valueOf(virStock));
        itemExt.setExtParam(JSON.toJSONString(itemDetail.getExtParam()));
        itemExt.setItemId(itemId);
        qihoItemExtDAO.updateByItemId(itemExt);

        //更新后也插入一条详情变更记录
        addRecord(itemExt);

        // 关联商品属性
        List<SkuPropertyDto> propertyList = itemDetail.getSkuPropertyList();
        skuPropertyService.updateItemId(propertyList, itemId);
        // 生效属性
        skuPropertyService.enableProperty(propertyList, itemId);

        // 更新sku信息
        itemSkuService.updateItemSku(itemDetail.getSkuList(), itemId);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#queryItemDetail(java.lang.Long)
     */
    @Override
    public ItemDetailDto queryItemDetail(Long itemId) {
        QihoItemDetailEntity qihoItemDetailEntity = qihoItemDAO.findById(itemId);
        MerchantDto merchantDto = merchantService.findByItemId(itemId);
        if (qihoItemDetailEntity == null) {
            return null;
        }
        ItemDto itemDto = convertToItemDto(Lists.newArrayList(qihoItemDetailEntity)).get(0);
        ItemDetailDto itemDetailDto = BeanUtils.copy(itemDto, ItemDetailDto.class);
        itemDetailDto.setDetail(qihoItemDetailEntity.getDetail());
        if (merchantDto != null) {
            itemDetailDto.setMerchantDto(merchantDto);
        }
        itemDetailDto.setPriceText(qihoItemDetailEntity.getPriceText());
        itemDetailDto.setGmtCreator(qihoItemDetailEntity.getGmtCreator());
        itemDetailDto.setGmtModifier(qihoItemDetailEntity.getGmtModifier());
        itemDetailDto.setGmtModified(qihoItemDetailEntity.getGmtModified());
        itemDetailDto.setSkuPropertyList(skuPropertyService.getSkuPropertyByItemId(itemId));
        itemDetailDto.setSkuList(itemSkuService.getSkuByItemId(itemId));
        itemDetailDto.setDeleted(qihoItemDetailEntity.getDeleted());
        itemDetailDto.setItemEvaluateList(itemEvaluateService.queryItmeEvalList(itemId));
        return itemDetailDto;
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#updateStatus(java.util.List, java.lang.String)
     */
    @Override
    public void updateStatus(List<Long> itemIds, String status, Long gmtModifier) {
        qihoItemDAO.updateStatus(itemIds, status, gmtModifier);
    }

    @Override
    public void deleteBatch(List<Long> itemIds, Long gmtModifier) {
        qihoItemDAO.deleteBatch(itemIds, gmtModifier);
        // 发布删除商品事件
        ItemDeleteEvent event = new ItemDeleteEvent();
        event.addAllItemId(itemIds);
        eventBus.post(event);
    }

    @Override
    public List<ItemDto> findItemDtoByItemIds(Set<Long> itemIds) {
        List<QihoItemDetailEntity> list = qihoItemDAO.findByIds(Lists.newArrayList(itemIds));
        List<QihoItemDetailEntity> newList = Lists.newArrayList();
        for (QihoItemDetailEntity entity : list) {
            if (!entity.getDeleted()) {
                newList.add(entity);
            }
        }
        return convertToItemDto(newList);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#findRecommendIndex()
     */
    @Override
    public List<ItemDto> findRecommendIndex() {
        List<QihoItemDetailEntity> recommendList = qihoItemDAO.findRecommendIndex();
        return convertToItemDto(recommendList);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#queryItemSimpleDto(java.lang.Long)
     */
    @Override
    public ItemSimpleDto queryItemSimpleDto(Long itemId) {
        QihoItemEntity entity = qihoItemDAO.findSimpleById(itemId);
        return BeanUtils.copy(entity, ItemSimpleDto.class);
    }

    @Override
    public List<ItemSimpleDto> queryItemSimpleList(ItemDto queryItem) {
        QihoItemEntity param = BeanUtils.copy(queryItem, QihoItemEntity.class);
        List<QihoItemDetailEntity> list = qihoItemDAO.querySimpleListByParam(param);
        return BeanUtils.copyList(list, ItemSimpleDto.class);
    }

    @Override
    public void updateExtParamByItemId(ItemAdvertDto dto) {
        QihoItemExtEntity itemExt = BeanUtils.copy(dto, QihoItemExtEntity.class);
        qihoItemExtDAO.updateByItemId(itemExt);
        addRecord(itemExt);
    }

    @Override
    public ItemAdvertDto queryExtParamByItemId(Long itemId) {
        QihoItemExtEntity qihoItemExtEntity = qihoItemExtDAO.queryExtParamByItemId(itemId);
        return BeanUtils.copy(qihoItemExtEntity, ItemAdvertDto.class);
    }

    @Override
    public Long queryItemStockByItemId(Long itemId) {
        List<Long> idList = Stream.of(itemId).collect(Collectors.toList());
        List<QihoItemSkuEntity> skuList = qihoItemSkuDAO.getSkuByItemIds(idList);
        List<Long> stockIds = Lists.newArrayList();
        for (QihoItemSkuEntity skuEntity : skuList) {
            stockIds.add(skuEntity.getStockId());
        }
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
        Long stockCount = 0L;
        for (Map.Entry<Long, Long> stock : stockMap.entrySet()) {
            stockCount += stock.getValue();
        }
        return stockCount;
    }

    /**
     * 这个方法主要是针对定时任务同步库存时候,出现消耗小于0,客户端调用时重新计算虚拟库存以及重置虚拟库存
     *
     * @param itemId
     * @return
     */
    @Override
    public Long updateVirtailStock(Long itemId) {
        Long allStock = queryItemStockByItemId(itemId);
        Long virStock = cacluateVirStock(allStock);
        QihoItemExtEntity extEntity = qihoItemExtDAO.queryExtParamByItemId(itemId);
        String extParam = extEntity.getExtParam();
        JSONObject extJson;
        if (StringUtils.isBlank(extParam)) {
            extJson = new JSONObject();
        } else {
            extJson = JSONObject.parseObject(extParam);
        }
        extJson.put(ItemConstants.ItemExtConstans.VIRTUALSTOCK, virStock);
        QihoItemExtEntity newItemAdvertEntity = new QihoItemExtEntity();
        newItemAdvertEntity.setItemId(itemId);
        newItemAdvertEntity.setExtParam(extJson.toJSONString());
        qihoItemExtDAO.updateByItemId(newItemAdvertEntity);
        return virStock;
    }

    @Override
    public PagenationDto<ItemDto> queryItemPage(ItemPageParams params) {
        if (params.getMerchantId() == null || params.getMerchantId().longValue() <= 0L) {
            throw new QihoException("无效的商家id");
        }
        PagenationDto<ItemDto> page = new PagenationDto<>();
        int count = qihoItemDAO.countSizeByparam(params);
        page.setTotal(count);
        if (count == 0) {
            page.setList(Lists.newArrayList());
            return page;
        }
        List<QihoItemDetailEntity> list = qihoItemDAO.queryByParam(params);
        page.setList(convertToItemDto(list));
        return page;
    }

    /**
     * 处理新增或者更新商品库存时的虚拟库存
     * true=新增,false=更新
     *
     * @param itemDetailDto
     * @return
     */
    private Long processVirtailStock(ItemDetailDto itemDetailDto, Boolean flag) {
        List<ItemSkuDto> list = itemDetailDto.getSkuList();
        Long allStock = 0L;
        Long changeStock = 0L;
        for (ItemSkuDto itemSkuDto : list) {
            allStock += itemSkuDto.getStock();
            if (null != itemSkuDto.getStockChanges()) {
                changeStock += itemSkuDto.getStockChanges();
            }
        }
        if (flag) {
            return cacluateVirStock(allStock);
        } else {//更新操作,计算虚拟库存用增量方式
            Map<String, String> extParam = itemDetailDto.getExtParam();
            String virStockStr = extParam.get(ItemConstants.ItemExtConstans.VIRTUALSTOCK);
            Long virStock;
            if (StringUtils.isBlank(virStockStr)) {
                virStock = cacluateVirStock(allStock);
            } else {
                virStock = Long.valueOf(virStockStr);
            }
            //增量修改虚拟库存
            return new BigDecimal(virStock).add(new BigDecimal(changeStock)).longValue();
        }
    }

    /**
     * 用于通过实际库存计算虚拟库存
     *
     * @param allStock
     * @return
     */
    private Long cacluateVirStock(Long allStock) {
        Long consumeDefault = StringRandUtil.getRandom(50, 30);
        BigDecimal consumeRate = new BigDecimal(consumeDefault)
            .divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP);
        return new BigDecimal(allStock).divide(consumeRate, 0, BigDecimal.ROUND_HALF_UP).longValue();
    }

    private void addRecord(QihoItemExtEntity itemExt) {
        //插入之后,同时插入一条详情操作记录
        QihoItemDetailRecordEntity recordEntity = new QihoItemDetailRecordEntity();
        recordEntity.setItemId(itemExt.getItemId());
        if (StringUtils.isBlank(itemExt.getDetail())) {
            return;
        }
        recordEntity.setDetail(itemExt.getDetail());
        recordEntity.setSourceType(1);
        recordEntity.setSourceId(itemExt.getId());
        itemDetailRecordService.addDetailRecord(recordEntity);
    }

    @Override
    public List<ItemDto> queryItemsByItemIds(List<Long> itemIdList) {
        return convertToItemDtoWithoutSkuStatus(qihoItemDAO.findByIds(itemIdList));
    }

    private List<ItemDto> convertToItemDtoWithoutSkuStatus(List<QihoItemDetailEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }

        List<Long> idList = Lists.newArrayList();
        for (QihoItemDetailEntity entity : list) {
            idList.add(entity.getId());
        }

        List<QihoItemSkuEntity> skuList = qihoItemSkuDAO.getSkuByItemIds(idList);
        // 过滤无效的sku对象(库存为0或状态为关闭)
        List<QihoItemSkuEntity> itemSkuEntityList = filterSkuStock(skuList);
        if (CollectionUtils.isEmpty(itemSkuEntityList)) {
            return Lists.newArrayList();
        }

        return setItemDtoInfo(list, itemSkuEntityList);
    }

    private List<QihoItemSkuEntity> filterSkuStock(List<QihoItemSkuEntity> skuList) {
        if (CollectionUtils.isEmpty(skuList)) {
            return Lists.newArrayList();
        }

        // 过滤出sku状态为false的对象
        List<QihoItemSkuEntity> itemSkuEntityList = skuList.stream()
                .filter(QihoItemSkuEntity::getSkuEnable).collect(Collectors.toList());

        // 提取所有stockId, 并查询对应的库存
        List<Long> stockIds = itemSkuEntityList.stream().map(QihoItemSkuEntity::getStockId).collect(Collectors.toList());
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();

        // 过滤出sku库存为0的对象
        itemSkuEntityList = itemSkuEntityList.stream()
                .filter(itemSkuEntity -> stockMap.get(itemSkuEntity.getStockId()) > 0).collect(Collectors.toList());

        return itemSkuEntityList;
    }
}
