package com.qiho.center.biz.service.impl.account;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.qiho.center.api.constant.AccountConstant;
import com.qiho.center.api.constant.AccountMenuConstant;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.account.BaiqiAccountDto;
import com.qiho.center.api.dto.account.BaiqiMainAndAgenAccountDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.enums.account.BaiqiAccountStatusEnum;
import com.qiho.center.api.enums.account.BaiqiAccountTypeEnum;
import com.qiho.center.api.params.account.AccountPageParam;
import com.qiho.center.api.params.menu.AccountMenuParams;
import com.qiho.center.biz.service.account.BaiqiAccountService;
import com.qiho.center.biz.service.agent.BaiqiAgentService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoMerchantDAO;
import com.qiho.center.common.daoh.qiho.account.BaiqiAccountMapper;
import com.qiho.center.common.daoh.qiho.agent.BaiqiAgentMapper;
import com.qiho.center.common.daoh.qiho.menu.BaiqiAccountMenuMapper;
import com.qiho.center.common.daoh.qiho.menu.BaiqiMenuMapper;
import com.qiho.center.common.entity.merchant.QihoMerchantEntity;
import com.qiho.center.common.entityd.qiho.account.BaiqiAccountEntity;
import com.qiho.center.common.entityd.qiho.agent.BaiqiAgentEntity;
import com.qiho.center.common.entityd.qiho.menu.BaiqiAccountMenuEntity;
import com.qiho.center.common.entityd.qiho.menu.BaiqiMenuEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by sunchangji on 2017/12/7.
 */
@Service("baiqiAccountService")
public class BaiqiAccountServiceImpl implements BaiqiAccountService {

    @Resource
    private BaiqiAccountMapper baiqiAccountMapper;
    @Resource
    private BaiqiAccountMenuMapper baiqiAccountMenuMapper;
    @Resource
    private BaiqiMenuMapper baiqiMenuMapper;
    @Resource
    private MerchantService merchantService;
    @Resource
    private BaiqiAgentService baiqiAgentService;

    @Autowired
    private BaiqiAgentMapper baiqiAgentMapper;

    @Autowired
    private QihoMerchantDAO qihoMerchantDAO;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Override
    public BaiqiAccountDto findByAccount(String account) {
        BaiqiAccountEntity accountEntity = baiqiAccountMapper.findByAccount(account);
        //判断是否为商家用户
        if(!Objects.isNull(accountEntity)&&(accountEntity.getAccountType() == BaiqiAccountTypeEnum.SUB_ACCOUNT.getCode()||
                accountEntity.getAccountType() == BaiqiAccountTypeEnum.MAIN_ACCOUNT.getCode())){
            MerchantDto merchantDto = merchantService.findById(accountEntity.getRelationId());
            //附加商家版本标识
            if(!Objects.isNull(merchantDto)){
                accountEntity.setVersionSign(merchantDto.getVersionSign());
            }
        }
        return BeanUtils.copy(accountEntity, BaiqiAccountDto.class);
    }

    @Override
    public BaiqiAccountDto findByMobile(String mobile) {
        BaiqiAccountEntity accountEntity = baiqiAccountMapper.findByMobile(mobile);
        //判断是否为商家用户
        if(!Objects.isNull(accountEntity)&&(accountEntity.getAccountType() == BaiqiAccountTypeEnum.SUB_ACCOUNT.getCode()||
                accountEntity.getAccountType() == BaiqiAccountTypeEnum.MAIN_ACCOUNT.getCode())){
            MerchantDto merchantDto = merchantService.findById(accountEntity.getRelationId());
            //附加商家版本标识
            if(!Objects.isNull(merchantDto)){
                accountEntity.setVersionSign(merchantDto.getVersionSign());
            }
        }
        return BeanUtils.copy(accountEntity, BaiqiAccountDto.class);
    }

    @Override
    public boolean deleteById(Long id) {

        //删除账号
        baiqiAccountMapper.deleteById(id);

        //删除账号权限
        baiqiAccountMenuMapper.deleteByAccountId(id);

        return true;
    }

    @Override
    public PagenationDto<BaiqiMainAndAgenAccountDto> findPageAccount(AccountPageParam pageQueryParams) {
        // update by chensong on 2018/03/30 due to agent-merchant-account-optimize requirement
        PagenationDto<BaiqiMainAndAgenAccountDto> pagenationDto = new PagenationDto<>();
        //  先根据merchantName or agentName 参数获取idList
        if (StringUtils.isNotEmpty(pageQueryParams.getMerchantName())) { // 有空格也要去查询
            List<QihoMerchantEntity> merchantEntityList = qihoMerchantDAO.listByMerchantName(pageQueryParams.getMerchantName());
            if (CollectionUtils.isEmpty(merchantEntityList)) {
                return pagenationDto.emptyPage();
            }
            pageQueryParams.setMerchantIdList(
                    merchantEntityList.stream().map(QihoMerchantEntity::getId).collect(Collectors.toList()));
        }
        if (StringUtils.isNotEmpty(pageQueryParams.getAgentName())) {
            List<BaiqiAgentEntity> agentEntityList = baiqiAgentMapper.listByAgentName(pageQueryParams.getAgentName());
            if (CollectionUtils.isEmpty(agentEntityList)) {
                return pagenationDto.emptyPage();
            }
            pageQueryParams.setAgentIdList(
                    agentEntityList.stream().map(BaiqiAgentEntity::getId).collect(Collectors.toList()));
        }

        int total = baiqiAccountMapper.countPageMainAccount(pageQueryParams);
        pagenationDto.setTotal(total);
        if (total == 0) {
            pagenationDto.setList(Collections.emptyList());
            return pagenationDto;
        }
        List<BaiqiAccountEntity> entities = baiqiAccountMapper.findPageAccount(pageQueryParams);

        pagenationDto.setList(convertDtoList(entities));
        return pagenationDto;
    }

    /**
     * 转换类型
     * @param entities
     * @return
     */
    private List<BaiqiMainAndAgenAccountDto> convertDtoList(List<BaiqiAccountEntity> entities){
        List<Long> merchantIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();
        setRelationIds(merchantIds, agentIds, entities);
        Map<Long, MerchantDto> merchantDtoMap = new HashMap<>();
        Map<Long, BaiqiAgentEntity> agentMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(merchantIds)) {
            merchantDtoMap = merchantService.findByIdList(merchantIds);
        }
        if (CollectionUtils.isNotEmpty(agentIds)) {
            agentMap = baiqiAgentService.findByIdsMap(agentIds);
        }
        List<BaiqiMainAndAgenAccountDto> mainAccountDtos = new ArrayList<>();
        for (BaiqiAccountEntity entity : entities) {
            Long relationId = entity.getRelationId();
            int accountType = entity.getAccountType();
            BaiqiMainAndAgenAccountDto accountDto = BeanUtils.copy(entity, BaiqiMainAndAgenAccountDto.class);
            accountDto.setPwd(AccountConstant.PD_VALUE);
            if (accountType == BaiqiAccountTypeEnum.AGENT_ACCOUNT.getCode()) {
                accountDto.setAccountTypeName(BaiqiAccountTypeEnum.AGENT_ACCOUNT.getTypeName());
                //设置代理商名称
                BaiqiAgentEntity agentEntity = agentMap.get(relationId);
                accountDto.setRelationName(agentEntity == null ? null : agentEntity.getAgentName());
            } else {
                accountDto.setAccountTypeName(BaiqiAccountTypeEnum.MAIN_ACCOUNT.getTypeName());
                MerchantDto merchantDto = merchantDtoMap.get(relationId);
                accountDto.setRelationName(merchantDto == null ? null : merchantDto.getMerchantName());
            }

            mainAccountDtos.add(accountDto);
        }
        return mainAccountDtos;
    }



    private void setRelationIds(List<Long> merchantIds, List<Long> agentIds, List<BaiqiAccountEntity> entities) {
        for (BaiqiAccountEntity entity : entities) {
            int accountType = entity.getAccountType();
            Long relationId = entity.getRelationId();
            if (accountType == BaiqiAccountTypeEnum.AGENT_ACCOUNT.getCode()) {
                agentIds.add(relationId);
            } else {
                merchantIds.add(relationId);
            }
        }
    }

    @Override
    public Boolean disableMainAccount(Long id) throws BizException {
        BaiqiAccountDto accountDto = baiqiAccountMapper.findById(id);
        if (accountDto == null) {
            throw new BizException("主账号不存在");
        }
        //禁用主账号或代理商
        boolean result = disable(id);
        if (!result) {
            return Boolean.FALSE;
        }
        int accountType = accountDto.getAccountType();
        String account = accountDto.getAccount();
        //剔除账号redis中用户登录信息
        advancedCacheClient.remove(DsConstants.LOGIN_USER_REDIS_KEY + account);
        if (accountType != BaiqiAccountTypeEnum.MAIN_ACCOUNT.getCode()) {
            return Boolean.TRUE;
        }
        //商家主账户,禁用所有主账号下子账户
        List<BaiqiAccountEntity> entities = baiqiAccountMapper.findAllSubAccountByParentId(id);
        if (CollectionUtils.isEmpty(entities)) {
            return Boolean.TRUE;
        }
        int total = baiqiAccountMapper.disableSubAccountByParentId(id);
        if (total <= 0) {
            return Boolean.FALSE;
        }
        //剔除主账号所有redis中登录信息和权限信息
        List<String> redisKeys = new ArrayList<>();
        for (BaiqiAccountEntity accountEntity : entities) {
            redisKeys.add(DsConstants.LOGIN_USER_REDIS_KEY + accountEntity.getAccount());
        }
        advancedCacheClient.remove(redisKeys.toArray(new String[redisKeys.size()]));
        return Boolean.TRUE;
    }

    @Override
    public List<BaiqiAccountDto> findByParentId(Long mainAid) {
        //查询
        List<BaiqiAccountEntity> accountList = baiqiAccountMapper.findByParentId(mainAid);

        return BeanUtils.copyList(accountList, BaiqiAccountDto.class);
    }

    @Override
    public List<String> findAllMainAccount() {
        return baiqiAccountMapper.findAllMainAccount();
    }

    @Override
    public Boolean enable(Long id) {

        BaiqiAccountEntity accountEntity = new BaiqiAccountEntity();
        accountEntity.setId(id);
        accountEntity.setState(BaiqiAccountStatusEnum.ENABLE.getCode());

        return baiqiAccountMapper.update(accountEntity) > 0;
    }

    @Override
    public Boolean disable(Long id) {
        BaiqiAccountEntity accountEntity = new BaiqiAccountEntity();
        accountEntity.setId(id);
        accountEntity.setState(BaiqiAccountStatusEnum.DISABLE.getCode());

        return baiqiAccountMapper.update(accountEntity) > 0;
    }

    @Override
    public Boolean save(BaiqiAccountDto accountDto) throws BizException {

        BaiqiAccountEntity accountEntity = BeanUtils.copy(accountDto, BaiqiAccountEntity.class);

        //账号id
        Long accountId = accountEntity.getId();
        if (accountId == null) {
            //insert
            insertAccount(accountEntity);

            accountId = accountEntity.getId();
        } else {
            //update
            updateAccount(accountEntity);
        }

        //菜单id字符集
        String menuIdStr = accountDto.getMenuIds();
        if (StringUtils.isBlank(menuIdStr)) {
            return true;
        }
        //账号类型
        int accountType = accountEntity.getAccountType();
        //有菜单id集并且是子账号，处理权限更新
        if (BaiqiAccountTypeEnum.SUB_ACCOUNT.getCode() == accountType) {
            // save menu
            doSaveMenu(accountId, menuIdStr, accountDto.getAccount());
        }

        return true;
    }

    private void doSaveMenu(Long accountId, String menuIdStr, String accountName) {
        //菜单id集
        List<Long> menuIds = Arrays.stream(menuIdStr.split(",")).map(Long::valueOf).collect(Collectors.toList());

        //菜单id检验
        List<BaiqiMenuEntity> menuEntityList = baiqiMenuMapper.selectByIdList(menuIds);
        if (CollectionUtils.isNotEmpty(menuEntityList)) {

            List<BaiqiAccountMenuEntity> insertList = Lists.newArrayListWithExpectedSize(menuEntityList.size());
            menuEntityList.forEach(e -> {

                BaiqiAccountMenuEntity accountMenuEntity = new BaiqiAccountMenuEntity();
                accountMenuEntity.setAccountId(accountId);
                accountMenuEntity.setMenuId(e.getId());

                insertList.add(accountMenuEntity);
            });

            //1、先删除账号权限
            baiqiAccountMenuMapper.deleteByAccountId(accountId);

            //2、批量插入
            baiqiAccountMenuMapper.insertBatch(insertList);

            //3、清除登陆认证
            advancedCacheClient.remove(AccountMenuConstant.LOGIN_USER_REDIS_KEY + accountName);
        }
    }

    private void insertAccount(BaiqiAccountEntity accountEntity) throws BizException {
        //账号id
        Long accountId = accountEntity.getId();
        //账号名称
        BaiqiAccountEntity existsAccount = baiqiAccountMapper.findByAccount(accountEntity.getAccount());
        if (existsAccount != null && !existsAccount.getId().equals(accountId)) {
            throw new BizException("账号名称已存在");
        }
        //手机号
        BaiqiAccountEntity existsMobile = baiqiAccountMapper.findByMobile(accountEntity.getMobile());
        if (existsMobile != null && !existsMobile.getId().equals(accountId)) {
            throw new BizException("手机号已存在");
        }
        Integer accountType = accountEntity.getAccountType();
        //子账户不做验证
        if (BaiqiAccountTypeEnum.SUB_ACCOUNT.getCode() == accountType) {
            baiqiAccountMapper.insert(accountEntity);
            return;
        }
        Long relationId = accountEntity.getRelationId();
        String account = baiqiAccountMapper.findAccountByTypeAndRelationId(accountType, relationId);
        if (StringUtils.isBlank(account)) {
            baiqiAccountMapper.insert(accountEntity);
            return;
        }

        //主账号新增要校验关联的商家或代理商是否已经被其他主账号关联过
        if (BaiqiAccountTypeEnum.MAIN_ACCOUNT.getCode() == accountType) {
            throw new BizException("商家id=[" + relationId + "]已与主账号=[" + account + "]关联,不可重复关联");
        } else {
            throw new BizException("代理商id=[" + relationId + "]已与代理商账号=[" + account + "]关联,不可重复关联");
        }
    }

    private void updateAccount(BaiqiAccountEntity accountEntity) throws BizException {

        //账号id
        Long accountId = accountEntity.getId();

        //手机号
        BaiqiAccountEntity existsMobile = baiqiAccountMapper.findByMobile(accountEntity.getMobile());
        if (existsMobile != null && !existsMobile.getId().equals(accountId)) {
            throw new BizException("手机号已存在");
        }

        baiqiAccountMapper.update(accountEntity);
    }

    @Override
    public BaiqiAccountDto findById(Long id) {

        BaiqiAccountDto accountDto = baiqiAccountMapper.findById(id);

        //添加菜单集
        AccountMenuParams params = new AccountMenuParams();
        params.setAccountId(id);

        List<BaiqiAccountMenuEntity> accountMenuList = baiqiAccountMenuMapper.selectByParam(params);
        if (CollectionUtils.isNotEmpty(accountMenuList)) {
            List<Long> menuIds = accountMenuList.stream().map(BaiqiAccountMenuEntity::getMenuId)
                .collect(Collectors.toList());
            accountDto.setMenuIds(Joiner.on(",").join(menuIds));
        }
        return accountDto;
    }

    @Override
    public String findAccountById(Long id) {
        if (id == null) {
            return "";
        }
        return baiqiAccountMapper.findAccountById(id);
    }

    @Override
    public Boolean update(BaiqiAccountEntity accountEntity) {
        return baiqiAccountMapper.update(accountEntity) > 0;
    }

    @Override
    public List<Long> findRelationIdsByAccountType(Integer accountType) {
        return baiqiAccountMapper.findRelationIdsByAccountType(accountType);
    }

    @Override
    public BaiqiAccountEntity findMainAccountByMerchantId(Long merchantId) {
        BaiqiAccountEntity accountEntity = baiqiAccountMapper.findMainAccountByMerchantId(merchantId);
        //判断是否为商家用户
        if(!Objects.isNull(accountEntity)&&(accountEntity.getAccountType() == BaiqiAccountTypeEnum.SUB_ACCOUNT.getCode()||
                accountEntity.getAccountType() == BaiqiAccountTypeEnum.MAIN_ACCOUNT.getCode())){
            MerchantDto merchantDto = merchantService.findById(accountEntity.getRelationId());
            //附加商家版本标识
            if(!Objects.isNull(merchantDto)){
                accountEntity.setVersionSign(merchantDto.getVersionSign());
            }
        }
        return accountEntity;
    }

    @Override
    public BaiqiAccountEntity findByTypeAndRelationId(Integer accountType, Long relationId) {
        return baiqiAccountMapper.findByTypeAndRelationId(accountType, relationId);
    }

    @Override
    public List<Long> filterMerchantIds(List<Long> merchantIds) {
        return baiqiAccountMapper.filterMerchantIds(merchantIds);
    }
}
