package com.qiho.center.biz.service.impl.finance;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import com.qiho.center.api.constant.AllocateAmountConstant;
import com.qiho.center.api.dto.finance.AllocateAmountDto;
import com.qiho.center.api.enums.finance.BaiqiFinanceRechargeStateEnum;
import com.qiho.center.api.enums.finance.CashRebateAmountEnum;
import com.qiho.center.api.enums.finance.FinanceDetailStatusEnum;
import com.qiho.center.api.enums.finance.FinanceOptTypeEnum;
import com.qiho.center.api.enums.finance.FinanceTypeEnum;
import com.qiho.center.api.enums.finance.ParterTypeEnum;
import com.qiho.center.api.enums.finance.PartnerResultStateEnum;
import com.qiho.center.api.enums.finance.TuiaTradeTypeEnum;
import com.qiho.center.biz.service.finance.AllocateAmountService;
import com.qiho.center.biz.service.finance.FinanceDetailService;
import com.qiho.center.biz.service.finance.FinanceService;
import com.qiho.center.biz.service.impl.finance.bean.CashRebateAmountBean;
import com.qiho.center.biz.service.impl.finance.bean.PartnerAllocateAmountBean;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceDetailMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceRechargeMapper;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceDetailEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceEntity;
import com.qiho.center.common.util.BaiqiSerialUtil;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.tuia.core.api.remoteservice.baiqi.RemoteBaiQiService;
import cn.com.duiba.wolf.utils.UUIDUtils;

/**
 * AllocateAmountImpl
 *
 * @author zhangshun
 * @version V1.0
 * @since 2017-12-27 20:15
 */
@Service
public class AllocateAmountImpl implements AllocateAmountService {

    private static final Logger LOGGER = LoggerFactory.getLogger(AllocateAmountImpl.class);

    @Resource
    private BaiqiFinanceDetailMapper baiqiFinanceDetailMapper;

    @Resource
    private FinanceService financeService;

    @Resource
    private RemoteBaiQiService remoteBaiQiService;

    @Resource
    private BaiqiFinanceMapper baiqiFinanceMapper;

    @Resource
    private BaiqiFinanceRechargeMapper baiqiFinanceRechargeMapper;

    @Resource
    private FinanceDetailService financeDetailService;

    /**
     * 调用推啊接口
     *
     * @return 0=失败、1=成功、2=超时或异常
     */
    @Override
    public PartnerResultStateEnum callPartnerAllocateAmount(PartnerAllocateAmountBean bean) {
        try {
            // 调用外部接口
            if (remoteBaiQiService
                .allocateAmount(bean.getBaiqiTradeNo(), bean.getTradeType().getCode(), bean.getAccountId(),
                    bean.getAmount())) {
                return PartnerResultStateEnum.SUCCEED;
            } else {
                return PartnerResultStateEnum.FAILED;
            }
        } catch (Exception e) {
            LOGGER.error("[金额分配]错误. BaiqiTradeNo:{}, {}", bean.getBaiqiTradeNo(), bean, e);
            return PartnerResultStateEnum.EXCEPTION;
        }
    }

    @Override
    public PartnerResultStateEnum checkPartnerAllocateAmount(String baiqiTradeNo) {
        try {
            if (remoteBaiQiService.checkTrade(baiqiTradeNo)) {
                return PartnerResultStateEnum.SUCCEED;
            } else {
                return PartnerResultStateEnum.FAILED;
            }
        } catch (Exception e) {
            LOGGER.error("[检查金额分配]错误, baiqiTradeNo:{}", baiqiTradeNo);
            return PartnerResultStateEnum.EXCEPTION;
        }
    }

    // 分割线

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PartnerResultStateEnum allocateAmountRecharge(AllocateAmountDto param) throws BizException {

        Assert.notNull(param, "参数为空");
        Assert.notNull(param.getAgentId(), "代理商账号ID为空");
        Assert.notNull(param.getAgentFinanceId(), "代理商账户ID为空");
        Assert.notNull(param.getMerchantId(), "商家账号ID为空");
        Assert.notNull(param.getMerchantFinanceId(), "商家账户ID为空");
        Assert.notNull(param.getAdvertId(), "广告主ID为空");
        Assert.notNull(param.getAmount(), "充值金额为空");
        Assert.notNull(param.getOperator(), "操作人为空");

        LOGGER.info("[金额划拨参数] {}", param);

        // 计算代理商现金和返点的支出金额
        CashRebateAmountBean amountBean = financeService
            .calcCashAndRebateAmount(param.getAgentFinanceId(), param.getAmount(), CashRebateAmountEnum.EXPENDITURE);

        if (null == amountBean) {
            LOGGER.error("计算代理商的现金和返点扣比为空");
            return PartnerResultStateEnum.FAILED;
        }

        LOGGER.info("[金额划拨分配比例] {}", amountBean);

        //交易号
        String tradeNo = UUIDUtils.createUUID();

        // 添加代理商支出流水，关联类型为(2:代理商)，操作类型为(4:代理商划拨)，状态为(1:处理中)
        BaiqiFinanceDetailEntity agentDetail = this
            .insertAgentDetailRecharge(param.getAgentFinanceId(), param.getAgentId(), param.getOperator(), tradeNo,
                amountBean);

        LOGGER.info("[代理商支出流水] {}", agentDetail);

        // 添加商家收入流水，关联类型为(0:商家主账号)，操作类型为(4:代理商划拨)，状态为(1:完成)
        BaiqiFinanceDetailEntity merchantDetail = this
            .insertMerchantDetailRecharge(param.getMerchantFinanceId(), param.getMerchantId(), param.getOperator(),
                tradeNo, amountBean);

        LOGGER.info("[商家收入流水] {}", merchantDetail);

        // 代理商账户总金额减少，
        amountBean.setOpType(CashRebateAmountEnum.EXPENDITURE);
        boolean isFinanceAmount = financeService.updateFinanceAmountById(agentDetail.getFinanceId(), amountBean);

        LOGGER.info("[代理商账户总金额减少] {}", isFinanceAmount);

        // 调用推啊
        PartnerAllocateAmountBean partnerAllocateAmountBean = new PartnerAllocateAmountBean();
        partnerAllocateAmountBean.setBaiqiTradeNo(agentDetail.getFinanceTradeNo());
        partnerAllocateAmountBean.setTradeType(TuiaTradeTypeEnum.RECHARGE);
        partnerAllocateAmountBean.setAccountId(param.getAdvertId());
        partnerAllocateAmountBean.setAmount(param.getAmount());

        PartnerResultStateEnum resultStateEnum = this.callPartnerAllocateAmount(partnerAllocateAmountBean);

        LOGGER.info("[调用金额划拨] {}", resultStateEnum);

        // 调用推啊接口成功
        if (PartnerResultStateEnum.SUCCEED == resultStateEnum) {
            LOGGER.info("[金额划拨]成功, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
            this.callPartnerRechargeAsSucceed(agentDetail, merchantDetail);
            return resultStateEnum;
        }

        // 调用推啊接口失败
        if (PartnerResultStateEnum.FAILED == resultStateEnum) {
            LOGGER.warn("[金额划拨]失败 BaiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
            this.callPartnerRechargeAsFailed(agentDetail, merchantDetail);
            return resultStateEnum;
        }

        // 调用推啊接口异常
        LOGGER.error("[金额划拨]错误, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
        this.callPartnerRechargeAsException(agentDetail, merchantDetail);

        return resultStateEnum;
    }

    /**
     * 调用外部接口返回成功（划拨）
     *
     * @param agent    代理商流水明细
     * @param merchant 商家流水明细
     */
    private Boolean callPartnerRechargeAsSucceed(BaiqiFinanceDetailEntity agent, BaiqiFinanceDetailEntity merchant)
        throws BizException {

        // 检查参数
        this.checkCallPartnerRecharge(agent);

        Assert.notNull(agent.getFinanceId(), "账户ID为空");
        Assert.notNull(agent.getRelationId(), "流水关联ID为空");
        Assert.notNull(agent.getExpenditureTotal(), "流水支出总金额为空");
        Assert.notNull(agent.getExpenditureCash(), "流水支出现金为空");
        Assert.notNull(agent.getExpenditureRebate(), "流水支出返点为空");
        Assert.notNull(agent.getOperator(), "流水操作人为空");
        Assert.notNull(agent.getFinanceTradeNo(), "流水交易号为空");

        CashRebateAmountBean amountBean = new CashRebateAmountBean();
        amountBean.setTotalAmount(agent.getExpenditureTotal());
        amountBean.setCashAmount(agent.getExpenditureCash());
        amountBean.setRebateAmount(agent.getExpenditureRebate());

        Long merchantId = merchant.getRelationId();
        Long merchantFinanceId = merchant.getFinanceId();

        // 1. 商家金额增加
        // 设置操作类型为商家存入
        amountBean.setOpType(CashRebateAmountEnum.INCOME);
        boolean isFinanceAmount = financeService.updateFinanceAmountById(merchantFinanceId, amountBean);
        LOGGER.info("[调用外部接口返回成功，商家金额增加] merchantId:{}, financeId:{}, {}", merchantId, merchantFinanceId,
            isFinanceAmount);

        // 2. 修改商家流水状态为完成
        merchant.setRemark("");
        merchant.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.error("[调用外部接口返回成功，修改商家流水状态为完成]执行错误. merchantId:{}, financeId:{}", merchantId, merchantFinanceId);
            throw new BizException("[调用外部接口返回成功]执行错误.");
        }

        // 3. 修改代理商流水状态为完成
        agent.setRemark("");
        agent.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.error("[调用外部接口返回成功，修改代理商流水状态为完成]执行成功. agentId:{}, financeId:{}", agent.getRelationId(),
                agent.getFinanceId());
            throw new BizException("[调用外部接口返回成功]执行错误.");
        }

        return Boolean.TRUE;
    }

    /**
     * 调用外部接口返回失败（划拨）
     *
     * @param agent    代理商流水明细
     * @param merchant 商家流水明细
     */
    private Boolean callPartnerRechargeAsFailed(BaiqiFinanceDetailEntity agent, BaiqiFinanceDetailEntity merchant)
        throws BizException {

        // 检查参数
        this.checkCallPartnerRecharge(agent);

        Assert.notNull(agent.getRelationId(), "流水关联ID为空");
        Assert.notNull(agent.getExpenditureTotal(), "流水支出总金额为空");
        Assert.notNull(agent.getExpenditureCash(), "流水支出现金为空");
        Assert.notNull(agent.getExpenditureRebate(), "流水支出返点为空");

        CashRebateAmountBean amountBean = new CashRebateAmountBean();

        // 设置操作类型为代理商存入
        amountBean.setOpType(CashRebateAmountEnum.INCOME);
        amountBean.setTotalAmount(agent.getExpenditureTotal());
        amountBean.setCashAmount(agent.getExpenditureCash());
        amountBean.setRebateAmount(agent.getExpenditureRebate());

        // 1. 代理商金额增加, 反还给代理商
        boolean isFinanceAmount = financeService.updateFinanceAmountById(agent.getFinanceId(), amountBean);

        LOGGER.info("[调用外部接口失败，金额反还给代理商] id:{}, {}", agent.getId(), isFinanceAmount);

        // 2. 修改代理商流水信息，状态为(8:失败)
        agent.setRemark("调用外部接口失败");
        agent.setState(FinanceDetailStatusEnum.FAILED.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.error("[调用外部接口失败，修改代理商流水信息]执行失败. id:{}", agent.getId());
            throw new BizException("[调用外部接口失败，修改代理商流水信息]执行错误.");
        }

        // 3. 修改商家流水信息，状态为(8:失败)
        merchant.setRemark("调用外部接口失败");
        merchant.setState(FinanceDetailStatusEnum.FAILED.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.info("[调用外部接口失败，修改商家流水信息]执行成功. id:{}", merchant.getId());
            throw new BizException("[调用外部接口失败]执行错误.");
        }

        return Boolean.TRUE;
    }

    /**
     * 调用外部接口超时或异常（划拨）
     *
     * @param agent    代理商流水明细
     * @param merchant 商家流水明细
     */
    private Boolean callPartnerRechargeAsException(BaiqiFinanceDetailEntity agent, BaiqiFinanceDetailEntity merchant)
        throws BizException {

        // 检查参数
        this.checkCallPartnerRecharge(agent);

        // 检查参数
        this.checkCallPartnerRecharge(merchant);

        // 1. 修改流水信息，状态为(9:异常)
        agent.setRemark("调用外部接口异常或超时");
        agent.setState(FinanceDetailStatusEnum.ERROR.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.error("[调用外部接口超时或异常，更新代理商流水信息]执行成功. id:{}", agent.getId());
            throw new BizException("[调用外部接口异常或超时]执行错误.");
        }

        // 2. 修改流水信息，状态为(9:异常)
        merchant.setRemark("调用外部接口异常或超时");
        merchant.setState(FinanceDetailStatusEnum.ERROR.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.info("[调用外部接口超时或异常，更新商家流水信息]执行成功. id:{}", merchant.getId());
            throw new BizException("[调用外部接口异常或超时]执行错误.");
        }

        return Boolean.TRUE;
    }

    private void checkCallPartnerRecharge(BaiqiFinanceDetailEntity detailEntity) {
        Assert.notNull(detailEntity, "流水参数为空");
        Assert.notNull(detailEntity.getId(), "流水ID为空");
        Assert.notNull(detailEntity.getRelationType(), "流水关联类型为空");
    }

    /**
     * 保存代理商流水（划拨）
     *
     * @param financeId          账户ID
     * @param agentId            代理商ID
     * @param operatorId         操作人ID
     * @param tradeNo            交易号
     * @param cashRebateAmountBo 代理商金额
     * @return
     * @throws BizException
     */
    private BaiqiFinanceDetailEntity insertAgentDetailRecharge(Long financeId, Long agentId, Long operatorId,
        String tradeNo, CashRebateAmountBean cashRebateAmountBo) throws BizException {

        BaiqiFinanceDetailEntity agentDetail = new BaiqiFinanceDetailEntity();

        agentDetail.setFinanceId(financeId);
        //关联类型为(2:代理商)
        agentDetail.setRelationType(FinanceTypeEnum.AGENT_ACCOUNT.getCode());
        agentDetail.setRelationId(agentId);

        // 流水号
        agentDetail.setSerialNo(
            BaiqiSerialUtil.getSerialNo(FinanceOptTypeEnum.AGENT_TRANSFER, FinanceTypeEnum.AGENT_ACCOUNT, agentId));

        agentDetail.setExpenditureTotal(cashRebateAmountBo.getTotalAmount());
        agentDetail.setExpenditureCash(cashRebateAmountBo.getCashAmount());
        agentDetail.setExpenditureRebate(cashRebateAmountBo.getRebateAmount());

        agentDetail.setOperator(operatorId);
        //操作类型为(4:代理商划拨)
        agentDetail.setOperatorType(FinanceOptTypeEnum.AGENT_TRANSFER.getNum());
        //状态为(5:处理中)
        agentDetail.setState(FinanceDetailStatusEnum.PROCESSING.getNum());

        //设置合作商信息
        agentDetail.setPartnerType(ParterTypeEnum.TUIA.getCode());
        agentDetail.setFinanceTradeNo(tradeNo);

        // 保存代理商流水
        if (0 < baiqiFinanceDetailMapper.insert(agentDetail)) {
            LOGGER.info("[保存代理商流水]执行成功. financeId:{}", agentDetail.getFinanceId());
            return agentDetail;
        }
        LOGGER.error("[保存代理商流水]执行错误. financeId:{}, {}", agentDetail.getFinanceId(), agentDetail);
        throw new BizException("[保存代理商流水]执行错误");
    }

    /**
     * 保存商家流水（划拨）
     *
     * @param financeId          账户ID
     * @param merchantId         商家id
     * @param operatorId         操作人ID
     * @param financeTradeNo     交易号
     * @param cashRebateAmountBo 商家金额
     * @return
     * @throws BizException
     */
    private BaiqiFinanceDetailEntity insertMerchantDetailRecharge(Long financeId, Long merchantId, Long operatorId,
        String financeTradeNo, CashRebateAmountBean cashRebateAmountBo) throws BizException {

        BaiqiFinanceDetailEntity merchantDetail = new BaiqiFinanceDetailEntity();
        merchantDetail.setFinanceId(financeId);

        //关联类型为(0:商家主账号)
        merchantDetail.setRelationType(FinanceTypeEnum.MAIN_ACCOUNT.getCode());
        merchantDetail.setRelationId(merchantId);

        // 流水号
        merchantDetail.setSerialNo(
            BaiqiSerialUtil.getSerialNo(FinanceOptTypeEnum.AGENT_TRANSFER, FinanceTypeEnum.MAIN_ACCOUNT, merchantId));

        merchantDetail.setIncomeTotal(cashRebateAmountBo.getTotalAmount());
        merchantDetail.setIncomeCash(cashRebateAmountBo.getCashAmount());
        merchantDetail.setIncomeRebate(cashRebateAmountBo.getRebateAmount());

        merchantDetail.setOperator(operatorId);
        //操作类型为(4:代理商划拨)
        merchantDetail.setOperatorType(FinanceOptTypeEnum.AGENT_TRANSFER.getNum());
        //状态为(1:完成)
        merchantDetail.setState(FinanceDetailStatusEnum.PROCESSING.getNum());

        //设置合作商信息
        merchantDetail.setPartnerType(ParterTypeEnum.TUIA.getCode());
        merchantDetail.setFinanceTradeNo(financeTradeNo);

        // 保存商家流水
        if (0 < baiqiFinanceDetailMapper.insert(merchantDetail)) {
            LOGGER.info("[保存商家流水错误]执行成功. financeId:{}", merchantDetail.getFinanceId());
            return merchantDetail;
        }

        LOGGER.error("[保存商家流水错误]执行错误. financeId:{}, {}", merchantDetail.getFinanceId(), merchantDetail);
        throw new BizException("[保存商家流水错误]执行错误.");
    }

    @Override
    public void retryCallPartner() {

        LOGGER.info("定时任务重试调用外部接口");

        // 查询状态为:9的流水
        List<BaiqiFinanceDetailEntity> detailEntityList = baiqiFinanceDetailMapper
            .selectByState(FinanceDetailStatusEnum.ERROR.getNum());

        if (CollectionUtils.isEmpty(detailEntityList)) {
            return;
        }

        LOGGER.info("重试调用外部接口, {}", detailEntityList);

        for (BaiqiFinanceDetailEntity detailEntity : detailEntityList) {
            try {

                // 代理商划拨
                if (FinanceTypeEnum.AGENT_ACCOUNT.getCode() == detailEntity.getRelationType()
                    && FinanceOptTypeEnum.AGENT_TRANSFER.getNum().equals(detailEntity.getOperatorType())) {
                    this.doRetryCallPartnerRecharge(detailEntity);
                }
                // 代理商回拨
                else if (FinanceTypeEnum.AGENT_ACCOUNT.getCode() == detailEntity.getRelationType()
                    && FinanceOptTypeEnum.AGENT_CALLBACK.getNum().equals(detailEntity.getOperatorType())) {
                    this.doRetryCallPartnerReverse(detailEntity);
                }
                //商家人工充值
                else if (FinanceOptTypeEnum.OFFLINE_RECHARGE.getNum().equals(detailEntity.getOperatorType())
                    && FinanceTypeEnum.MAIN_ACCOUNT.getCode() == detailEntity.getRelationType()) {
                    this.doRetryCallMerchantArtificialRecharge(detailEntity);
                }
            } catch (Exception e) {
                LOGGER.error("重试调用合作商接口错误，id:{}", detailEntity.getId(), e);
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void doRetryCallMerchantArtificialRecharge(BaiqiFinanceDetailEntity detailEntity) {
        LOGGER.info("检查商家人工充值, baiqiTradeNo:{}, {}", detailEntity.getFinanceTradeNo(), detailEntity);

        PartnerResultStateEnum resultStateEnum = checkPartnerAllocateAmount(detailEntity.getFinanceTradeNo());

        // 调用推啊接口成功
        if (PartnerResultStateEnum.SUCCEED == resultStateEnum) {
            LOGGER.info("[检查商家人工充值]成功, baiqiTradeNo:{}, {}", detailEntity.getFinanceTradeNo(), detailEntity);
            this.artificialRechargeToSuccess(detailEntity);
            return;
        }

        // 调用推啊接口失败
        if (PartnerResultStateEnum.FAILED == resultStateEnum) {
            LOGGER.warn("[检查商家人工充值]失败, baiqiTradeNo:{}, {}", detailEntity.getFinanceTradeNo(), detailEntity);
            this.artificialRechargeToFailed(detailEntity);
            return;
        }

        // 异常后超时后
        if (System.currentTimeMillis()
            >= detailEntity.getGmtCreate().getTime() + AllocateAmountConstant.CALL_PARTNER_ALLOCATE_AMOUNT_TRY_TIME) {
            // 异常后，按失败处理
            LOGGER.info("[检查商家人工充值]错误, 超过一天按失败处理, baiqiTradeNo:{}, {}", detailEntity.getFinanceTradeNo(), detailEntity);
            this.artificialRechargeToFailed(detailEntity);
        } else {
            LOGGER.error("[检查商家人工充值]错误, baiqiTradeNo:{}, {}", detailEntity.getFinanceTradeNo(), detailEntity);
        }
    }

    private void artificialRechargeToFailed(BaiqiFinanceDetailEntity detailEntity) {
        //检查参数
        this.checkCallArtificialRecharge(detailEntity);
        //更新流水状态为失败
        BaiqiFinanceDetailEntity financeDetailEntity = new BaiqiFinanceDetailEntity();
        financeDetailEntity.setState(FinanceDetailStatusEnum.FAILED.getNum());
        financeDetailEntity.setId(detailEntity.getId());
        baiqiFinanceDetailMapper.updateStateById(financeDetailEntity);
        //更新主订单状态为待财务主管审批
        baiqiFinanceRechargeMapper.updateState(BaiqiFinanceRechargeStateEnum.WAIT_CHARGE.getState(),
            Long.parseLong(detailEntity.getFinanceTradeNo()), "系统");

    }

    private void artificialRechargeToSuccess(BaiqiFinanceDetailEntity detailEntity) {
        //检查参数
        this.checkCallArtificialRecharge(detailEntity);
        //充值总金额
        Long incomeTotalAmout = detailEntity.getIncomeTotal();
        //财务主表增加金额
        BaiqiFinanceEntity financeEntity = baiqiFinanceMapper
            .findByRelation(detailEntity.getRelationType(), detailEntity.getRelationId());
        financeEntity.setCashAmount(detailEntity.getIncomeCash() == null ?
            financeEntity.getCashAmount() :
            detailEntity.getIncomeCash() + financeEntity.getCashAmount());
        financeEntity.setRebateAmount(detailEntity.getIncomeRebate() == null ?
            financeEntity.getRebateAmount() :
            detailEntity.getIncomeRebate() + financeEntity.getRebateAmount());
        financeEntity.setTotalAmount(financeEntity.getTotalAmount() + incomeTotalAmout);
        baiqiFinanceMapper.updateAmount(financeEntity);

        //更新流水状态
        BaiqiFinanceDetailEntity financeDetailEntity = new BaiqiFinanceDetailEntity();
        financeDetailEntity.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
        financeDetailEntity.setId(detailEntity.getId());
        baiqiFinanceDetailMapper.updateStateById(financeDetailEntity);
        //更新充值订单状态
        baiqiFinanceRechargeMapper.updateState(BaiqiFinanceRechargeStateEnum.SUCCEED.getState(),
            Long.parseLong(detailEntity.getFinanceTradeNo()), "系统");
    }

    private void checkCallArtificialRecharge(BaiqiFinanceDetailEntity detailEntity) {
        Assert.notNull(detailEntity, "流水参数为空");
        Assert.notNull(detailEntity.getId(), "流水ID为空");
        Assert.notNull(detailEntity.getRelationType(), "流水关联类型为空");
    }

    @Transactional(rollbackFor = Exception.class)
    public void doRetryCallPartnerRecharge(BaiqiFinanceDetailEntity agentEntity) throws BizException {
        LOGGER.info("检查金额划拨, baiqiTradeNo:{}, {}", agentEntity.getFinanceTradeNo(), agentEntity);

        BaiqiFinanceDetailEntity merchantDetail = financeDetailService
            .findFinanceDetailByTradeNo(FinanceTypeEnum.MAIN_ACCOUNT.getCode(), agentEntity.getFinanceTradeNo());

        if (null == merchantDetail) {
            return;
        }

        PartnerResultStateEnum resultStateEnum = checkPartnerAllocateAmount(agentEntity.getFinanceTradeNo());

        // 调用推啊接口成功
        if (PartnerResultStateEnum.SUCCEED == resultStateEnum) {
            LOGGER.info("[检查金额划拨]成功, baiqiTradeNo:{}, {}", agentEntity.getFinanceTradeNo(), agentEntity);
            this.callPartnerRechargeAsSucceed(agentEntity, merchantDetail);
            return;
        }

        // 调用推啊接口失败
        if (PartnerResultStateEnum.FAILED == resultStateEnum) {
            LOGGER.warn("[检查金额划拨]失败, baiqiTradeNo:{}, {}", agentEntity.getFinanceTradeNo(), agentEntity);
            this.callPartnerRechargeAsFailed(agentEntity, merchantDetail);
            return;
        }

        // 异常后超时后
        if (System.currentTimeMillis()
            >= agentEntity.getGmtCreate().getTime() + AllocateAmountConstant.CALL_PARTNER_ALLOCATE_AMOUNT_TRY_TIME) {
            // 异常后，按失败处理
            LOGGER.info("[检查金额划拨]错误, 超过一天按失败处理, baiqiTradeNo:{}, {}", agentEntity.getFinanceTradeNo(), agentEntity);
            this.callPartnerRechargeAsFailed(agentEntity, merchantDetail);
        } else {
            LOGGER.error("[检查金额划拨]错误, baiqiTradeNo:{}, {}", agentEntity.getFinanceTradeNo(), agentEntity);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void doRetryCallPartnerReverse(BaiqiFinanceDetailEntity agentDetail) throws BizException {
        LOGGER.info("检查金额回拨, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);

        BaiqiFinanceDetailEntity metchantDetail = financeDetailService
            .findFinanceDetailByTradeNo(FinanceTypeEnum.MAIN_ACCOUNT.getCode(), agentDetail.getFinanceTradeNo());

        if (null == metchantDetail) {
            return;
        }

        PartnerResultStateEnum resultStateEnum = checkPartnerAllocateAmount(agentDetail.getFinanceTradeNo());

        // 调用推啊接口成功
        if (PartnerResultStateEnum.SUCCEED == resultStateEnum) {
            LOGGER.info("[检查金额回拨]成功, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
            this.callPartnerReverseAsSucceed(metchantDetail, agentDetail);
            return;
        }

        // 调用推啊接口失败
        if (PartnerResultStateEnum.FAILED == resultStateEnum) {
            LOGGER.warn("[检查金额回拨]失败, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
            this.callPartnerReverseAsFailed(metchantDetail, agentDetail);
            return;
        }

        // 异常后超时后
        if (System.currentTimeMillis()
            >= agentDetail.getGmtCreate().getTime() + AllocateAmountConstant.CALL_PARTNER_ALLOCATE_AMOUNT_TRY_TIME) {
            // 异常后，按失败处理
            LOGGER.info("[检查金额回拨]错误, 超过一天按失败处理, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
            this.callPartnerReverseAsFailed(metchantDetail, agentDetail);
        } else {
            LOGGER.error("[检查金额回拨]错误, baiqiTradeNo:{}, {}", agentDetail.getFinanceTradeNo(), agentDetail);
        }
    }

    //

    @Override
    public PartnerResultStateEnum allocateAmountReverse(AllocateAmountDto param) throws BizException {

        Assert.notNull(param, "参数为空");
        Assert.notNull(param.getAgentId(), "代理商账号ID为空");
        Assert.notNull(param.getAgentFinanceId(), "代理商账户ID为空");
        Assert.notNull(param.getMerchantId(), "商家账号ID为空");
        Assert.notNull(param.getMerchantFinanceId(), "商家账户ID为空");
        Assert.notNull(param.getAdvertId(), "广告主ID为空");
        Assert.notNull(param.getAmount(), "充值金额为空");
        Assert.notNull(param.getOperator(), "操作人为空");

        LOGGER.info("[金额回拨参数] {}", param);

        // 计算代理商现金和返点的支出金额
        CashRebateAmountBean amountBean = financeService
            .calcCashAndRebateAmount(param.getMerchantFinanceId(), param.getAmount(), CashRebateAmountEnum.EXPENDITURE);

        if (null == amountBean) {
            LOGGER.error("计算代理商的现金和返点扣比为空");
            return PartnerResultStateEnum.FAILED;
        }

        LOGGER.info("[金额回拨分配比例] {}", amountBean);

        String tradeNo = UUIDUtils.createUUID();

        // 1. 添加代理商收入流水，关联类型为(2:代理商)，操作类型为(5:代理商回拨)，状态为(1)
        BaiqiFinanceDetailEntity agentDetail = this
            .insertAgentDetailReverse(param.getAgentFinanceId(), param.getAgentId(), param.getOperator(), tradeNo,
                amountBean);
        LOGGER
            .info("[调用外部接口返回成功，添加代理商收入流水] agentId:{}, financeId:{}, {}", param.getAgentId(), param.getAgentFinanceId(),
                agentDetail);

        // 保存商家支出流水
        BaiqiFinanceDetailEntity merchantDetail = this
            .insertMerchantDetailReverse(param.getMerchantFinanceId(), param.getMerchantId(), param.getOperator(),
                tradeNo, amountBean);
        LOGGER.info("[回拨商家流水] {}", merchantDetail);

        // 更新商家金额（支出）
        amountBean.setOpType(CashRebateAmountEnum.EXPENDITURE);
        boolean isFinanceAmount = financeService.updateFinanceAmountById(merchantDetail.getFinanceId(), amountBean);
        LOGGER.info("[代理商账户总金额减少] {}", isFinanceAmount);

        // 调用推啊
        PartnerAllocateAmountBean partnerAllocateAmountBean = new PartnerAllocateAmountBean();
        partnerAllocateAmountBean.setBaiqiTradeNo(merchantDetail.getFinanceTradeNo());
        partnerAllocateAmountBean.setTradeType(TuiaTradeTypeEnum.REVERSE);
        partnerAllocateAmountBean.setAccountId(param.getAdvertId());
        partnerAllocateAmountBean.setAmount(param.getAmount());

        PartnerResultStateEnum resultStateEnum = this.callPartnerAllocateAmount(partnerAllocateAmountBean);

        LOGGER.info("[调用金额回拨] {}", resultStateEnum);

        // 调用推啊接口成功
        if (PartnerResultStateEnum.SUCCEED == resultStateEnum) {
            LOGGER.info("[金额回拨]成功, baiqiTradeNo:{}, {}", merchantDetail.getFinanceTradeNo(), merchantDetail);
            this.callPartnerReverseAsSucceed(merchantDetail, agentDetail);
            return resultStateEnum;
        }

        // 调用推啊接口失败
        if (PartnerResultStateEnum.FAILED == resultStateEnum) {
            LOGGER.warn("[金额回拨]失败 BaiqiTradeNo:{}, {}", merchantDetail.getFinanceTradeNo(), merchantDetail);
            this.callPartnerReverseAsFailed(merchantDetail, agentDetail);
            return resultStateEnum;
        }

        // 调用推啊接口异常
        LOGGER.error("[金额回拨]错误, baiqiTradeNo:{}, {}", merchantDetail.getFinanceTradeNo(), merchantDetail);
        this.callPartnerReverseAsException(merchantDetail, agentDetail);

        return resultStateEnum;
    }

    private void checkCallPartnerReverse(BaiqiFinanceDetailEntity detailEntity) {
        Assert.notNull(detailEntity, "流水参数为空");
        Assert.notNull(detailEntity.getId(), "流水ID为空");
        Assert.notNull(detailEntity.getRelationType(), "流水关联类型为空");
    }

    /**
     * 调用外部接口返回失败（回拨）
     *
     * @param merchant 商家流水明细
     * @param agent    代理商流水明细
     */
    private Boolean callPartnerReverseAsFailed(BaiqiFinanceDetailEntity merchant, BaiqiFinanceDetailEntity agent)
        throws BizException {

        // 检查参数
        this.checkCallPartnerReverse(merchant);

        Assert.notNull(merchant.getRelationId(), "流水关联ID为空");
        Assert.notNull(merchant.getExpenditureTotal(), "流水支出总金额为空");
        Assert.notNull(merchant.getExpenditureCash(), "流水支出现金为空");
        Assert.notNull(merchant.getExpenditureRebate(), "流水支出返点为空");

        CashRebateAmountBean amountBean = new CashRebateAmountBean();

        // 设置操作类型为代理商存入
        amountBean.setOpType(CashRebateAmountEnum.INCOME);
        amountBean.setTotalAmount(merchant.getExpenditureTotal());
        amountBean.setCashAmount(merchant.getExpenditureCash());
        amountBean.setRebateAmount(merchant.getExpenditureRebate());

        // 1. 商家金额增加, 反还给商家
        boolean isFinanceAmount = financeService.updateFinanceAmountById(merchant.getFinanceId(), amountBean);

        LOGGER.info("[调用外部接口失败，金额反还给商家] id:{}, {}", merchant.getId(), isFinanceAmount);

        // 2. 修改流水信息，状态为(8:失败)
        merchant.setRemark("调用外部接口失败");
        merchant.setState(FinanceDetailStatusEnum.FAILED.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.error("[调用外部接口失败，更新商家流水]执行错误. id:{}", merchant.getId());
            throw new BizException("[调用外部接口失败]执行错误.");
        }

        agent.setRemark("调用外部接口失败");
        agent.setState(FinanceDetailStatusEnum.FAILED.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.info("[调用外部接口失败，更新代理商流水]执行错误。id:{}", merchant.getId());
            throw new BizException("[调用外部接口失败]执行错误.");
        }

        return Boolean.TRUE;
    }

    /**
     * 调用外部接口超时或异常（回拨）
     *
     * @param merchant 商家流水明细
     * @param agent    代理商流水明细
     */
    private Boolean callPartnerReverseAsException(BaiqiFinanceDetailEntity merchant, BaiqiFinanceDetailEntity agent)
        throws BizException {

        // 检查参数
        this.checkCallPartnerReverse(merchant);

        // 检查参数
        this.checkCallPartnerReverse(agent);

        // 1. 修改商家流水信息，状态为(9:异常)
        merchant.setRemark("调用外部接口异常或超时");
        merchant.setState(FinanceDetailStatusEnum.ERROR.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.error("[调用外部接口超时或异常，更新商家流水信息]执行成功. id:{}", merchant.getId());
            throw new BizException("[调用外部接口异常或超时]执行错误.");
        }

        // 2. 修改代理商流水信息，状态为(9:异常)
        agent.setRemark("调用外部接口异常或超时");
        agent.setState(FinanceDetailStatusEnum.ERROR.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.error("[调用外部接口超时或异常，更新代理商流水信息]执行成功. id:{}", agent.getId());
            throw new BizException("[调用外部接口异常或超时]执行错误.");
        }

        return Boolean.TRUE;
    }

    /**
     * 调用外部接口返回成功（回拨）
     *
     * @param merchant 商家流水明细
     * @param agent    代理商流水明细
     */
    private Boolean callPartnerReverseAsSucceed(BaiqiFinanceDetailEntity merchant, BaiqiFinanceDetailEntity agent)
        throws BizException {

        // 检查参数
        this.checkCallPartnerReverse(merchant);

        Assert.notNull(merchant.getFinanceId(), "账户ID为空");
        Assert.notNull(merchant.getRelationId(), "流水关联ID为空");
        Assert.notNull(merchant.getExpenditureTotal(), "流水支出总金额为空");
        Assert.notNull(merchant.getExpenditureCash(), "流水支出现金为空");
        Assert.notNull(merchant.getExpenditureRebate(), "流水支出返点为空");
        Assert.notNull(merchant.getOperator(), "流水操作人为空");
        Assert.notNull(merchant.getFinanceTradeNo(), "流水交易号为空");

        CashRebateAmountBean amountBean = new CashRebateAmountBean();
        amountBean.setTotalAmount(merchant.getExpenditureTotal());
        amountBean.setCashAmount(merchant.getExpenditureCash());
        amountBean.setRebateAmount(merchant.getExpenditureRebate());

        // 1. 代理商金额增加
        // 设置操作类型为代理商存入
        amountBean.setOpType(CashRebateAmountEnum.INCOME);
        boolean isFinanceAmount = financeService.updateFinanceAmountById(agent.getFinanceId(), amountBean);
        LOGGER.info("[调用外部接口返回成功，代理商金额增加] agentId:{}, financeId:{}, {}", agent.getRelationId(), agent.getFinanceId(),
            isFinanceAmount);

        // 2. 修改代理商流水状态为完成
        agent.setRemark("");
        agent.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(agent)) {
            LOGGER.error("[调用外部接口返回成功，修改代理商流水状态为完成]执行成功. agentId:{}, financeId:{}", agent.getRelationId(),
                agent.getFinanceId());
            throw new BizException("[调用外部接口返回成功]执行错误.");
        }

        // 3. 修改商家流水状态为完成
        merchant.setRemark("");
        merchant.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
        if (0 == baiqiFinanceDetailMapper.updateStateById(merchant)) {
            LOGGER.error("[调用外部接口返回成功，修改商家流水状态为完成]执行成功. merchantId:{}, financeId:{}", merchant.getRelationId(),
                merchant.getFinanceId());
            throw new BizException("[调用外部接口返回成功]执行错误.");

        }

        return Boolean.TRUE;
    }

    /**
     * 保存商家流水（回拨）
     *
     * @param financeId
     * @param merchantId
     * @param operatorId
     * @param cashRebateAmountBo
     * @return
     * @throws BizException
     */
    private BaiqiFinanceDetailEntity insertMerchantDetailReverse(Long financeId, Long merchantId, Long operatorId,
        String tradeNo, CashRebateAmountBean cashRebateAmountBo) throws BizException {

        BaiqiFinanceDetailEntity merchantDetail = new BaiqiFinanceDetailEntity();
        merchantDetail.setFinanceId(financeId);

        //关联类型为(1:商家主账号)
        merchantDetail.setRelationType(FinanceTypeEnum.MAIN_ACCOUNT.getCode());
        merchantDetail.setRelationId(merchantId);

        // 流水号
        merchantDetail.setSerialNo(
            BaiqiSerialUtil.getSerialNo(FinanceOptTypeEnum.AGENT_CALLBACK, FinanceTypeEnum.MAIN_ACCOUNT, merchantId));

        merchantDetail.setExpenditureTotal(cashRebateAmountBo.getTotalAmount());
        merchantDetail.setExpenditureCash(cashRebateAmountBo.getCashAmount());
        merchantDetail.setExpenditureRebate(cashRebateAmountBo.getRebateAmount());

        merchantDetail.setOperator(operatorId);
        //操作类型为(5:代理商回拨)
        merchantDetail.setOperatorType(FinanceOptTypeEnum.AGENT_CALLBACK.getNum());
        //状态为(1:处理中)
        merchantDetail.setState(FinanceDetailStatusEnum.PROCESSING.getNum());

        //设置合作商信息
        merchantDetail.setPartnerType(ParterTypeEnum.TUIA.getCode());
        merchantDetail.setFinanceTradeNo(tradeNo);

        // 保存商家流水
        if (0 < baiqiFinanceDetailMapper.insert(merchantDetail)) {
            LOGGER.info("[保存商家流水错误]执行成功. financeId:{}", merchantDetail.getFinanceId());
            return merchantDetail;
        }

        LOGGER.error("[保存商家流水错误]执行错误. financeId:{}, {}", merchantDetail.getFinanceId(), merchantDetail);
        throw new BizException("[保存商家流水错误]执行错误.");
    }

    /**
     * 保存代理商流水（回拨）
     *
     * @param financeId
     * @param agentId
     * @param operatorId
     * @param financeTradeNo
     * @param cashRebateAmountBo
     * @return
     * @throws BizException
     */
    private BaiqiFinanceDetailEntity insertAgentDetailReverse(Long financeId, Long agentId, Long operatorId,
        String financeTradeNo, CashRebateAmountBean cashRebateAmountBo) throws BizException {

        BaiqiFinanceDetailEntity agentDetail = new BaiqiFinanceDetailEntity();

        agentDetail.setFinanceId(financeId);
        //关联类型为(2:代理商)
        agentDetail.setRelationType(FinanceTypeEnum.AGENT_ACCOUNT.getCode());
        agentDetail.setRelationId(agentId);

        // 流水号
        agentDetail.setSerialNo(
            BaiqiSerialUtil.getSerialNo(FinanceOptTypeEnum.AGENT_CALLBACK, FinanceTypeEnum.AGENT_ACCOUNT, agentId));

        agentDetail.setIncomeTotal(cashRebateAmountBo.getTotalAmount());
        agentDetail.setIncomeCash(cashRebateAmountBo.getCashAmount());
        agentDetail.setIncomeRebate(cashRebateAmountBo.getRebateAmount());

        agentDetail.setOperator(operatorId);
        //操作类型为(5:代理商回拨)
        agentDetail.setOperatorType(FinanceOptTypeEnum.AGENT_CALLBACK.getNum());
        //状态为(10:完成)
        agentDetail.setState(FinanceDetailStatusEnum.PROCESSING.getNum());

        //设置合作商信息
        agentDetail.setPartnerType(ParterTypeEnum.TUIA.getCode());
        agentDetail.setFinanceTradeNo(financeTradeNo);

        // 保存代理商流水
        if (0 < baiqiFinanceDetailMapper.insert(agentDetail)) {
            LOGGER.info("[保存代理商流水]执行成功. financeId:{}", agentDetail.getFinanceId());
            return agentDetail;
        }
        LOGGER.error("[保存代理商流水]执行错误. financeId:{}, {}", agentDetail.getFinanceId(), agentDetail);
        throw new BizException("[保存代理商流水]执行错误");
    }

}
