package com.qiho.center.biz.service.impl.finance;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.finance.FinanceDto;
import com.qiho.center.api.dto.finance.HistoricalBalanceDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.enums.account.BaiqiAccountTypeEnum;
import com.qiho.center.api.enums.finance.CashRebateAmountEnum;
import com.qiho.center.api.enums.finance.FinanceStatusEnum;
import com.qiho.center.api.enums.finance.FinanceTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.PageQueryParams;
import com.qiho.center.api.params.finance.BaiqiMerchantFreetextParam;
import com.qiho.center.api.params.finance.FinanceBasePageParam;
import com.qiho.center.biz.service.agent.BaiqiAgentService;
import com.qiho.center.biz.service.finance.FinanceService;
import com.qiho.center.biz.service.impl.finance.bean.CashRebateAmountBean;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.dao.QihoMerchantDAO;
import com.qiho.center.common.daoh.qiho.agent.BaiqiAgentMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiHistoricalBalanceMapper;
import com.qiho.center.common.entity.merchant.QihoMerchantEntity;
import com.qiho.center.common.entityd.qiho.agent.BaiqiAgentEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiHistoricalBalanceEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

/**
 * FinanceServiceImpl
 *
 * @author zhangshun
 * @version V1.0
 * @since 2017-12-27 20:15
 */
@Service
public class FinanceServiceImpl implements FinanceService {

    @Resource
    private BaiqiFinanceMapper baiqiFinanceMapper;
    @Resource
    private MerchantService merchantService;
    @Resource
    private BaiqiAgentService baiqiAgentService;
    @Resource
    private BaiqiHistoricalBalanceMapper historicalBalanceMapper;

    @Autowired
    private QihoMerchantDAO qihoMerchantDAO;

    @Autowired
    private BaiqiAgentMapper baiqiAgentMapper;

    @Override
    public Boolean insertFinance(FinanceDto dto) {
        if (dto.getRelationType() == null || dto.getRelationId() == null) {
            throw new QihoException("资金账户未指定商家或代理商");
        }
        BaiqiFinanceEntity entity = new BaiqiFinanceEntity();
        entity.setRelationType(dto.getRelationType().getCode());
        return baiqiFinanceMapper.insertAccountFinance(entity) > 0;
    }

    @Override
    public PagenationDto<FinanceDto> findAll(FinanceBasePageParam pageParam) {
        PagenationDto<FinanceDto> pagenationDto = new PagenationDto<>();
        // update by chensong on 2018/03/29
        // 这里根据参数路由 如果有merchantName 或 agentName作为查询条件 则先查出对应的商家id或代理商id
        if (StringUtils.isNotEmpty(pageParam.getMerchantName())) { //这里有空格也要查询  真恶心。。。。。。
            List<QihoMerchantEntity> merchantEntityList = qihoMerchantDAO.listByMerchantName(pageParam.getMerchantName());
            if (CollectionUtils.isEmpty(merchantEntityList)) {
                // 如果没有查出数据  直接返回
                return pagenationDto.emptyPage();
            }
            pageParam.setMerchantIdList(
                    merchantEntityList.stream().map(QihoMerchantEntity::getId).collect(Collectors.toList())
            );
        }
        if (StringUtils.isNotEmpty(pageParam.getAgentName())) {
            List<BaiqiAgentEntity> baiqiAgentEntityList = baiqiAgentMapper.listByAgentName(pageParam.getAgentName());
            if (CollectionUtils.isEmpty(baiqiAgentEntityList)) {
                return pagenationDto.emptyPage();
            }
            pageParam.setAgentIdList(
                    baiqiAgentEntityList.stream().map(BaiqiAgentEntity::getId).collect(Collectors.toList())
            );
        }

        Integer count = baiqiFinanceMapper.countPageMainAccount(pageParam);
        pagenationDto.setTotal(count);
        if (count.intValue() == 0) {
            return pagenationDto.emptyPage();
        }
        List<BaiqiFinanceEntity> entities = baiqiFinanceMapper.findAll(pageParam);

        pagenationDto.setList(convertDtoList(entities));
        return pagenationDto;
    }

    /**
     * 类型转换
     * @param entities
     * @return
     */
    private List<FinanceDto> convertDtoList(List<BaiqiFinanceEntity> entities){
        if (CollectionUtils.isEmpty(entities)) {
            return Lists.newArrayList();
        }
        List<Long> merchantIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();
        setRelationIds(merchantIds, agentIds, entities);
        Map<Long, MerchantDto> merchantDtoMap = new HashMap<>();
        Map<Long, BaiqiAgentEntity> agentMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(merchantIds)) {
            merchantDtoMap = merchantService.findByIdList(merchantIds);
        }
        if (CollectionUtils.isNotEmpty(agentIds)) {
            agentMap = baiqiAgentService.findByIdsMap(agentIds);
        }
        List<FinanceDto> financeDtos = new ArrayList<>();
        for (BaiqiFinanceEntity entity : entities) {
            Long relationId = entity.getRelationId();
            int relationType = entity.getRelationType();
            FinanceDto accountDto = entityToDto(entity);
            if (relationType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                //设置代理商名称
                BaiqiAgentEntity agentEntity = agentMap.get(relationId);
                accountDto.setName(agentEntity == null ? "" : agentEntity.getAgentName());
            } else {
                MerchantDto merchantDto = merchantDtoMap.get(relationId);
                accountDto.setName(merchantDto == null ? "" : merchantDto.getMerchantName());
            }
            financeDtos.add(accountDto);
        }
        return financeDtos;
    }

    @Override
    public FinanceDto findFinanceByMerchantId(long merchantId) {
        BaiqiFinanceEntity entity = baiqiFinanceMapper.selectByMerchantId(merchantId);
        return BeanUtils.copy(entity, FinanceDto.class);
    }

    @Override
    public CashRebateAmountBean calcCashAndRebateAmount(Long financeId, Long amount, CashRebateAmountEnum opType) {

        Assert.notNull(financeId, "financeId为空");
        Assert.notNull(amount, "amount为空");

        BaiqiFinanceEntity financeEntity = baiqiFinanceMapper.findById(financeId);

        if (null == financeEntity) {
            return null;
        }

        CashRebateAmountBean amountBean = new CashRebateAmountBean();

        // 设置操作类型
        amountBean.setOpType(opType);

        // 如果账户金额为空0,全部扣现金
        if (0 == financeEntity.getTotalAmount()) {

            amountBean.setTotalAmount(Math.abs(amount));
            amountBean.setCashAmount(amountBean.getTotalAmount());
            amountBean.setRebateAmount(0L);

        } else {

            // 扣款金额
            BigDecimal amountTmp = new BigDecimal(Math.abs(amount));

            // 代理商总金额
            BigDecimal amountTotal = new BigDecimal(Math.abs(financeEntity.getTotalAmount()));
            // 代理商现金
            BigDecimal cashAmount = new BigDecimal(Math.abs(financeEntity.getCashAmount()));

            // 现金扣款 = (代理商现金余额 / 代理商总余额）* 充值金额
            BigDecimal costCashTmp = cashAmount.divide(amountTotal, 10, BigDecimal.ROUND_HALF_EVEN).multiply(amountTmp);

            // 现金扣款金额
            Long costCash = costCashTmp.setScale(0, RoundingMode.CEILING).longValue();

            amountBean.setTotalAmount(amountTmp.longValue());
            amountBean.setCashAmount(costCash);
            amountBean.setRebateAmount(amountTmp.longValue() - costCash);
        }

        return amountBean;
    }

    /**
     * 更新代理商金额
     *
     * @param financeId            账户ID
     * @param cashRebateAmountBean 代理商金额
     * @return
     * @throws cn.com.duiba.boot.exception.BizException
     */
    @Override
    public Boolean updateFinanceAmountById(Long financeId, CashRebateAmountBean cashRebateAmountBean)
        throws BizException {

        Assert.notNull(cashRebateAmountBean, "金额为空");
        Assert.notNull(cashRebateAmountBean.getTotalAmount(), "总金额为空");
        Assert.notNull(cashRebateAmountBean.getCashAmount(), "现金为空");
        Assert.notNull(cashRebateAmountBean.getRebateAmount(), "返点金额为空");
        Assert.notNull(cashRebateAmountBean.getOpType(), "金额操作类型为空");

        BaiqiFinanceEntity financeEntity = baiqiFinanceMapper.findById(financeId);

        Assert.notNull(financeEntity, "账户不存在. financeId:" + financeId);

        // 存入操作
        if (CashRebateAmountEnum.INCOME == cashRebateAmountBean.getOpType()) {
            financeEntity.setTotalAmount(financeEntity.getTotalAmount() + cashRebateAmountBean.getTotalAmount());
            financeEntity.setCashAmount(financeEntity.getCashAmount() + cashRebateAmountBean.getCashAmount());
            financeEntity.setRebateAmount(financeEntity.getRebateAmount() + cashRebateAmountBean.getRebateAmount());
        }
        // 支出操作
        else {
            financeEntity.setTotalAmount(financeEntity.getTotalAmount() - cashRebateAmountBean.getTotalAmount());
            financeEntity.setCashAmount(financeEntity.getCashAmount() - cashRebateAmountBean.getCashAmount());
            financeEntity.setRebateAmount(financeEntity.getRebateAmount() - cashRebateAmountBean.getRebateAmount());
        }

        if (1 == baiqiFinanceMapper.updateAmountById(financeEntity)) {
            return Boolean.TRUE;
        }

        throw new BizException("更新账户金额错误");
    }

    @Override
    public BaiqiFinanceEntity findByRelation(FinanceTypeEnum typeEnum, Long relationId) {
        Assert.notNull(relationId, "关联id为空");
        Assert.notNull(typeEnum, "关联类型空");
        return baiqiFinanceMapper.findByRelation(typeEnum.getCode(), relationId);
    }

    /*
     * 将entity 转换为 dto
     */
    private FinanceDto entityToDto(BaiqiFinanceEntity entity) {
        FinanceDto financeDto = BeanUtils.copy(entity, FinanceDto.class);
        financeDto.setRelationType(FinanceTypeEnum.getByCode(entity.getRelationType()));
        financeDto.setState(FinanceStatusEnum.getByCode(entity.getState()));
        return financeDto;
    }

    private void setRelationIds(List<Long> merchantIds, List<Long> agentIds, List<BaiqiFinanceEntity> entities) {
        for (BaiqiFinanceEntity entity : entities) {
            int accountType = entity.getRelationType();
            Long relationId = entity.getRelationId();
            if (accountType == BaiqiAccountTypeEnum.AGENT_ACCOUNT.getCode()) {
                agentIds.add(relationId);
            } else {
                merchantIds.add(relationId);
            }
        }
    }

    @Override
    public Integer findBaseInfoCount() {
        return baiqiFinanceMapper.findBaseInfoCount();
    }

    @Override
    public List<FinanceDto> findByParam(BaiqiMerchantFreetextParam pageParam) {
        List<BaiqiFinanceEntity> entities = baiqiFinanceMapper.findByParam(pageParam);
        List<FinanceDto> dtoList = entitiesToDtoList(entities);
        return dtoList;
    }

    private List<FinanceDto> entitiesToDtoList(List<BaiqiFinanceEntity> entities) {
        List<FinanceDto> dtoList = new ArrayList<>();
        for (BaiqiFinanceEntity entity : entities) {
            FinanceDto dto = new FinanceDto();
            dto.setRelationId(entity.getRelationId());
            dto.setRelationType(FinanceTypeEnum.getByCode(entity.getRelationType()));
            dto.setTotalAmount(entity.getTotalAmount());
            dto.setCashAmount(entity.getCashAmount());
            dto.setRebateAmount(entity.getRebateAmount());
            dto.setGmtCreate(entity.getGmtCreate());
            dto.setGmtModified(entity.getGmtModified());
            dtoList.add(dto);
        }
        return dtoList;
    }

    @Override
    public PagenationDto<HistoricalBalanceDto> findAllHistoricalBalance(PageQueryParams pageParam) {
        List<BaiqiHistoricalBalanceEntity> entities = historicalBalanceMapper.findAll(pageParam);
        PagenationDto<HistoricalBalanceDto> pagenationDto = new PagenationDto<>();
        if (CollectionUtils.isEmpty(entities)) {
            pagenationDto.setList(Collections.emptyList());
            pagenationDto.setTotal(0);
            return pagenationDto;
        }
        pagenationDto.setTotal(entities.size());
        List<Long> merchantIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();
        setRelationIdsWithBaiqiHistoricalBalanceEntity(merchantIds, agentIds, entities);
        Map<Long, MerchantDto> merchantDtoMap = new HashMap<>();
        Map<Long, BaiqiAgentEntity> agentMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(merchantIds)) {
            merchantDtoMap = merchantService.findByIdList(merchantIds);
        }
        if (CollectionUtils.isNotEmpty(agentIds)) {
            agentMap = baiqiAgentService.findByIdsMap(agentIds);
        }
        List<HistoricalBalanceDto> historicalBalanceDto = new ArrayList<>();
        for (BaiqiHistoricalBalanceEntity entity : entities) {
            Long relationId = entity.getRelationId();
            int relationType = entity.getRelationType();
            HistoricalBalanceDto accountDto = BeanUtils.copy(entity, HistoricalBalanceDto.class);
            accountDto.setRelationType(FinanceTypeEnum.getByCode(entity.getRelationType()));
            if (relationType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                //设置代理商名称
                BaiqiAgentEntity agentEntity = agentMap.get(relationId);
                accountDto.setName(agentEntity == null ? null : agentEntity.getAgentName());
            } else {
                MerchantDto merchantDto = merchantDtoMap.get(relationId);
                accountDto.setName(merchantDto == null ? null : merchantDto.getMerchantName());
            }
            historicalBalanceDto.add(accountDto);
        }
        pagenationDto.setList(historicalBalanceDto);
        pagenationDto.setTotal(historicalBalanceDto.size());
        return pagenationDto;
    }

    private void setRelationIdsWithBaiqiHistoricalBalanceEntity(List<Long> merchantIds, List<Long> agentIds, List<BaiqiHistoricalBalanceEntity> entities) {
        for (BaiqiHistoricalBalanceEntity entity : entities) {
            int accountType = entity.getRelationType();
            Long relationId = entity.getRelationId();
            if (accountType == BaiqiAccountTypeEnum.AGENT_ACCOUNT.getCode()) {
                agentIds.add(relationId);
            } else {
                merchantIds.add(relationId);
            }
        }
    }
}
