package com.qiho.center.biz.service.impl.order;

import java.util.Calendar;
import java.util.List;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.order.DeliveryRecordDto;
import com.qiho.center.api.enums.DeliveryRecordStateEnum;
import com.qiho.center.api.params.DeliveryRecordQueryParams;
import com.qiho.center.biz.job.OrderFileDeliveryJob;
import com.qiho.center.biz.service.order.DeliveryRecordService;
import com.qiho.center.common.daoh.qiho.order.DeliveryRecordMapper;
import com.qiho.center.common.entityd.qiho.order.DeliveryRecordEntity;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * @author jiafeng.geng
 * @create 2018-03-08 16:12:24
 **/
@Service
public class DeliveryRecordServiceImpl implements DeliveryRecordService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DeliveryRecordServiceImpl.class);

    @Autowired
    DeliveryRecordMapper deliveryRecordMapper;

    @Resource
    private ExecutorService executorService;

    @Resource
    private OrderFileDeliveryJob orderFileDeliveryJob;

    @Override
    public Boolean insertDeliveryRecord(DeliveryRecordDto dto) throws BizException {
        DeliveryRecordEntity entity = BeanUtils.copy(dto, DeliveryRecordEntity.class);
        try {

            if (0 == deliveryRecordMapper.insertDeliveryRecord(entity)) {
                return Boolean.FALSE;
            }

            // 异步调用 上传物流单号启动任务
            executorService.execute(new OrderFileDeliveryJobRunnable(orderFileDeliveryJob));

            return Boolean.TRUE;

        } catch (Exception e) {
            LOGGER.error("上传物流单号保存失败", e);
            throw new BizException("保存失败");
        }
    }

    @Override
    public PagenationDto<DeliveryRecordDto> deliveryRecordPageQuery(DeliveryRecordQueryParams params) {
        PagenationDto<DeliveryRecordDto> page = new PagenationDto<>();
        if (params.getOffset() == null || params.getMax() == null) {
            page.setTotal(0);
            return page;
        }
        Integer count = deliveryRecordMapper.countByQuery(params);
        page.setTotal(count);
        if (count.intValue() == 0) {
            return page;
        }

        List<DeliveryRecordDto> deliveryRecordDtoList = BeanUtils
            .copyList(deliveryRecordMapper.listByQuery(params), DeliveryRecordDto.class);

        for (DeliveryRecordDto dto : deliveryRecordDtoList) {
            if (dto.getState() == DeliveryRecordStateEnum.NEW.getCode()) {
                dto.setRemark("未开始");
            } else if (dto.getState() == DeliveryRecordStateEnum.RUNNING.getCode()) {
                dto.setRemark(
                    "正在处理，请稍候 (" + (dto.getSucceedCount() + dto.getFailedCount() + dto.getUpdatedCount()) + "/" + dto
                        .getTotalCount() + ")");
            } else if (dto.getState() == DeliveryRecordStateEnum.COMPLETE.getCode()) {
                String recordType = dto.getRecordType() == 1 ? "确认发货" : "取消发货";
                StringBuilder remark = new StringBuilder();
                int failedCount = dto.getFailedCount();
                int succeedCount = dto.getSucceedCount();
                int updatedCount = dto.getUpdatedCount();
                String failedList = dto.getRemark();
                remark.append(recordType).append("成功").append(succeedCount).append("个，更新物流").append(updatedCount)
                    .append("个，").append(recordType).append("失败").append(failedCount).append("个，请检查上传文件中")
                    .append(recordType).append("失败数据（数字表示行数）:").append(failedList);
                dto.setRemark(remark.toString());
            }
        }
        page.setList(deliveryRecordDtoList);
        return page;
    }

    @Override
    public int findTodayCount(Long merchantId) {
        Calendar startTime = Calendar.getInstance();
        startTime.set(Calendar.HOUR_OF_DAY, 0);
        startTime.set(Calendar.MINUTE, 0);
        startTime.set(Calendar.SECOND, 0);
        startTime.set(Calendar.MILLISECOND, 0);

        Calendar endTime = Calendar.getInstance();
        endTime.set(Calendar.HOUR_OF_DAY, 23);
        endTime.set(Calendar.MINUTE, 59);
        endTime.set(Calendar.SECOND, 59);
        endTime.set(Calendar.MILLISECOND, 999);

        return deliveryRecordMapper.findTodayCount(merchantId, startTime.getTime(), endTime.getTime());
    }

    static class OrderFileDeliveryJobRunnable implements Runnable {

        private final OrderFileDeliveryJob orderFileDeliveryJob;

        OrderFileDeliveryJobRunnable(OrderFileDeliveryJob orderFileDeliveryJob) {
            this.orderFileDeliveryJob = orderFileDeliveryJob;
        }

        @Override
        public void run() {
            try {
                orderFileDeliveryJob.doStart();
            } catch (Exception e) {
                LOGGER.warn("上传物流单号启动任务错误", e);
            }
        }
    }
}
