package com.qiho.center.biz.service.impl.order;

import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.dto.UpdateStockReqDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.biz.model.EdbItemStockResponse;
import com.qiho.center.biz.model.ErpStock;
import com.qiho.center.biz.service.impl.OrderStockService;
import com.qiho.center.common.daoh.qiho.QihoItemSkuMapper;
import com.qiho.center.common.daoh.qiho.QihoOrderItemInfoMapper;
import com.qiho.center.common.entityd.qiho.QihoItemSkuEntity;
import com.qiho.center.common.entityd.qiho.QihoOrderItemInfoEntity;
import com.qiho.center.common.enums.InnerLogTypeEnum;
import com.qiho.center.common.log.InnerLog;
import com.qiho.center.common.util.AppLogUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Created by weny.cai on 2017/8/15.
 */
@Service
public class OrderStockServiceImpl implements OrderStockService {

    private static final Logger logger = LoggerFactory.getLogger(OrderStockServiceImpl.class);
    @Autowired
    private QihoOrderItemInfoMapper qihoOrderItemInfoMapper;

    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    @Autowired
    QihoItemSkuMapper qihoItemSkuMapper;

    @Autowired
    private RemoteStockService remoteStockService;

    /**
     * 返还订单的库存
     *
     * @param orderId
     * @return
     */
    @Override
    public DubboResult<Boolean> returnSkuStockByOrderId(String orderId) {
        List<QihoOrderItemInfoEntity> list = qihoOrderItemInfoMapper.selectQihoOrderItemInfosByOrderId(orderId);
        List<UpdateStockReqDto> increateList = Lists.newArrayList();
        for (QihoOrderItemInfoEntity qihoOrderItemInfo : list) {
            UpdateStockReqDto reqDto = new UpdateStockReqDto();
            reqDto.setStockId(qihoOrderItemInfo.getStockId());
            reqDto.setStock(qihoOrderItemInfo.getQuantity());
            increateList.add(reqDto);
        }
        DubboResult<Boolean> result = remoteStockBackendService.increaseItemStockBatch(increateList);
        if (!result.isSuccess()) {
            AppLogUtil.warn(logger, "订单{}:库存返还失败!", orderId);
            throw new QihoException("库存返还失败!");
        }
        return result;
    }

    @Override
    public void handleEdbErpStocks(List<UpdateStockReqDto> increaseStock, List<UpdateStockReqDto> decreaseStock, List<EdbItemStockResponse> responses) {
        List<String> paramSkuNos = responses.stream().map(EdbItemStockResponse::getBarCode).collect(Collectors.toList());
        //批量查询
        List<QihoItemSkuEntity> qihoItemSkus = qihoItemSkuMapper.selectALLItemSku(paramSkuNos);//通过商家编号获取存在本系统的sku数据
        if (qihoItemSkus.isEmpty()) {
            return;
        }
        //查询erp中还未通知给erp的商品对应的订单数量
        Map<String, Map<String, BigDecimal>> erpQ = qihoItemSkuMapper.selectErpOrderQuantityBySkuNo(paramSkuNos);
        //获取order中待付款和待审核商品订单的数量
        Map<String, Map<String, BigDecimal>> ordQ = qihoItemSkuMapper.selectOrderQuantityBySkuNo(paramSkuNos);
        List<Long> stockIds = toConvertStockIds(qihoItemSkus);
        //批量查询所有库存的信息
        DubboResult<List<StockDto>> stockDs = remoteStockService.findBatchByIds(stockIds);
        Map<String, QihoItemSkuEntity> qihoItemSkuMap = toConvertSkuMap(qihoItemSkus);
        Map<Long, StockDto> stockMaps = toConvertMap(stockDs);
        for (EdbItemStockResponse stock : responses) {//开始批量处理
            QihoItemSkuEntity sku = qihoItemSkuMap.get(stock.getBarCode());
            if (sku != null){
                handlerEdbErpStock(increaseStock, decreaseStock, erpQ, ordQ, stockMaps, stock, sku);
            }

        }
    }

    /**
     * 旺店通库存同步
     * 处理库存数据
     */
    public void handlerErpStocks(List<UpdateStockReqDto> increaseStock, List<UpdateStockReqDto> decreaseStock, List<ErpStock> stocks) {
        List<String> paramSkuNo = toConvert(stocks);//获取所有的商家编号
        //批量查询
        List<QihoItemSkuEntity> qihoItemSkus = qihoItemSkuMapper.selectALLItemSku(paramSkuNo);//通过商家编号获取存在本系统的sku数据
        if (!qihoItemSkus.isEmpty()) {
            Map<String, Map<String, BigDecimal>> erpQ = qihoItemSkuMapper.selectErpOrderQuantityBySkuNo(paramSkuNo);//获取erp中订单的数量
            Map<String, Map<String, BigDecimal>> ordQ = qihoItemSkuMapper.selectOrderQuantityBySkuNo(paramSkuNo);//获取order中订单的数量
            List<Long> stockIds = toConvertStockIds(qihoItemSkus);
            DubboResult<List<StockDto>> stockDs = remoteStockService.findBatchByIds(stockIds);//批量查询所有库存的信息
            Map<String, QihoItemSkuEntity> qihoItemSkuMap = toConvertSkuMap(qihoItemSkus);
            Map<Long, StockDto> stockMaps = toConvertMap(stockDs);
            for (ErpStock stock : stocks) {//开始批量处理
                QihoItemSkuEntity sku = qihoItemSkuMap.get(stock.getSpecNo());
                if (sku == null) continue;
                handlerErpStock(increaseStock, decreaseStock, erpQ, ordQ, stockMaps, stock, sku);
            }
        }
    }

    /**
     * 处理单品库存的逻辑
     *
     * @param increaseStock
     * @param decreaseStock
     * @param erpQ
     * @param ordQ
     * @param stockMaps
     * @param stock
     * @param sku
     */
    private void handlerEdbErpStock(List<UpdateStockReqDto> increaseStock, List<UpdateStockReqDto> decreaseStock, Map<String, Map<String, BigDecimal>> erpQ, Map<String, Map<String, BigDecimal>> ordQ, Map<Long, StockDto> stockMaps, EdbItemStockResponse stock, QihoItemSkuEntity sku) {
        long erpNum = 0;
        long orderNum = 0;
        String paramSkuNo = stock.getBarCode();
        if (erpQ.get(paramSkuNo) != null) {
            erpNum = erpQ.get(paramSkuNo).get("quantity").longValue();
        }
        if (ordQ.get(paramSkuNo) != null) {
            orderNum = ordQ.get(paramSkuNo).get("quantity").longValue();
        }
        long quantity = erpNum + orderNum;
        long totalSkuNum = stock.getSellStock() - quantity;
        long creaseNum = 0;
        long stockId = sku.getStockId();
        //如果订单库存超过了ERP实际库存  库存置为零防止继续超卖
        long remoteStock = stockMaps.get(stockId).getStock();
        if (totalSkuNum >= 0) {
            creaseNum = totalSkuNum - remoteStock;
        } else {
            creaseNum = creaseNum - remoteStock;
            logger.warn("库存{}出现超卖,请检查库存，库存已置为0", stockId);
        }
        syncEdbStockLog(stock, quantity, creaseNum, remoteStock);
        if (creaseNum == 0) return;
        UpdateStockReqDto updateStockReqDto = new UpdateStockReqDto();
        updateStockReqDto.setStockId(stockId);
        updateStockReqDto.setStock(Math.abs(creaseNum));
        if(creaseNum>0&&notExist(updateStockReqDto,increaseStock)){
            increaseStock.add(updateStockReqDto);
        } else if(creaseNum<0&&notExist(updateStockReqDto,decreaseStock)){
            decreaseStock.add(updateStockReqDto);
        }
    }

    /**
     * 判断库存集合中是否
     * 已经存在该库存
     * @param updateStockReqDto
     * @param list
     * @return
    */
    private boolean notExist(UpdateStockReqDto updateStockReqDto, List<UpdateStockReqDto> list) {
        for (UpdateStockReqDto stockReqDto : list) {
            if (stockReqDto.getStockId() == updateStockReqDto.getStockId()) {
                return false;
            }
        }
        return true;
    }

        /**
         * 处理单品库存的逻辑
         *
         * @param increaseStock
         * @param decreaseStock
         * @param erpQ
         * @param ordQ
         * @param stockMaps
         * @param stock
         * @param sku
         */
    private void handlerErpStock(List<UpdateStockReqDto> increaseStock, List<UpdateStockReqDto> decreaseStock, Map<String, Map<String, BigDecimal>> erpQ, Map<String, Map<String, BigDecimal>> ordQ, Map<Long, StockDto> stockMaps, ErpStock stock, QihoItemSkuEntity sku) {
        long erpNum = 0;
        long orderNum = 0;
        if (erpQ.get(stock.getSpecNo()) != null) {
            erpNum = erpQ.get(stock.getSpecNo()).get("quantity").longValue();
        }
        if (ordQ.get(stock.getSpecNo()) != null) {
            orderNum = ordQ.get(stock.getSpecNo()).get("quantity").longValue();
        }
        long quantity = erpNum + orderNum;
        long totalSkuNum = (long) stock.getStockNum() - quantity;
        long creaseNum = 0;
        long stockId = sku.getStockId();
        //如果订单库存超过了ERP实际库存  库存置为零防止继续超卖
        long remoteStock = stockMaps.get(stockId).getStock();
        if (totalSkuNum >= 0) {
            creaseNum = totalSkuNum - remoteStock;
        } else {
            creaseNum = creaseNum - remoteStock;
            logger.warn("库存{}出现超卖,请检查库存，库存已置为0", stockId);
        }
        syncStockLog(stock, quantity, creaseNum, remoteStock);
        if (creaseNum == 0) return;
        UpdateStockReqDto updateStockReqDto = new UpdateStockReqDto();
        updateStockReqDto.setStockId(stockId);
        updateStockReqDto.setStock(Math.abs(creaseNum));
        if (creaseNum > 0) {
            increaseStock.add(updateStockReqDto);
        } else if (creaseNum < 0) {
            decreaseStock.add(updateStockReqDto);
        }
    }


    /**
     * 记录E店包同步库存同步日志
     *
     * @param stock
     * @param quantity
     * @param creaseNum
     * @param remoteStock
     */
    private void syncEdbStockLog(EdbItemStockResponse stock, long quantity, long creaseNum, long remoteStock) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("specNo", stock.getBarCode());
        jsonObject.put("syncStock", stock.getSellStock());
        jsonObject.put("quantity", quantity);
        jsonObject.put("systemStock", remoteStock);
        jsonObject.put("creaseNum", creaseNum);
        InnerLog.log(InnerLogTypeEnum.SYNC_STOCK_RULE, jsonObject.toJSONString());
    }

    /**
     * 记录同步日志
     *
     * @param stock
     * @param quantity
     * @param creaseNum
     * @param remoteStock
     */
    private void syncStockLog(ErpStock stock, long quantity, long creaseNum, long remoteStock) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("specNo", stock.getSpecNo());
        jsonObject.put("syncStock", stock.getStockNum());
        jsonObject.put("quantity", quantity);
        jsonObject.put("systemStock", remoteStock);
        jsonObject.put("creaseNum", creaseNum);
        InnerLog.log(InnerLogTypeEnum.SYNC_STOCK_RULE, jsonObject.toJSONString());
    }


    private Map<String, QihoItemSkuEntity> toConvertSkuMap(List<QihoItemSkuEntity> qihoItemSkus) {
        Map<String, QihoItemSkuEntity> result = new HashMap<>();
        for (QihoItemSkuEntity itemSkus : qihoItemSkus) {
            result.put(itemSkus.getSkuNo(), itemSkus);
        }
        return result;
    }

    private List<Long> toConvertStockIds(List<QihoItemSkuEntity> qihoItemSkus) {
        List<Long> result = new ArrayList<>();
        for (QihoItemSkuEntity itemSkus : qihoItemSkus) {
            result.add(itemSkus.getStockId());
        }
        return result;
    }

    private Map<Long, StockDto> toConvertMap(DubboResult<List<StockDto>> stockDs) {
        Map<Long, StockDto> result = new HashMap<>();
        List<StockDto> list = stockDs.getResult();
        for (StockDto stockDto : list) {
            result.put(stockDto.getStockID(), stockDto);
        }
        return result;
    }

    private List<String> toConvert(List<ErpStock> stocks) {
        List<String> result = new ArrayList<>();
        for (ErpStock stock : stocks) {
            if (StringUtils.isNotBlank(stock.getSpecNo()))
                result.add(String.valueOf(stock.getSpecNo()));
        }
        return result;
    }

}
