package com.qiho.center.biz.service.menu.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import com.qiho.center.api.constant.AccountMenuConstant;
import com.qiho.center.api.dto.menu.AccountMenuDto;
import com.qiho.center.api.params.menu.AccountMenuParams;
import com.qiho.center.api.params.menu.MenuParams;
import com.qiho.center.biz.service.menu.AccountMenuService;
import com.qiho.center.biz.util.AccountMenuUtil;
import com.qiho.center.common.daoh.qiho.menu.BaiqiAccountMenuMapper;
import com.qiho.center.common.daoh.qiho.menu.BaiqiMenuMapper;
import com.qiho.center.common.entityd.qiho.menu.BaiqiAccountMenuEntity;
import com.qiho.center.common.entityd.qiho.menu.BaiqiMenuEntity;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 用户菜单接口
 *
 * @author zhangshun
 * @version V1.0
 * @since 2017-12-07 19:34
 */
@Service
public class AccountMenuServiceImpl implements AccountMenuService {
    private static final Logger LOGGER = LoggerFactory.getLogger(AccountMenuServiceImpl.class);

    @Resource
    private BaiqiMenuMapper baiqiMenuMapper;

    @Resource
    private BaiqiAccountMenuMapper baiqiAccountMenuMapper;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Override
    public Boolean saveAccountMenuList(Long accountId, Integer accountType, String accountName, List<Long> menuIdList)
        throws BizException {

        //检查账号类型
        AccountMenuUtil.checkMasterAccount(accountType);

        if (CollectionUtils.isEmpty(menuIdList)) {
            throw new BizException("菜单列表不能为空！");
        }

        try {

            List<BaiqiAccountMenuEntity> entityList = new ArrayList<>();

            Date curDate = Calendar.getInstance().getTime();

            BaiqiAccountMenuEntity entity = null;

            for (Long menuId : menuIdList) {
                entity = new BaiqiAccountMenuEntity();
                entity.setGmtModified(curDate);
                entity.setGmtCreate(curDate);
                entity.setAccountId(accountId);
                entity.setMenuId(menuId);
                entityList.add(entity);
            }

            // 批量插入
            baiqiAccountMenuMapper.insertBatch(entityList);

            // 移除缓存
            this.removeCacheLoginAndMenu(accountName);

            return Boolean.TRUE;
        } catch (Exception e) {
            LOGGER.error("保存用户权限错误！", e);
            return Boolean.FALSE;
        }
    }

    @Override
    public Boolean updateAccountMenuList(Long accountId, Integer accountType, String accountName, List<Long> menuIdList)
        throws BizException {

        //检查账号类型
        AccountMenuUtil.checkMasterAccount(accountType);

        if (CollectionUtils.isEmpty(menuIdList)) {
            throw new BizException("菜单列表不能为空！");
        }

        try {

            // 删除改用的菜单
            baiqiAccountMenuMapper.deleteByAccountId(accountId);

            // 重新添加用户菜单
            this.saveAccountMenuList(accountId, accountType, accountName, menuIdList);

            return Boolean.TRUE;
        } catch (Exception e) {
            LOGGER.error("修改用户权限错误！", e);
            return Boolean.FALSE;
        }
    }

    @Override
    public Integer deleteByAccountId(Long accountId, Integer accountType, String accountName) throws BizException {

        //检查账号类型
        AccountMenuUtil.checkMasterAccount(accountType);

        // 移除缓存
        this.removeCacheLoginAndMenu(accountName);

        return baiqiAccountMenuMapper.deleteByAccountId(accountId);
    }

    @Override
    public List<AccountMenuDto> findAll() throws BizException {
        // 查询全部菜单
        MenuParams params = new MenuParams();
        return AccountMenuUtil.convertMenuEntityToDto(baiqiMenuMapper.selectByParam(params));
    }

    @Override
    public List<AccountMenuDto> findByParamAsTree(AccountMenuParams params) throws BizException {

        Assert.notNull(params.getAccountId(), "AccountId为空");

        return AccountMenuUtil.convertMenuListToTree(this.getMenuListBySubAccount(params));
    }

    private List<AccountMenuDto> getMenuListBySubAccount(AccountMenuParams params) throws BizException {

        // 查询全部菜单
        List<AccountMenuDto> accountMenuDtoList = this.findAll();

        if (CollectionUtils.isEmpty(accountMenuDtoList)) {
            return Collections.emptyList();
        }

        // 去掉商家和账号
        AccountMenuDto accountMenuDtoAll = null;
        Iterator<AccountMenuDto> iteratorAll = accountMenuDtoList.iterator();
        while (iteratorAll.hasNext()) {
            accountMenuDtoAll = iteratorAll.next();
            if (hideMenuAccountANDMerchant(accountMenuDtoAll)) {
                iteratorAll.remove();
            }
        }

        // 没有账号，返回全部
        if (-1 == params.getAccountId()) {
            return accountMenuDtoList;
        }

        // 查询子账号菜单
        List<Long> accountMenuEntityList = AccountMenuUtil
            .convertMenuEntityToListMenuId(baiqiAccountMenuMapper.selectByParam(params));

        // 子账号没有菜单，返回空
        if (CollectionUtils.isEmpty(accountMenuEntityList)) {
            return accountMenuDtoList;
        }

        for (AccountMenuDto accountMenuDto : accountMenuDtoList) {
            if (accountMenuEntityList.contains(accountMenuDto.getId())) {
                accountMenuDto.setSelected(true);
            } else {
                accountMenuDto.setSelected(false);
            }
        }

        return accountMenuDtoList;
    }

    /**
     * 菜单编辑时候，过滤账号和商家菜单
     *
     * @param accountMenuDto
     * @return
     */
    private Boolean hideMenuAccountANDMerchant(AccountMenuDto accountMenuDto) {
        String menuCode = accountMenuDto.getMenuCode();
        if (AccountMenuConstant.ACCOUNT.equals(menuCode) || AccountMenuConstant.SUB_ACCOUNT.equals(menuCode)
            || AccountMenuConstant.MERCHANT.equals(menuCode) || AccountMenuConstant.MERCHANT_LIST.equals(menuCode)) {
            return true;
        }
        return false;
    }

    @Override
    public boolean isOperationByAccountIdAndMenuId(Long accountId, Integer accountType, Long menuId) {

        // 是主账号，直接通过
        if (AccountMenuUtil.isMasterAccount(accountType)) {
            return true;
        }

        // cacheMapKey
        String cacheMapKey = this.getCacheMapKey(accountId, menuId);

        // 查询Redis
        Map<String, Boolean> cacheMap = this.getCache(accountId);

        // 如果redis缓存已经存在
        if (null == cacheMap) {
            cacheMap = new HashMap<>();
        } else {
            if (cacheMap.containsKey(cacheMapKey)) {
                return cacheMap.get(cacheMapKey);
            }
        }

        // 如果redis缓存不存在，查询数据库

        // 查询 菜单信息
        BaiqiMenuEntity menuEntity = baiqiMenuMapper.selectById(menuId);
        if (null == menuEntity || 1 == menuEntity.getDeleted()) {
            // 菜单信息不存在
            cacheMap.put(cacheMapKey, Boolean.FALSE);
            // 设置缓存
            this.setCache(accountId, cacheMap);
            return cacheMap.get(cacheMapKey);
        }

        // 查询 用户菜单信息
        BaiqiAccountMenuEntity accountMenuEntity = baiqiAccountMenuMapper.selectByAccountIdAndMenuId(accountId, menuId);

        if (null == accountMenuEntity) {

            // 用户菜单没有权限
            cacheMap.put(cacheMapKey, Boolean.FALSE);
            // 设置缓存
            this.setCache(accountId, cacheMap);
            return cacheMap.get(cacheMapKey);

        }

        cacheMap.put(cacheMapKey, Boolean.TRUE);
        this.setCache(accountId, cacheMap);

        return cacheMap.get(cacheMapKey);
    }

    @Override
    public List<AccountMenuDto> findByAccountId(Long accountId, Integer accountType) throws BizException {
        List<AccountMenuDto> accountMenuDtoList = null;
        if (AccountMenuUtil.isMasterAccount(accountType)) {// 是主账号
            accountMenuDtoList = this.findByMainAccount();
        } else if (AccountMenuUtil.isSubAccount(accountType)) {// 是子主账号
            accountMenuDtoList = this.findBySubAccount(accountId);
        } else if (AccountMenuUtil.isAgentAccount(accountType)) {// 是代理商账号
            accountMenuDtoList = this.findByAgentAccount();
        }
        return AccountMenuUtil.convertMenuListToTree(accountMenuDtoList);
    }

    private List<AccountMenuDto> findByMainAccount() throws BizException {

        List<AccountMenuDto> accountMenuDtoList = this.findAll();
        if (CollectionUtils.isEmpty(accountMenuDtoList)) {
            return Collections.emptyList();
        }

        Iterator<AccountMenuDto> iterator = accountMenuDtoList.iterator();

        String menuCode = null;
        while (iterator.hasNext()) {
            menuCode = iterator.next().getMenuCode();
            if (AccountMenuConstant.MERCHANT.equals(menuCode) || AccountMenuConstant.MERCHANT_LIST.equals(menuCode)) {
                iterator.remove();
            }
        }

        return accountMenuDtoList;
    }

    private List<AccountMenuDto> findBySubAccount(Long accountId) {
        // 查询用户菜单
        AccountMenuParams params = new AccountMenuParams();
        params.setAccountId(accountId);

        // 查询子账号菜单
        List<Long> menuIdList = AccountMenuUtil
            .convertMenuEntityToListMenuId(baiqiAccountMenuMapper.selectByParam(params));

        if (CollectionUtils.isEmpty(menuIdList)) {
            return Collections.emptyList();
        }

        List<BaiqiMenuEntity> accountMenuEntityList = baiqiMenuMapper.selectByParam(new MenuParams());

        if (CollectionUtils.isEmpty(accountMenuEntityList)) {
            return Collections.emptyList();
        }

        return AccountMenuUtil.convertMenuEntityToDto(accountMenuEntityList, menuIdList);
    }

    private List<AccountMenuDto> findByAgentAccount() throws BizException {

        List<AccountMenuDto> accountMenuDtoList = this.findAll();

        Iterator<AccountMenuDto> iterator = accountMenuDtoList.iterator();

        AccountMenuDto accountMenuDto = null;
        while (iterator.hasNext()) {
            accountMenuDto = iterator.next();
            if (!isMenuAgentAccount(accountMenuDto)) {
                iterator.remove();
            }
        }

        return accountMenuDtoList;
    }

    private Boolean isMenuAgentAccount(AccountMenuDto accountMenuDto) {
        String menuCode = accountMenuDto.getMenuCode();
        if (AccountMenuConstant.PERSON_CENTER.equals(menuCode) // 个人中心
            || AccountMenuConstant.MOD_PD.equals(menuCode) // 修改密码
            || AccountMenuConstant.RECORD_LOG.equals(menuCode) // 操作日志
            || AccountMenuConstant.MERCHANT.equals(menuCode) // 商家管理
            || AccountMenuConstant.MERCHANT_LIST.equals(menuCode) // 商家列表
            || AccountMenuConstant.FINANCE.equals(menuCode) //财务中心
            || AccountMenuConstant.FINANCE_DETAIL.equals(menuCode) // 财务明细
            || AccountMenuConstant.FINANCE_OVERVIEW.equals(menuCode) // 财务总览
            ) {
            return true;
        }
        return false;
    }

    /**
     * 移除登录缓存和用户菜单缓存
     *
     * @param accountName
     */
    private void removeCacheLoginAndMenu(String accountName) {
        /**
         * 暂时不需要执行
         * 移除用户菜单redis
         * advancedCacheClient.remove(AccountMenuConstant.USER_MENU_REDIS_KEY + accountId);
         */
        // 移除登录信息
        advancedCacheClient.remove(AccountMenuConstant.LOGIN_USER_REDIS_KEY + accountName);
    }

    private String getCacheMapKey(Long accountId, Long menuId) {
        return accountId + "_" + menuId;
    }

    private Map<String, Boolean> getCache(Long accountId) {
        return advancedCacheClient.get(AccountMenuConstant.USER_MENU_REDIS_KEY + accountId);
    }

    private boolean setCache(Long accountId, Map<String, Boolean> value) {
        return advancedCacheClient
            .set(AccountMenuConstant.USER_MENU_REDIS_KEY + accountId, value, AccountMenuConstant.USER_MENU_REDIS_EXP,
                TimeUnit.DAYS);
    }

}
