package com.qiho.center.biz.service.menu.impl;

import com.qiho.center.api.constant.AccountMenuConstant;
import com.qiho.center.api.dto.menu.MenuDto;
import com.qiho.center.api.params.menu.MenuParams;
import com.qiho.center.biz.service.account.BaiqiAccountService;
import com.qiho.center.biz.service.menu.MenuService;
import com.qiho.center.common.daoh.qiho.menu.BaiqiMenuMapper;
import com.qiho.center.common.entityd.qiho.menu.BaiqiMenuEntity;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;

/**
 * 菜单接口
 *
 * @author zhangshun
 * @version V1.0
 * @since 2017-12-07 16:21
 */
@Service
public class MenuServiceImpl implements MenuService {
    private static final Logger LOGGER = LoggerFactory.getLogger(MenuServiceImpl.class);

    @Resource
    private BaiqiMenuMapper baiqiMenuMapper;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Resource
    private BaiqiAccountService baiqiAccountService;

    @Override
    public boolean initMenu() {
        try {

            // 菜单排序
            this.updateSortByCode("AllOrder", "全部订单", 100);
            this.updateSortByCode("ExamineOrder", "待审核订单", 101);
            this.updateSortByCode("ERPSubOrder", "待发货订单", 102);
            this.updateSortByCode("AlreadySendOrder", "已发货订单", 103);
            this.updateSortByCode("CancelSendOrder", "取消发货订单", 104);
            this.updateSortByCode("CustomerServiceOrder", "售后子订单", 105);

            return Boolean.TRUE;
        } catch (Exception e) {
            LOGGER.error("初始化数据菜单错误", e);
            return Boolean.FALSE;
        }
    }

    @Override
    public Long saveMenu(MenuDto menuDto) throws BizException {

        BaiqiMenuEntity entity = menuDtoToEntity(menuDto);

        if (null == entity) {
            return 0L;
        }

        Integer result = baiqiMenuMapper.insert(entity);

        if (0 >= result) {
            return 0L;
        }

        // 清除缓存
        this.removeCacheLogin();

        return entity.getId();
    }

    @Override
    public Integer updateMenu(MenuDto menuDto) throws BizException {

        BaiqiMenuEntity entity = menuDtoToEntity(menuDto);

        if (null == entity) {
            return 0;
        }

        entity.setSorted(getSorted(entity.getSorted()));

        Integer result = baiqiMenuMapper.updateById(entity);

        // 清除缓存
        this.removeCacheLogin();

        return result;
    }

    @Override
    public Integer updateByCode(MenuDto menuDto) throws BizException {

        BaiqiMenuEntity entity = menuDtoToEntity(menuDto);

        if (null == entity) {
            return 0;
        }

        entity.setSorted(getSorted(entity.getSorted()));

        Integer result = baiqiMenuMapper.updateByCode(entity);

        // 清除缓存
        this.removeCacheLogin();

        return result;
    }

    @Override
    public Integer deleteById(Long id) throws BizException {

        Integer result = baiqiMenuMapper.deleteById(id);

        // 清除缓存
        this.removeCacheLogin();

        return result;
    }

    @Override
    public Integer deleteByCode(String code) throws BizException {

        Integer result = baiqiMenuMapper.deleteByCode(code);

        // 清除缓存
        this.removeCacheLogin();

        return result;
    }

    @Override
    public MenuDto findByCode(String code) throws BizException {
        MenuParams params = new MenuParams();
        params.setMenuCode(code);
        List<MenuDto> menuDtoList = findByParam(params);
        if (CollectionUtils.isEmpty(menuDtoList)) {
            return null;
        }
        return menuDtoList.get(0);
    }

    @Override
    public List<MenuDto> findByParam(MenuParams menuParams) throws BizException {
        List<BaiqiMenuEntity> entityList = baiqiMenuMapper.selectByParam(menuParams);
        return menuEntityToDto(entityList);
    }

    private void removeCacheLogin() {

        // 查询所有的用户
        List<String> accountList = baiqiAccountService.findAllMainAccount();
        if (CollectionUtils.isEmpty(accountList)) {
            return;
        }

        // 移除登录的缓存
        for (String account : accountList) {
            advancedCacheClient.remove(AccountMenuConstant.LOGIN_USER_REDIS_KEY + account);
        }
    }

    public static BaiqiMenuEntity menuDtoToEntity(MenuDto menuDto) {
        if (menuDto == null) {
            return null;
        }
        BaiqiMenuEntity baiqiMenuEntity = new BaiqiMenuEntity();
        baiqiMenuEntity.setId(menuDto.getId());
        baiqiMenuEntity.setParentId(menuDto.getParentId());
        baiqiMenuEntity.setMenuCode(menuDto.getMenuCode());
        baiqiMenuEntity.setMenuName(menuDto.getMenuName());
        baiqiMenuEntity.setRemark(menuDto.getRemark());
        baiqiMenuEntity.setSorted(menuDto.getSorted());
        baiqiMenuEntity.setDeleted(menuDto.getDeleted());
        baiqiMenuEntity.setGmtCreate(menuDto.getGmtCreate());
        baiqiMenuEntity.setGmtModified(menuDto.getGmtModified());
        return baiqiMenuEntity;
    }

    public static MenuDto menuEntityToDto(BaiqiMenuEntity entity) {
        if (entity == null) {
            return null;
        }
        MenuDto menuDto = new MenuDto();
        menuDto.setId(entity.getId());
        menuDto.setParentId(entity.getParentId());
        menuDto.setMenuCode(entity.getMenuCode());
        menuDto.setMenuName(entity.getMenuName());
        menuDto.setRemark(entity.getRemark());
        menuDto.setSorted(entity.getSorted());
        menuDto.setDeleted(entity.getDeleted());
        menuDto.setGmtCreate(entity.getGmtCreate());
        menuDto.setGmtModified(entity.getGmtModified());
        return menuDto;
    }

    public static List<MenuDto> menuEntityToDto(List<BaiqiMenuEntity> entityList) {
        if (null == entityList || entityList.isEmpty()) {
            return Collections.emptyList();
        }
        return entityList.stream().map(MenuServiceImpl::menuEntityToDto).collect(Collectors.toList());
    }

    private int getSorted(Integer sort) {
        if (null == sort || sort < 0) {
            return 0;
        } else if (sort > 1000) {
            return 1000;
        } else {
            return sort;
        }
    }

    /**
     * 根据code修改排序值
     *
     * @param code
     * @param sort
     * @return
     * @throws BizException
     */
    private Boolean updateSortByCode(String code, String name, int sort) throws BizException {
        MenuDto menuDto = new MenuDto();
        menuDto.setMenuCode(code);
        menuDto.setMenuName(name);
        menuDto.setSorted(getSorted(sort));
        if (1 == updateByCode(menuDto)) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }
}