package com.qiho.center.biz.service.page.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.dto.ItemDto;
import com.qiho.center.api.dto.page.BaiqiPageDto;
import com.qiho.center.api.dto.page.CollItemDto;
import com.qiho.center.api.dto.page.CollPageDto;
import com.qiho.center.api.dto.tag.TagDto;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.enums.page.CollItemStatusEnum;
import com.qiho.center.api.params.CollItemListQueryParam;
import com.qiho.center.biz.service.ItemSaleCountService;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.page.BaiqiPageService;
import com.qiho.center.biz.service.page.CollItemService;
import com.qiho.center.biz.service.page.CollPageService;
import com.qiho.center.common.daoh.qiho.page.CollPageMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoItemTagMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoTagMapper;
import com.qiho.center.common.entityd.qiho.page.CollPageEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoItemTagEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoTagEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author jiafeng.geng
 * @date 2018-07-19.
 */
@Service
public class CollPageServiceImpl implements CollPageService {

    private final static Logger LOGGER = LoggerFactory.getLogger(CollPageServiceImpl.class);

    @Resource
    private CollPageMapper collPageMapper;

    @Resource
    private BaiqiPageService baiqiPageService;

    @Resource
    private CollItemService collItemService;

    @Resource
    private ItemService itemService;

    @Resource
    private ItemSaleCountService itemSaleCountService;

    @Resource
    private QihoItemTagMapper qihoItemTagMapper;

    @Resource
    private QihoTagMapper qihoTagMapper;


    @Override
    public List<CollPageDto> queryList() {
        List<CollPageEntity> collPageEntityList = collPageMapper.queryList();
        return collPageEntityList.isEmpty() ? null : converToCollPageDto(collPageEntityList);
    }

    private List<CollPageDto> converToCollPageDto(List<CollPageEntity> collPageEntityList) {
        List<CollPageDto> collPageDtoList = BeanUtils.copyList(collPageEntityList, CollPageDto.class);

        // 提取所有集合页对应的页面模板id
        List<Long> pageSkinIdList = collPageDtoList.stream()
                .map(CollPageDto::getPageSkinId)
                .distinct()
                .collect(Collectors.toList());
        // 根据页面模板集合查询页面模板对象
        List<BaiqiPageDto> baiqiPageDtoList = baiqiPageService.findByIds(pageSkinIdList);

        // 根据每个集合页的页面模板id获取并设置其对应的页面模板名称
        collPageDtoList.forEach(collPageDto ->
                baiqiPageDtoList.forEach(baiqiPageDto -> {
                    if (collPageDto.getPageSkinId().longValue() == baiqiPageDto.getId().longValue()) {
                        collPageDto.setPageSkinName(baiqiPageDto.getPageName());
                    }
                })
        );

        return collPageDtoList;
    }

    @Override
    public Integer addCollPage(CollPageDto collPageDto) {
        CollPageEntity collPageEntity = BeanUtils.copy(collPageDto, CollPageEntity.class);
        return collPageMapper.addCollPage(collPageEntity);
    }

    @Override
    public Integer updateCollPage(CollPageDto collPageDto) {
        CollPageEntity collPageEntity = BeanUtils.copy(collPageDto, CollPageEntity.class);
        return collPageMapper.updateCollPage(collPageEntity);
    }

    @Override
    public List<ItemDto> queryCollItemIndex(CollItemListQueryParam params) {
        // 集合页商品列表
        List<CollItemDto> collItemDtoList;

        if (CollectionUtils.isEmpty(params.getItemIds())) {
            // 参数中没有商品id集合时,根据集合页id查询商品列表
            collItemDtoList = collItemService.queryItemListBySort(params.getCollId());
        } else {
            // 参数中有商品id集合时,根据集合页id和商品id集合查询商品列表
            collItemDtoList = collItemService.queryItemListBySortWithItemIds(params.getCollId(), params.getItemIds());
        }

        if (CollectionUtils.isEmpty(collItemDtoList)) {
            LOGGER.info("查询到的商品对象集合为空, params={}", params);
            return Lists.newArrayList();
        }

        // 过滤在集合页中为禁用状态的商品， 并使用map记录集合页商品信息及其顺序
        Map<Long, CollItemDto> map = Maps.newLinkedHashMap();
        List<Long> itemIds = Lists.newArrayList();
        for (CollItemDto collItemDto : collItemDtoList) {
            if (collItemDto.getCollItemStatus().intValue() == CollItemStatusEnum.STATUS_ON.getValue().intValue()) {
                map.put(collItemDto.getId(), collItemDto);
                itemIds.add(collItemDto.getId());
            }
        }

        // 查询商品信息
        List<ItemDto> itemDtoList = itemService.queryItemsByItemIds(itemIds);
        Map<Long, Integer> saleMap = itemSaleCountService.queryItemSaleNum(itemIds);

        List<ItemDto> resultList = Lists.newArrayList();
        Map<Long, ItemDto> resultMap = Maps.newHashMap();
        for (ItemDto itemDto : itemDtoList) {
            // 如果商品下架或者库存为0,则剔除掉
            if (StringUtils.equals(ItemStatusEnum.STATUS_OFF.getCode(), itemDto.getItemStatus())
                    || itemDto.getStock() == null || itemDto.getStock().intValue() <= 0) {
                continue;
            }

            // 设置样式信息
            setStyle(itemDto, collItemDtoList);

            // 设置一下库存
            itemDto.setSaleNum(null == saleMap.get(itemDto.getId()) ? 0 : saleMap.get(itemDto.getId()));

            // 设置一下商品的虚拟库存剩余比率
            itemDto.setVirtualRemainRate(setVirRemain(itemDto));

            resultMap.put(itemDto.getId(), itemDto);
        }
        // 进行重新排序
        map.forEach((key, value) -> {
            if (null != resultMap.get(key)) {
                resultList.add(resultMap.get(key));
            }
        });

        // 提取商品id集合
        List<Long> itemIdList = resultList.stream().map(ItemDto::getId).collect(Collectors.toList());

        // 根据商品id集合查询并设置对应的标签对象
        return setItemTags(itemIdList, resultList);
    }

    private void setStyle(ItemDto item, List<CollItemDto> collItemDtoList) {
        for (CollItemDto dto : collItemDtoList) {
            if (item.getId().longValue() == dto.getId().longValue()) {
                if (StringUtils.isNotBlank(dto.getStyleConfig())) {
                    item.setStyleConfig(dto.getStyleConfig());
                }
                item.setTabItemStatus(dto.getCollItemStatus());
            }
        }
    }

    private Double setVirRemain(ItemDto item) {
        Long stock = item.getEnableStock();
        if (stock == 0L) {
            return 0.00;
        }
        // 当虚拟库存或虚拟库存小于实际时要重新计算
        Long virStock = Long.valueOf(item.getExtParamValue(ItemConstants.ItemExtConstans.VIRTUALSTOCK) == null ?
                "0" : item.getExtParamValue(ItemConstants.ItemExtConstans.VIRTUALSTOCK));
        if (virStock == 0L || virStock < stock) {
            virStock = itemService.updateVirtailStock(item.getId());
        }
        BigDecimal paramNs = new BigDecimal(virStock);
        BigDecimal itemStocks = new BigDecimal(stock);
        return itemStocks.divide(paramNs, 2, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    private List<ItemDto> setItemTags(List<Long> itemIds, List<ItemDto> itemDtoList) {
        if (CollectionUtils.isEmpty(itemIds)) {
            return itemDtoList;
        }

        // 根据商品id集合查询对象集合
        List<QihoItemTagEntity> itemTags = qihoItemTagMapper.findByItemIds(itemIds);
        if (CollectionUtils.isEmpty(itemTags)) {
            LOGGER.info("获取到的商品-标签集合为空, itemIds={}", itemIds);
            return itemDtoList;
        }

        // 从对象集合中提取tagId集合, 并根据tagId集合查询对应的标签对象集合
        List<Long> tagIds = itemTags.stream().map(QihoItemTagEntity::getTagId).collect(Collectors.toList());
        List<QihoTagEntity> tagList = qihoTagMapper.findByIds(tagIds);
        if (CollectionUtils.isEmpty(tagList)) {
            LOGGER.info("获取到的百奇屏蔽标签为空, itemIds={}", itemIds);
            return itemDtoList;
        }

        // 使用map将tagIds和tagList关联
        List<TagDto> tagDtoList = BeanUtils.copyList(tagList, TagDto.class);
        Map<Long, TagDto> map = new HashMap<>();
        for (TagDto tagDto : tagDtoList) {
            for (Long tagId : tagIds) {
                if (tagDto.getId().longValue() == tagId.longValue()) {
                    map.put(tagId, tagDto);
                    break;
                }
            }
        }

        // 设置商品对象中的标签集合
        int num = 0;
        for (ItemDto itemDto : itemDtoList) {
            List<TagDto> tagDtos = Lists.newArrayList();
            for (QihoItemTagEntity itemTagEntity : itemTags) {
                if (itemTagEntity.getItemId().longValue() == itemDto.getId()) {
                    tagDtos.add(map.get(itemTagEntity.getTagId()));
                }
            }
            itemDtoList.get(num++).setTagDtoList(tagDtos);
        }

        return itemDtoList;
    }
}
