package com.qiho.center.biz.subscribe;

import cn.com.duiba.tuia.activity.center.api.dto.ActivityOrderDto;
import cn.com.duiba.tuia.activity.center.api.remoteservice.RemoteActivityOrderService;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.baiqi.shield.center.api.dto.BlackListCheckDto;
import com.baiqi.shield.center.api.param.BlackListCheckParam;
import com.baiqi.shield.center.api.remoteservice.blacklist.RemoteShieldBlackListService;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.ChannelInfoDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.OrderItemDto;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.ordertmp.OrderTmpDetailDto;
import com.qiho.center.api.dto.sms.ChuangLanSmsReportDto;
import com.qiho.center.api.dto.strategy.FilterOrderCheckDto;
import com.qiho.center.api.enums.ShotOrder.StrategyType;
import com.qiho.center.api.enums.SmsTypeEnum;
import com.qiho.center.api.enums.ordertmp.OrderTmpRiskCheckFailEnum;
import com.qiho.center.api.enums.ordertmp.OrderTmpStatusEnum;
import com.qiho.center.biz.engine.impl.FilterOrderEngine;
import com.qiho.center.biz.event.ChuangLanReportReceiveEvent;
import com.qiho.center.biz.event.OrderTmpCreateEvent;
import com.qiho.center.biz.event.OrderTmpToFormalEvent;
import com.qiho.center.biz.service.ItemDetailRecordService;
import com.qiho.center.biz.service.impl.check.IPMobileCheckService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.service.order.OrderStrategyService;
import com.qiho.center.biz.service.ordersms.OrderSmsService;
import com.qiho.center.biz.service.ordertmp.OrderTmpService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.dao.QihoOrderItemInfoDAO;
import com.qiho.center.common.entity.order.QihoOrderItemInfoEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderSmsEntity;
import com.qiho.center.common.support.BizEventBus;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**临时订单事件 订阅者
 * @author chensong
 * @create 2018-05-11 15:39
 **/
@Component
@BizEventListener
public class OrderTmpSubscribe {

    private final static Logger LOGGER = LoggerFactory.getLogger(OrderTmpSubscribe.class);

    @Autowired
    private OrderTmpService orderTmpService;

    @Autowired
    private BizEventBus eventBus;

    @Autowired
    private OrderService orderService;

    @Autowired
    private RemoteShieldBlackListService remoteShieldBlackListService;

    @Autowired
    private FilterOrderEngine filterOrderEngine;

    @Autowired
    private OrderSmsService orderSmsService;

    @Autowired
    private OrderStrategyService orderStrategyService;

    @Autowired
    IPMobileCheckService ipMobileCheckService;

    @Autowired
    private ItemDetailRecordService itemDetailRecordService;

    @Autowired
    private QihoOrderItemInfoDAO qihoOrderItemInfoDAO;


    @Autowired
    private RemoteActivityOrderService remoteActivityOrderService;


    /**
     * 处理 临时订单创建事件
     * @param event
     */
    @Subscribe
    public void processOrderTmpCreateEvent(OrderTmpCreateEvent event){

        DBTimeProfile.enter("临时订单创建事件处理， orderId= " + event.getOrderId() );
        String orderId = event.getOrderId();
        OrderTmpDetailDto orderTmpDetailDto = orderTmpService.findByOrderId(orderId);
        if (null == orderTmpDetailDto) {
            DBTimeProfile.release();
            return;
        }

        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(orderTmpDetailDto, OrderSnapshotDto.class);

        // 新增订单商品详情
        OrderDto order = buildOrder(orderTmpDetailDto);
        QihoOrderItemInfoEntity qihoOrderItemInfo = BeanUtils.copy(order.getOrderItem(), QihoOrderItemInfoEntity.class);
        qihoOrderItemInfo.setOrderId(orderId);
        //获取一下最后版本的商品详情记录ID
        Long detailRecordId = itemDetailRecordService
                .getDetailRecordId(qihoOrderItemInfo.getItemId(), order.getPlanCode());
        qihoOrderItemInfo.setDetailRecordId(detailRecordId);
        qihoOrderItemInfoDAO.insert(qihoOrderItemInfo);

        DBTimeProfile.enter("ip手机号归属地检测");
        try{
            // 添加IP手机归属地检测
            ipMobileCheckService.checkToSave(orderSnapshotDto);
        } catch (Exception e){
            LOGGER.error("ip手机号归属地检测异常，orderId：{}", orderId, e);
        }
        DBTimeProfile.release();


        // 黑名单校验
        DBTimeProfile.enter("黑名单校验");
        boolean blackListCheckResult = blackListCheck(orderTmpDetailDto);
        DBTimeProfile.release();
        if (!blackListCheckResult) {
            // 命中黑名单 将订单置为无效
            orderTmpService.updateOrderTmpToInvalid(orderId, "黑名单不通过", OrderTmpRiskCheckFailEnum.BLACKLIST_FAIL);
            DBTimeProfile.release();
            return;
        }

        // 过滤规则校验 在这里并不校验手机号状态
        DBTimeProfile.enter("过滤规则同步校验");
        boolean filterResult = filterOrderCheck(orderTmpDetailDto);
        DBTimeProfile.release();
        if (filterResult) {
            // 命中过滤规则 将订单置为无效
            DBTimeProfile.release();
            return;
        }

        // 同步风控校验通过
        // 发送短信
        SmsTypeEnum smsType = orderSmsService.sendMessage(orderTmpDetailDto);

        // 判断当前订单是否需要进行异步手机号校验
        boolean needPhoneCheck = orderStrategyService.needCheckPhoneStatues(orderTmpDetailDto.getMerchantId(), StrategyType.FILTER_ORDER_RULE);

        if (smsType != null && smsType.isNeedReport() && needPhoneCheck) {
            // 需要手机号校验 且 短信通道可以推送消息报告时  流程阻塞
            LOGGER.info("需要手机号校验 且 短信通道可以推送消息报告时 流程暂时结束,orderId,{}", orderId);
        } else {
            // 不需要手机号校验 或 短信通道不推送消息报告时   继续走业务校验
            DBTimeProfile.enter("业务校验");
            boolean result = orderTmpService.bizValidate(orderTmpDetailDto);
            DBTimeProfile.release();
            if (result) {
                // 业务校验成功 发送订单变更为正式订单事件
                OrderTmpToFormalEvent toFormalEvent = new OrderTmpToFormalEvent();
                toFormalEvent.setOrderId(orderId);
                eventBus.post(toFormalEvent);
                LOGGER.info("临时订单创建事件将订单变更为正式订单，orderId： {} ", orderId);
            }
        }

        DBTimeProfile.release();
    }

    /**
     * 黑名单校验
     * @param orderTmpDetailDto
     * @return   true-通过 false-不通过
     */
    private boolean blackListCheck(OrderTmpDetailDto orderTmpDetailDto){
        BlackListCheckParam blackListCheckParam = new BlackListCheckParam();
        blackListCheckParam.setItemId(orderTmpDetailDto.getItemId());
        blackListCheckParam.setMobile(orderTmpDetailDto.getMobile());
        blackListCheckParam.setMerchantId(orderTmpDetailDto.getMerchantId());
        BlackListCheckDto blackListCheckDto = remoteShieldBlackListService.checkBlackList(blackListCheckParam);
        return blackListCheckDto.getCheckResult();
    }

    /**
     * 过滤规则校验
     * @param orderTmpDetailDto
     * @return true - 命中规则     false - 通过
     */
    private boolean filterOrderCheck(OrderTmpDetailDto orderTmpDetailDto){
        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(orderTmpDetailDto, OrderSnapshotDto.class);
        FilterOrderCheckDto filterOrderCheckDto = new FilterOrderCheckDto();
        filterOrderCheckDto.setOrderSnapshotDto(orderSnapshotDto);
        filterOrderEngine.process(filterOrderCheckDto);
        return filterOrderCheckDto.isCheckResult();
    }


    /**
     * 处理 临时订单变更为正式订单事件
     * @param event
     */
    @Subscribe
    public void processOrderTmpToFormal(OrderTmpToFormalEvent event){
        String orderId = event.getOrderId();
        OrderTmpDetailDto orderTmpDetailDto = orderTmpService.findByOrderId(orderId);

        if (null == orderTmpDetailDto) {
            LOGGER.warn("临时订单变更为正式订单失败，临时订单不存在，orderId：{}", orderId);
            return ;
        }

        // 创建正式订单
        orderService.createOrderFromTmp(buildOrder(orderTmpDetailDto), buildChannel(orderTmpDetailDto), orderTmpDetailDto.getPayType());

    }

    /***
     * 处理创蓝消息推送事件
     * @param event
     */
    @Subscribe
    public void chuangLanReportReceiveEvent(ChuangLanReportReceiveEvent event){
        ChuangLanSmsReportDto smsReport = event.getChuangLanSmsReportDto();
        BaiqiOrderSmsEntity orderSmsEntity = event.getOrderSmsEntity();
        if (smsReport == null || orderSmsEntity == null) {
            return ; // 查不到数据，结束流程
        }

        //  临时订单只有是在待过滤状态下才进行手机号校验
        OrderTmpDetailDto orderTmpDetailDto = orderTmpService.findByOrderId(orderSmsEntity.getOrderId());
        if (null == orderTmpDetailDto
                || orderTmpDetailDto.getOrderStatus() != OrderTmpStatusEnum.TO_FILTER.getNum()) {
            return;
        }

        // 进行手机号校验
        boolean phoneCheckResult = orderStrategyService.checkPhoneStatusForFilterRule(orderTmpDetailDto);
        if (!phoneCheckResult) {
            // 手机号校验不通过  订单置为无效
            orderTmpService.updateOrderTmpToInvalid(orderTmpDetailDto.getOrderId(), "手机号状态校验不通过", OrderTmpRiskCheckFailEnum.FILTER_ORDER_RULE_FAIL);
            return;
        }

        // 进行业务校验
        boolean bizCheckResult = orderTmpService.bizValidate(orderTmpDetailDto);
        if (!bizCheckResult) {
            // 业务校验不通过  订单置为无效
            return;
        }

        // 业务校验成功 发送订单变更为正式订单事件
        OrderTmpToFormalEvent toFormalEvent = new OrderTmpToFormalEvent();
        toFormalEvent.setOrderId(orderTmpDetailDto.getOrderId());
        eventBus.post(toFormalEvent);
        LOGGER.info("接收创蓝消息后将临时订单变更为正式订单，orderId： {} ", orderTmpDetailDto.getOrderId());
    }

    /**
     * 构建订单数据
     * @param orderTmpDetailDto
     * @return
     */
    private OrderDto buildOrder(OrderTmpDetailDto orderTmpDetailDto){
        OrderDto order = BeanUtils.copy(orderTmpDetailDto, OrderDto.class);
        OrderItemDto orderItem = BeanUtils.copy(orderTmpDetailDto, OrderItemDto.class);
        orderItem.setCostPrice(orderTmpDetailDto.getItemCost());
        order.setOrderItem(orderItem);
        return order;
    }

    /**
     * 构建渠道信息
     * @param orderTmpDetailDto
     * @return
     */
    private ChannelInfoDto buildChannel(OrderTmpDetailDto orderTmpDetailDto){
        ChannelInfoDto channel = BeanUtils.copy(orderTmpDetailDto, ChannelInfoDto.class);
        channel.setaCid(orderTmpDetailDto.getTuiaCid());

        // update note: 如果web层有传appid则直接用，如果链接中不带有appid调推啊的接口获取
        if (orderTmpDetailDto.getAppId() == null) {
            channel.setAppId(queryAppIdByTuiaId(orderTmpDetailDto.getTuiaId()));
        }

        return channel;
    }


    /**
     * 根据tuiaId获取appId
     * @param tuiaId
     * @return
     */
    private Long queryAppIdByTuiaId(String tuiaId) {
        if (StringUtils.isEmpty(tuiaId)) {
            return null;
        }

        try {
            // 截取 tuiaId中taw-后面的部分做为tuia orderId
            String tuiaOrderId = StringUtils.substringAfter(tuiaId, "taw-");
            Long consumerId = remoteActivityOrderService.selectConsumerIdByOrderId(tuiaOrderId);

            ActivityOrderDto activityOrderDto = remoteActivityOrderService.selectByOrderIdAndConsumerId(tuiaOrderId, consumerId).getResult();
            if (null != activityOrderDto) {
                return activityOrderDto.getAppId();
            }
            return null;
        } catch (Exception e) {
            LOGGER.warn("根据TuiaId获取AppId失败, tuiaId={}", tuiaId, e);
            return null;
        }
    }





}
