package com.qiho.center.biz.bo;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.qiho.center.api.constant.ConfigConstant;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.ErpManufacturerType;
import com.qiho.center.api.enums.ErpStatusEnum;
import com.qiho.center.api.enums.ErpTypeEnum;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.OrderApprovalStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.service.ConfigService;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderItemInfoDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderItemInfoEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.ERPClientUtil;
import com.qiho.center.common.util.EdbLib;

import cn.com.duiba.wolf.utils.DateUtils;

/**
 * Created by liuyao on 2017/6/12.
 */
@Service
@BizEventListener
public class ErpOrderBo {

    private static final Logger      LOG = LoggerFactory.getLogger(ErpOrderBo.class);

    @Autowired
    private ErpOrderService          erpOrderService;
    @Autowired
    private OrderBo                  orderBo;
    @Autowired
    private OrderService             orderService;
    @Autowired
    private QihoOrderSnapshotDAO     qihoOrderSnapshotDAO;
    @Autowired
    private QihoOrderItemInfoDAO     qihoOrderItemInfoDAO;
    @Autowired
    private LogisticsOrderService    logisticsOrderService;
    @Autowired
    private FundOrderService         fundOrderService;
    @Resource
    private ExecutorService          executorService;
    @Value("${qiho.erp.shop.no}")
    private String                   shopNo;
    @Value("${qiho.erp.auto.reliver.url}")
    private String                   erpAutoReliverUrl;

    private ListeningExecutorService service;

    @Autowired
    private EdbLib edbLib;

    @Autowired
    private ConfigService configService;
    @Autowired
    private ExpressBo expressBo;

    @PostConstruct
    public void init() {
        service = MoreExecutors.listeningDecorator(executorService);
    }

    /** 手动取消发货文案 */
    private static final  String CANCEL_SEND_TEXT = "手动发货-取消发货";

    private static String numAndLetterRegEx = "^[A-Za-z0-9]{1,40}$";

    @Autowired
    KuaiDi100Bo kuaiDi100Bo;

    /**
     * 取消发货
     *
     * @param erpId
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void cancelDeliver(String erpId) {
        checkEnableErpOrder(erpId);
        ErpOrderDto erpOrderDto = erpOrderService.updateErpStatus(erpId, ErpStatusEnum.FAILED, ErpStatusEnum.TO_NOTIFY,
                "取消发货");
        if (Objects.equal(erpOrderDto.getErpType(), ErpTypeEnum.SEND.getCode())) {
            orderBo.rejectOrder(erpOrderDto.getOrderId(), CANCEL_SEND_TEXT, null);
        }
        refund(erpOrderDto.getOrderId());

    }

    /**
     * @param orderId
     */
    private void refund(String orderId) {

        List<FundOrderDto> list = fundOrderService.findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isEmpty(list)) {
            throw new QihoException("数据异常，付款子订单不存在，orderId=" + orderId);
        }
        FundOrderDto fundOrder = list.get(0);
        if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
            fundOrder.setFundStatus(FundStatusEnum.FAILED.getCode());
            fundOrderService.update(fundOrder);
            QihoOrderSnapshotEntity order = new QihoOrderSnapshotEntity();
            order.setOrderId(orderId);
            order.setFundStatus(FundStatusEnum.FAILED.getCode());
            qihoOrderSnapshotDAO.updataFundInfo(order);
        } else if (!StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())
                && StringUtils.equals(fundOrder.getFundStatus(), FundStatusEnum.SUCCESS.getCode())) {
            fundOrderService.refund(fundOrder.getOrderId(), fundOrder.getPayType(), fundOrder.getAmt());
        }

    }

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    /**
     * 确认发货
     *
     * @param params
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void resolveDeliver(ResolveDeliverParams params) {
        // 忽略大小写的写法
        Pattern pat = Pattern.compile(numAndLetterRegEx, Pattern.CASE_INSENSITIVE);
        Matcher matcher = pat.matcher(params.getExpressCode());
        // 字符串是否与正则表达式相匹配
        boolean rs = matcher.matches();
        if(!rs){//数字和字母的检验
            throw new QihoException("快递单号格式有误，只能是数字或者数字加字母!");
        }

        LogisticsOrderEntity logisticsOrderEntity = qihoLogisticsOrderDAO.findByErpId(params.getErpId());
        if(logisticsOrderEntity != null) {//物流子订单号已经存在  则更新物流单号
            LogisticsOrderEntity entity = new LogisticsOrderEntity();
            entity.setPostId(params.getExpressCode());
            entity.setLogisticsName(params.getBaiqiLogisticsName());
            entity.setLogisticsCode(params.getBaiqiLogisticsCode());
            entity.setErpId(params.getErpId());
            qihoLogisticsOrderDAO.updateByErpId(entity);
            //更新订单快照表信息
            QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
            orderSnapshot.setPostId(params.getExpressCode());
            orderSnapshot.setLogisticsName(params.getBaiqiLogisticsName());
            orderSnapshot.setLogisticsCode(params.getBaiqiLogisticsCode());
            orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());
            qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);

            //重新订阅快递
            List<LogisticsOrderDo> list = Lists.newArrayList();
            LogisticsOrderDo logisticsOrderDo = new LogisticsOrderDo();
            logisticsOrderDo.setLogisticsCode(params.getBaiqiLogisticsCode());
            logisticsOrderDo.setPostId(params.getExpressCode());
            list.add(logisticsOrderDo);
            expressBo.subscribeExpress(list);
        }else{
            checkEnableErpOrder(params.getErpId());
            ErpOrderDto erpOrderDto = erpOrderService.updateErpStatus(params.getErpId(), ErpStatusEnum.SUCCESS,
                    ErpStatusEnum.TO_NOTIFY, null);
            // 创建物流子订单
            logisticsOrderService.initLogisticsOrder(erpOrderDto, params);
        }

    }

    private void checkEnableErpOrder(String erpId){
        ErpOrderDto erpOrder = erpOrderService.findErpOrderByErpId(erpId);
        if (erpOrder == null ) {
            throw new QihoException("订单不存在");
        }
        if (!StringUtils.equals(erpOrder.getErpStatus(), ErpStatusEnum.TO_NOTIFY.getCode())) {
            throw new QihoException("订单非待通知状态");
        }
        if (!StringUtils.equals(erpOrder.getDelivery(), DeliveryEnum.MANUAL.getCode())) {
            throw new QihoException("订单非手动发货方式");
        }
    }

    /**
     * ERP自动发货
     *
     * @author zhanglihui
     * @param erp
     */
    public void autoDelivery(QihoErpOrderEntity erp) {
        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(erp.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_SEND.getCode())) {
            AppLogUtil.warn(LOG, "订单非代发货状态，【orderId={}, orderStatus={}】", order.getOrderId(), order.getOrderStatus());
            return;
        }
        String erpManufacturerType = configService.findByName(ConfigConstant.ERP_MANUFACTURER_TYPE);
        if(Objects.equal(erpManufacturerType, ErpManufacturerType.EDB.getCode())){
            AutoEDianBaoReliverRunnable runnable = new AutoEDianBaoReliverRunnable(erp,order);
            AutoErpEDianBaoFutureCallback callback = new AutoErpEDianBaoFutureCallback();
            callback.setQihoErpOrderEntity(erp);
            ListenableFuture<JSONObject> future = service.submit(runnable);
            Futures.addCallback(future, callback);
        }else if(Objects.equal(erpManufacturerType, ErpManufacturerType.WDT.getCode())){
            AutoErpReliverRunnable runnable = new AutoErpReliverRunnable(erp,order);
            AutoErpReliverFutureCallback callback = new AutoErpReliverFutureCallback();
            callback.setQihoErpOrderEntity(erp);
            ListenableFuture<JSONObject> future = service.submit(runnable);
            Futures.addCallback(future, callback);
        }else {
            LOG.warn("ERP厂商不存在  数据库配置有误:{}",erpManufacturerType);
            return;
        }
    }


    /**
     * E店宝自动发货
     */
    private class AutoEDianBaoReliverRunnable implements Callable<JSONObject> {

        private QihoErpOrderEntity      qihoErpOrderEntity;
        private QihoOrderSnapshotEntity qihoOrderSnapshotEntity;

        public AutoEDianBaoReliverRunnable(QihoErpOrderEntity qihoErpOrderEntity,
                                           QihoOrderSnapshotEntity qihoOrderSnapshotEntity){
            this.qihoErpOrderEntity = qihoErpOrderEntity;
            this.qihoOrderSnapshotEntity = qihoOrderSnapshotEntity;
        }

        @Override
        public JSONObject call() {
            Map<String, String> params = edbLib.edbGetCommonParams("edbTradeAdd");
            params.put("xmlValues", getParams());
            String result = edbLib.edbRequstPost(params);
            return JSONObject.parseObject(result);
        }

        private String getParams() {
            StringBuilder builder = new StringBuilder();
            builder.append("<info>");
            builder.append("<orderInfo>");
            builder.append("<out_tid>"+qihoErpOrderEntity.getErpId()+"</out_tid>");
            builder.append("<shop_id>1</shop_id>");
            builder.append("<buyer_id>"+qihoErpOrderEntity.getConsumerName()+"</buyer_id>");
            builder.append("<buyer_msg>"+qihoOrderSnapshotEntity.getMessage()+"</buyer_msg>");
            builder.append("<consignee>"+qihoErpOrderEntity.getConsumerName()+"</consignee>");
            builder.append("<mobilPhone>"+qihoErpOrderEntity.getMobile()+"</mobilPhone>");
            builder.append("<privince>"+qihoErpOrderEntity.getProvince()+"</privince>");
            builder.append("<city>"+qihoErpOrderEntity.getCity()+"</city>");
            builder.append("<area>"+qihoErpOrderEntity.getDistrict()+"</area>");
            builder.append("<address>"+qihoErpOrderEntity.getAddress()+"</address>");//收货地址，请包含省市区
            // 付款状态:0未付款 1部分付款 2已付款
            String payStatus = "未付款";
            //货到付款=1 已付款=0
            int deliveryTerm = 1;
            if (!StringUtils.equals(qihoOrderSnapshotEntity.getPayType(), PayTypeEnum.COD.getCode())) {
                deliveryTerm = 0;
                payStatus = "已付款";
            }
            float codAmount = (float) qihoOrderSnapshotEntity.getOrderAmt() / 100; //订单金额
            builder.append("<actual_freight_get>0</actual_freight_get>");//实收运费，代收买家的运费，货到付款就是0
            builder.append("<is_COD>"+deliveryTerm+"</is_COD>");
            builder.append("<order_totalMoney>"+codAmount+"</order_totalMoney>");
            builder.append("<product_totalMoney>"+codAmount+"</product_totalMoney>");
            builder.append("<order_date>"+DateUtils.getSecondStr(qihoErpOrderEntity.getGmtCreate())+"</order_date>");
            builder.append("<express>京东快递</express>");
            builder.append("<pay_status>"+payStatus+"</pay_status>");
            builder.append("</orderInfo>");

            QihoOrderItemInfoEntity orderItemInfo = qihoOrderItemInfoDAO.findByOrderId(qihoOrderSnapshotEntity.getOrderId());
            builder.append("<product_info>");
            builder.append("<product_item>");
            builder.append("<barCode>"+orderItemInfo.getSkuNo()+"</barCode>");
            builder.append("<product_title>"+orderItemInfo.getItemName()+"</product_title>");//商品标题
            builder.append("<orderGoods_Num>"+orderItemInfo.getQuantity()+"</orderGoods_Num>");//订货数量
            builder.append("<cost_Price>"+(float) orderItemInfo.getSellingPrice() / 100+"</cost_Price>");//成交单价
            builder.append("<out_tid>"+qihoErpOrderEntity.getErpId()+"</out_tid>");//对应orderInfo中的out_tid表示订单明细
            builder.append("</product_item>");

            builder.append("</product_info>");
            builder.append("</info>");
            return builder.toString();
        }
    }

    /**
     * 自动发货回调
     */
    private class AutoErpEDianBaoFutureCallback implements FutureCallback<JSONObject> {

        private QihoErpOrderEntity qihoErpOrderEntity;

        @Override
        @Transactional(DsConstants.DATABASE_QIHO)
        public void onSuccess(JSONObject result) {
            if (result.get("Success")==null) {
                String errorMessage = "ERP自动发货返回失败";
                erpOrderService.updateErpStatus(qihoErpOrderEntity.getErpId(), ErpStatusEnum.FAILED,
                        ErpStatusEnum.TO_NOTIFY, errorMessage);
                closeOrder(qihoErpOrderEntity.getOrderId(), errorMessage);
                LOG.error("ERP自动发货返回失败:" + result.getString("message"));
            } else {
                erpOrderService.updateErpStatus(qihoErpOrderEntity.getErpId(), ErpStatusEnum.SUCCESS,
                        ErpStatusEnum.TO_NOTIFY, null);
            }
        }

        @Override
        public void onFailure(Throwable t) {
            LOG.error("ERP订单自动发货请求失败,ERP={}", qihoErpOrderEntity.getErpId(), t);
        }

        /**
         * qihoErpOrderEntity.
         *
         * @param qihoErpOrderEntity the qihoErpOrderEntity to set
         */
        public void setQihoErpOrderEntity(QihoErpOrderEntity qihoErpOrderEntity) {
            this.qihoErpOrderEntity = qihoErpOrderEntity;
        }

    }

    /**
     * 旺店通自动发货
     */
    private class AutoErpReliverRunnable implements Callable<JSONObject> {

        private QihoErpOrderEntity      qihoErpOrderEntity;
        private QihoOrderSnapshotEntity qihoOrderSnapshotEntity;

        public AutoErpReliverRunnable(QihoErpOrderEntity qihoErpOrderEntity,QihoOrderSnapshotEntity qihoOrderSnapshotEntity){
            this.qihoErpOrderEntity = qihoErpOrderEntity;
            this.qihoOrderSnapshotEntity = qihoOrderSnapshotEntity;
        }

        @Override
        public JSONObject call() {
            Map<String, String> params = getParams();
            String result = ERPClientUtil.execute(erpAutoReliverUrl, params);
            return JSONObject.parseObject(result);
        }

        private Map<String, String> getParams() {
            Map<String, String> parmas = Maps.newHashMap();
            parmas.put("shop_no", shopNo);

            JSONArray tradeList = new JSONArray();
            JSONObject tradeItem = new JSONObject();
            tradeItem.put("tid", qihoErpOrderEntity.getErpId());
            tradeItem.put("trade_status", 30);
            // 付款状态:0未付款 1部分付款 2已付款
            int payStatus = 0;
            // 发货条件:1款到发货 2货到付款(包含部分货到付款) 3分期付款
            int deliveryTerm = 2;
            // 货到付款金额
            float codAmount = (float) qihoOrderSnapshotEntity.getOrderAmt() / 100;
            // 已付金额
            float paid = 0;
            if (!StringUtils.equals(qihoOrderSnapshotEntity.getPayType(), PayTypeEnum.COD.getCode())) {
                // 在线支付
                payStatus = 2;
                deliveryTerm = 1;
                codAmount = 0;
                paid = (float) qihoOrderSnapshotEntity.getOrderAmt() / 100;
            }
            tradeItem.put("pay_status", payStatus);
            tradeItem.put("delivery_term", deliveryTerm);
            tradeItem.put("trade_time", DateUtils.getSecondStr(qihoErpOrderEntity.getGmtCreate()));
            tradeItem.put("buyer_nick", qihoErpOrderEntity.getConsumerName());
            tradeItem.put("receiver_name", qihoErpOrderEntity.getConsumerName());
            tradeItem.put("receiver_province", qihoErpOrderEntity.getProvince());
            tradeItem.put("receiver_city", qihoErpOrderEntity.getCity());
            tradeItem.put("receiver_district", qihoErpOrderEntity.getDistrict());
            tradeItem.put("receiver_address", qihoErpOrderEntity.getAddress());
            tradeItem.put("receiver_mobile", qihoErpOrderEntity.getMobile());
            tradeItem.put("logistics_type", 4);// 文档中4代表圆通速递
            tradeItem.put("buyer_message", qihoOrderSnapshotEntity.getMessage());
            tradeItem.put("post_amount", 0);

            tradeItem.put("cod_amount", codAmount);
            tradeItem.put("ext_cod_fee", 0);
            tradeItem.put("other_amount", 0);
            tradeItem.put("paid", paid);

            JSONArray orderList = new JSONArray();
            JSONObject orderItem = new JSONObject();
            QihoOrderItemInfoEntity orderItemInfo = qihoOrderItemInfoDAO.findByOrderId(qihoOrderSnapshotEntity.getOrderId());
            orderItem.put("oid", qihoErpOrderEntity.getErpId() + "-1");
            orderItem.put("num", orderItemInfo.getQuantity());
            orderItem.put("price", (float) orderItemInfo.getSellingPrice() / 100);
            orderItem.put("status", 10);
            orderItem.put("refund_status", 0);
            orderItem.put("goods_id", orderItemInfo.getItemId());
            orderItem.put("spec_id", orderItemInfo.getSkuId());
            orderItem.put("goods_no", orderItemInfo.getItemNo());
            orderItem.put("goods_name", orderItemInfo.getItemName());
            orderItem.put("spec_no", orderItemInfo.getSkuNo());
            orderItem.put("spec_name", orderItemInfo.getSkuName());
            orderItem.put("cid", 1);// 无用
            orderList.add(orderItem);
            tradeItem.put("order_list", orderList);
            tradeList.add(tradeItem);
            parmas.put("trade_list", tradeList.toJSONString());
            return parmas;
        }
    }

    /**
     * 自动发货回调
     */
    private class AutoErpReliverFutureCallback implements FutureCallback<JSONObject> {

        private QihoErpOrderEntity qihoErpOrderEntity;

        @Override
        @Transactional(DsConstants.DATABASE_QIHO)
        public void onSuccess(JSONObject result) {
            if (result.getInteger("code") != 0) {
                String errorMessage = "ERP自动发货返回失败";
                erpOrderService.updateErpStatus(qihoErpOrderEntity.getErpId(), ErpStatusEnum.FAILED,
                        ErpStatusEnum.TO_NOTIFY, errorMessage);
                closeOrder(qihoErpOrderEntity.getOrderId(), errorMessage);
                LOG.error("ERP自动发货返回失败:" + result.getString("message"));
            } else {
                erpOrderService.updateErpStatus(qihoErpOrderEntity.getErpId(), ErpStatusEnum.SUCCESS,
                        ErpStatusEnum.TO_NOTIFY, null);
            }
        }

        @Override
        public void onFailure(Throwable t) {
            LOG.error("ERP订单自动发货请求失败,ERP={}", qihoErpOrderEntity.getErpId(), t);
        }

        /**
         * qihoErpOrderEntity.
         *
         * @param qihoErpOrderEntity the qihoErpOrderEntity to set
         */
        public void setQihoErpOrderEntity(QihoErpOrderEntity qihoErpOrderEntity) {
            this.qihoErpOrderEntity = qihoErpOrderEntity;
        }

    }

    /**
     * erp订单审核失败
     *
     * @author chencheng
     * @param param
     * @return
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer reviewFaildErpOrder(ErpOrderDto dto) {
        LogisticsOrderDto logisticsOrderDto = logisticsOrderService.findByErpId(dto.getErpId());
        // 物流信息存在，不改变
        if (logisticsOrderDto != null) {
            return 0;
        }
        ErpOrderDto updatedto = erpOrderService.updateErpStatus(dto.getErpId(), ErpStatusEnum.FAILED,
                ErpStatusEnum.SUCCESS, dto.getFailedMessage());
        if (updatedto == null) {
            return 0;
        }
        closeOrder(updatedto.getOrderId(), dto.getFailedMessage());
        return 1;
    }

    private void closeOrder(String orderId, String remark) {
        // 变更主订单状态
        orderBo.rejectOrder(orderId, remark, OrderApprovalStatusEnum.APPROVE_FAILD);
        // 退款
        refund(orderId);
    }

}
