package com.qiho.center.biz.bo;

import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.enums.ExpressBirdLogisticsStatusEnum;
import com.qiho.center.api.enums.ExpressBirdOrderStatusEnum;
import com.qiho.center.api.enums.logistics.LogisticsExpressPlatformEnum;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.logistics.ExpressBirdService;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.biz.service.logistics.LogisticsSubscribeService;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiExpressCodeMappingEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsSubscribeEntity;
import com.qiho.center.common.params.kuaidiniao.ExpressResult;
import com.qiho.center.common.params.kuaidiniao.ExpressResultItem;
import com.qiho.center.common.util.ExpressBirdUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;


/**
 * 快递鸟
 *
 * @author: wangjin
 * @create 2018-06-07 10:43
 **/
@Service
public class ExpressBirdBo {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExpressBirdBo.class);

    @Resource
    private ExpressBirdService expressBirdService;
    @Resource
    private LogisticsService logisticsService;
    @Resource
    private LogisticsOrderService logisticsOrderService;
    @Resource
    private LogisticsSubscribeService subscribeService;

    /**
     * 控制流速 预热启动
     */
    private static RateLimiter rateLimiter = RateLimiter.create(30,1L,TimeUnit.SECONDS);
    /**
     * 处理快递鸟推送过来的消息
     *
     * @param code   快递鸟的快递公司编码
     * @param postId 物流单号
     * @param status 快递鸟的状态，数字编码，例：311，4，401...
     * @return
     */
    public boolean dealExpressBirdStatus(String code, String postId, String status) {

        LOGGER.info("快递鸟 推送信息：快递公司编码：{} ， 快递单号: {} ，快递单当前状态: {}", code, postId, status);
        LogisticsOrderDto dto = new LogisticsOrderDto();

        ExpressBirdLogisticsStatusEnum expressBirdLogisticsStatusEnum = ExpressBirdLogisticsStatusEnum.getByState(status);
        if (expressBirdLogisticsStatusEnum != null) {
            dto.setLogisticsStatus(expressBirdLogisticsStatusEnum.getDesc());
        } else {
            dto.setLogisticsStatus(status);
        }

        ExpressBirdOrderStatusEnum orderStatusEnum = ExpressBirdOrderStatusEnum.getByExpressBird(status);
        if (orderStatusEnum != null) {
            dto.setOrderStatus(orderStatusEnum.getStatus());
        } else {
            return true;
        }

        dto.setPostId(postId);

        BaiqiExpressCodeMappingEntity codeMappingEntity = expressBirdService.findByBaiqiLogisticsCode(code,
                LogisticsExpressPlatformEnum.BIRD.getCode());
        if (codeMappingEntity == null) {
            LOGGER.error("快递鸟推送的快递公司代码表中不存在:", code);
            return false;
        }
        BaiqiLogisticsEntity baiqiLogisticsEntity = logisticsService
                .findById(codeMappingEntity.getLogisticsId());
        if (baiqiLogisticsEntity == null) {
            LOGGER.error("快递鸟推送的快递公司所映射的百奇物流公司不存在:", codeMappingEntity.getLogisticsId());
            return false;
        }
        dto.setLogisticsCode(baiqiLogisticsEntity.getLogisticsCode());

        logisticsOrderService.update(dto);
        return true;
    }


    /**
     * 快递鸟订阅
     *
     * @param list
     */
    public void dealPostOrders(List<LogisticsOrderDo> list) {

        //以每30单为一组
        List<List<LogisticsOrderDo>> groupList = Lists.partition(list, 30);

        BaiqiExpressCodeMappingEntity expressLogisticsEntity;

        for (List<LogisticsOrderDo> listItem : groupList) {

            for (LogisticsOrderDo item : listItem) {

                expressLogisticsEntity = expressBirdService.findByBaiqiLogisticsCode(item.getLogisticsCode(),
                        LogisticsExpressPlatformEnum.BIRD.getCode());
                if (expressLogisticsEntity == null) {
                    LOGGER.error("在快递鸟对应的物流中不存在百奇快递公司代码:{},{}", item.getLogisticsCode(),item.getPostId());
                } else {
                    //订阅对象
                    BaiqiLogisticsSubscribeEntity subscribeEntity = new BaiqiLogisticsSubscribeEntity();
                    subscribeEntity.setLogisticsCode(item.getLogisticsCode());
                    subscribeEntity.setExpressPlatform(LogisticsExpressPlatformEnum.BIRD.getName());
                    subscribeEntity.setPostId(item.getPostId());
                    //保存
                    saveOrder(subscribeEntity);

                    //限流 30单/s
                    rateLimiter.acquire();
                    //订阅
                    subscribeEntity.setLogisticsCode(expressLogisticsEntity.getLogisticsCode());
                    postOrder(subscribeEntity);
                }

            }

        }
    }

    /**
     * 保存订阅
     * @param subscribeEntity
     */
    private void saveOrder(BaiqiLogisticsSubscribeEntity subscribeEntity){

        //订阅之前先保存
        int count = subscribeService.countLogisticsSubscribe(subscribeEntity.getPostId());
        if (count <= 0) {
            subscribeService.insertLogisticsSubscribe(subscribeEntity);
        }
    }

    /**
     * 订阅处理
     * @param subscribeEntity
     */
    private void postOrder(BaiqiLogisticsSubscribeEntity subscribeEntity) {

        //订阅轨迹
        boolean flag = ExpressBirdUtil.postOrder(subscribeEntity.getLogisticsCode(), subscribeEntity.getPostId());

        //订阅成功
        if (flag) {
            subscribeService.changeLogisticsSubscribe(subscribeEntity);
        }
    }

    /**
     * 快递鸟即时查询
     *
     * @param logisticsCode
     * @param postId
     * @return
     */
    public List<LogisticsProcessDto> queryWaybillTrace(String logisticsCode, String postId) {
        if (StringUtils.isBlank(postId)) {
            return Collections.emptyList();
        }

        LOGGER.info("快递鸟 即时查询：快递公司编码：{} ， 快递单号: {} ", logisticsCode, postId);

        // 查询快递100公司信息
        BaiqiExpressCodeMappingEntity expressLogisticsEntity = expressBirdService.findByBaiqiLogisticsCode(logisticsCode,
                LogisticsExpressPlatformEnum.BIRD.getCode());
        if (expressLogisticsEntity == null) {
            LOGGER.error("获取不到快递鸟物流公司信息");
            return Collections.emptyList();
        }

        List<LogisticsProcessDto> resultList = null;
        try {

            // 获取快递鸟查询结果
            ExpressResult result = ExpressBirdUtil.queryWaybillTrace(expressLogisticsEntity.getLogisticsCode(), postId);


            if (result != null && CollectionUtils.isNotEmpty(result.getTraces())) {

                //快递鸟走单信息
                List<ExpressResultItem> list = result.getTraces();

                resultList = Lists.newArrayListWithCapacity(list.size());
                for (ExpressResultItem item : list) {
                    LogisticsProcessDto logisticsProcessDto = new LogisticsProcessDto();
                    logisticsProcessDto.setWaybillNo(postId);
                    logisticsProcessDto.setUploadTime(DateFormatUtils.format(item.getAcceptTime(), "yyyy-MM-dd HH:mm:ss"));
                    logisticsProcessDto.setProcessInfo(item.getAcceptStation());
                    resultList.add(logisticsProcessDto);
                }
            }
        } catch (Exception e) {
            LOGGER.warn("快递鸟运单走件流程异常， postId = " + postId, e);
        }

        return resultList;
    }

    /**
     * 即时查询状态
     * @param logisticsCode
     * @param postId
     * @return
     */
    public void queryStatus(String logisticsCode, String postId, String orderId){
        String status = "404";
        if (StringUtils.isBlank(postId)) {
            status = "404";
        }

        LOGGER.info("快递鸟 修复 即时查询：快递公司编码：{} ， 快递单号: {} ", logisticsCode, postId);

        // 查询快递100公司信息
        BaiqiExpressCodeMappingEntity expressLogisticsEntity = expressBirdService.findByBaiqiLogisticsCode(logisticsCode,
                LogisticsExpressPlatformEnum.BIRD.getCode());
        if (expressLogisticsEntity == null) {
            LOGGER.error("获取不到快递鸟物流公司信息");
            status = "404";
        }


        try {

            // 获取快递鸟查询结果
            ExpressResult result = ExpressBirdUtil.queryWaybillTrace(expressLogisticsEntity.getLogisticsCode(), postId);

            if (result != null) {
                status = result.getStateEx();
            }
            LOGGER.info("快递鸟 查询 {}",result);

        } catch (Exception e) {
            LOGGER.warn("快递鸟运单走件流程异常， postId = " + postId, e);
        }

        // 更新订单
        dealExpressBirdStatus2(logisticsCode,postId,status,orderId);
    }

    /**
     * 处理快递鸟 即时查询的消息
     *
     * @param code   百奇的快递公司编码
     * @param postId 物流单号
     * @param status 快递鸟的状态
     * @return
     */
    public boolean dealExpressBirdStatus2(String code, String postId, String status, String orderId) {

        LOGGER.info("快递鸟 修复 推送信息：快递公司编码：{} ， 快递单号: {} ，快递单当前状态: {}", code, postId, status);
        LogisticsOrderEntity entity = new LogisticsOrderEntity();

        ExpressBirdLogisticsStatusEnum expressBirdLogisticsStatusEnum = ExpressBirdLogisticsStatusEnum.getByState(status);
        if (expressBirdLogisticsStatusEnum != null) {
            entity.setLogisticsStatus(expressBirdLogisticsStatusEnum.getDesc());
        } else {
            entity.setLogisticsStatus(status);
        }

        ExpressBirdOrderStatusEnum orderStatusEnum = ExpressBirdOrderStatusEnum.getByExpressBird(status);
        if (orderStatusEnum != null) {
            entity.setOrderStatus(orderStatusEnum.getStatus());
        } else {
            return true;
        }

        entity.setPostId(postId);

        entity.setLogisticsCode(code);
        entity.setOrderId(orderId);
        logisticsOrderService.updateStatus(entity);
        return true;
    }

}
