package com.qiho.center.biz.bo;

import com.google.common.collect.Lists;
import com.qiho.center.api.constant.ConfigConstant;
import com.qiho.center.api.dto.ConfigDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.enums.logistics.LogisticsExpressPlatformEnum;
import com.qiho.center.biz.bo.domain.ExpressPlatformDecideResultBean;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.service.ConfigService;
import com.qiho.center.biz.service.logistics.ExpressSubscribeService;
import com.qiho.center.common.constant.ExpressEnvironment;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutorService;

/**
 * 快递工具
 *
 * @author: wangjin
 * @create 2018-06-11 17:13
 **/
@Service
public class ExpressBo {

    /** 快递鸟不支持的物流公司 顺丰，百世，德创 */
    private static final List<String> EXPRESSBIRD_UPSUPPORT_LOGISTICS_CODE = Lists.newArrayList("SF", "BEST_EX", "BTWL", "DCWL");

    @Autowired
    private KuaiDi100Bo kuaiDi100Bo;

    @Autowired
    private ExpressBirdBo expressBirdBo;

    @Autowired
    private ExpressEnvironment expressEnvironment;

    @Autowired
    private ConfigService configService;

    @Autowired
    private ExpressSubscribeService expressSubscribeService;

    @Autowired
    private ExecutorService executorService;

    /**
     * 选择物流平台，订阅
     *
     * @param list
     * @return
     */
    public void subscribeExpress(List<LogisticsOrderDo> list) {

        // 选择快递订阅平台
        ExpressPlatformDecideResultBean decideReault = decideExpressPlatform(list);

        // 向快递鸟订阅
        List<LogisticsOrderDo> birdOrder = decideReault.getExpressBirdList();
        if (CollectionUtils.isNotEmpty(birdOrder)) {
            // 进行订阅
            expressBirdBo.dealPostOrders(birdOrder);
            // 保存订阅信息
            executorService.submit(() -> expressSubscribeService.saveSubscribeData(birdOrder, LogisticsExpressPlatformEnum.BIRD));
        }

        // 向快递100订阅
        List<LogisticsOrderDo> hundredOrder = decideReault.getExpressHundredList();
        if (CollectionUtils.isNotEmpty(hundredOrder)) {
            kuaiDi100Bo.dealpostOrders(hundredOrder);
            executorService.submit(() -> expressSubscribeService.saveSubscribeData(hundredOrder, LogisticsExpressPlatformEnum.HUNDRED));
        }
    }


    /**
     * 选择物流平台，即时查询
     *
     * @param logisticsCode
     * @param postId
     * @return
     */
    public List<LogisticsProcessDto> queryWaybillTrace(String logisticsCode, String postId) {
        if (expressEnvironment.isExpressBird() && checkExpress(logisticsCode)) {
            List<LogisticsProcessDto> list = expressBirdBo.queryWaybillTrace(logisticsCode, postId);
            if (CollectionUtils.isEmpty(list)){
                return Collections.emptyList();
            }
            Collections.reverse(list);
            return list;
        }

        return kuaiDi100Bo.queryWaybillTrace(logisticsCode, postId);
    }

    /**
     * 即时查询状态
     * @param logisticsCode
     * @param postId
     * @param orderId
     * @return
     */
    public void queryStatus(String logisticsCode, String postId,String orderId){

        // 目前只走快递 100 做修复
        kuaiDi100Bo.queryStatus(logisticsCode, postId,orderId);

    }

    /**
     * 判断快递公司
     *
     * @param code
     * @return
     */
    private boolean checkExpress(String code) {
        //百世，德创，顺丰
        return !("BTWL".equals(code) || "BEST_EX".equals(code) || "SF".equals(code) || "DCWL".equals(code));
    }


    /**
     * 快递订阅平台决策
     *
     * @param list
     * @return
     */
    private ExpressPlatformDecideResultBean decideExpressPlatform(List<LogisticsOrderDo> list) {

        ExpressPlatformDecideResultBean resultBean = new ExpressPlatformDecideResultBean();
        if (CollectionUtils.isEmpty(list)) {
            return resultBean;
        }

        // 获取快递鸟的流量比例
        double birdSubscribeRate = 0;
        ConfigDto configDto = configService.fetchConfigByName(ConfigConstant.EXPRESSBIRD_SUBSCRIBE_RATE);
        if (configDto != null) {
            birdSubscribeRate = Double.valueOf(configDto.getConfigValue());
        }

        for (LogisticsOrderDo logisticsOrderDo : list) {

            // 如果快递鸟不支持该物流  则使用快递100
            if (EXPRESSBIRD_UPSUPPORT_LOGISTICS_CODE.contains(logisticsOrderDo.getLogisticsCode())) {
                resultBean.addHundredOrder(logisticsOrderDo);
                continue;
            }

            double randomNum = new Random().nextDouble();
            if (randomNum < birdSubscribeRate) {
                resultBean.addBirdOrder(logisticsOrderDo);
            } else {
                resultBean.addHundredOrder(logisticsOrderDo);
            }
        }

        return resultBean;
    }




}
