package com.qiho.center.biz.bo;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.enums.KuaiDi100LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.KuaiDi100OrderStatusEnum;
import com.qiho.center.api.enums.logistics.LogisticsExpressPlatformEnum;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.logistics.Express100LogisticsService;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.biz.service.logistics.LogisticsSubscribeService;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsSubscribeEntity;
import com.qiho.center.common.entityd.qiho.logistics.Express100LogisticsEntity;
import com.qiho.center.common.params.kuaidi100.Result;
import com.qiho.center.common.params.kuaidi100.ResultItem;
import com.qiho.center.common.params.kuaidi100.TaskResponse;
import com.qiho.center.common.util.KuaiDi100Util;

/**
 * 快递100
 *
 * @author: zhanghuifeng
 * Date:   11:58 2017/11/22
 */
@Service
public class KuaiDi100Bo {

    private static final Logger LOGGER = LoggerFactory.getLogger(KuaiDi100Bo.class);

    @Resource
    private LogisticsOrderService logisticsOrderService;
    @Resource
    private Express100LogisticsService express100LogisticsService;
    @Resource
    private LogisticsService logisticsService;
    @Resource
    private LogisticsSubscribeService subscribeService;

    /**
     * 控制流速 预热启动
     */
    private static RateLimiter rateLimiter = RateLimiter.create(20,1L,TimeUnit.SECONDS);

    /**
     * 处理快递100物流推送的信息
     *
     * @param com    快递100中快递公司编码
     * @param postId 快递单号
     * @param status 快递单当前状态
     * @return boolean
     */
    public Boolean dealKuaiDi100Status(String com, String postId, String status, String condition) {

        LOGGER.info("快递100 推送信息：快递公司编码：{}， 快递单号: {} ，快递单当前状态: {}, 快递单明细状态标记:{}", com, postId, status, condition);

        LogisticsOrderDto dto = new LogisticsOrderDto();
        KuaiDi100LogisticsOrderStatusEnum kuaiDi100LogisticsOrderStatusEnum = KuaiDi100LogisticsOrderStatusEnum
                .getByState(status);
        if (kuaiDi100LogisticsOrderStatusEnum != null) {
            dto.setLogisticsStatus(kuaiDi100LogisticsOrderStatusEnum.getDesc());
        } else {
            dto.setLogisticsStatus(status);
        }
        KuaiDi100OrderStatusEnum statusEnum = KuaiDi100OrderStatusEnum.getByKuaiDi100Status(status);
        if (statusEnum == null) {
            return true;
        }
        //快递100拒签状态特殊处理
        if (statusEnum == KuaiDi100OrderStatusEnum.UNDER_WAY && "JQ001".equals(condition)) {
            dto.setOrderStatus(KuaiDi100OrderStatusEnum.REJECTED.getStatus());
        } else {
            dto.setOrderStatus(statusEnum.getStatus());
        }
        dto.setPostId(postId);
        Express100LogisticsEntity express100LogisticsEntity = express100LogisticsService.findByCode(com);
        if (express100LogisticsEntity == null) {
            LOGGER.error("快递100推送的快递公司代码表中不存在:", com);
            return false;
        }
        BaiqiLogisticsEntity baiqiLogisticsEntity = logisticsService
                .findById(express100LogisticsEntity.getBaiqiLogisticsId());
        if (baiqiLogisticsEntity == null) {
            LOGGER.error("快递100推送的快递公司所映射的百奇物流公司不存在:", express100LogisticsEntity.getBaiqiLogisticsId());
            return false;
        }
        dto.setLogisticsCode(baiqiLogisticsEntity.getLogisticsCode());
        logisticsOrderService.update(dto);
        return true;
    }


    /**
     * 向快递100订阅
     *
     * @param list
     */
    public void dealpostOrders(List<LogisticsOrderDo> list) {
        //以每20单为一组
        List<List<LogisticsOrderDo>> groupList = Lists.partition(list, 20);

        Express100LogisticsEntity express100LogisticsEntity = null;

        for (List<LogisticsOrderDo> listFinal : groupList) {

            for (LogisticsOrderDo logisticsOrderDo : listFinal) {
                express100LogisticsEntity = express100LogisticsService
                        .findByBaiqiLogisticsCode(logisticsOrderDo.getLogisticsCode());
                if (express100LogisticsEntity == null) {
                    LOGGER.error("在快递100物流表中不存在百奇快递公司代码:{},{}", logisticsOrderDo.getLogisticsCode(),logisticsOrderDo.getPostId());
                } else {
                    //订阅对象
                    BaiqiLogisticsSubscribeEntity subscribeEntity = new BaiqiLogisticsSubscribeEntity();
                    subscribeEntity.setLogisticsCode(logisticsOrderDo.getLogisticsCode());
                    subscribeEntity.setExpressPlatform(LogisticsExpressPlatformEnum.HUNDRED.getName());
                    subscribeEntity.setPostId(logisticsOrderDo.getPostId());
                    saveOrder(subscribeEntity);


                    //限流
                    rateLimiter.acquire();
                    //订阅
                    subscribeEntity.setLogisticsCode(express100LogisticsEntity.getLogisticsCode());
                    postOrder(subscribeEntity);
                }
            }
        }
    }

    private void saveOrder(BaiqiLogisticsSubscribeEntity subscribeEntity){
        try {
            //订阅之前先保存
            int count = subscribeService.countLogisticsSubscribe(subscribeEntity.getPostId());
            if (count <= 0) {
                subscribeService.insertLogisticsSubscribe(subscribeEntity);
            }
        }catch (Exception e){
            LOGGER.error("订阅保存失败 {}",subscribeEntity.getPostId(),e);
        }

    }

    /**
     * 快递100订阅
     * @param subscribeEntity
     */
    private void postOrder(BaiqiLogisticsSubscribeEntity subscribeEntity) {

        try {
            //订阅轨迹
            TaskResponse response = KuaiDi100Util.postOrder(subscribeEntity.getLogisticsCode(), subscribeEntity.getPostId());
            //订阅成功
            if (response != null && StringUtils.isNotBlank(response.getReturnCode()) && !response.getReturnCode().equals("500")) {
                LOGGER.info("快递订阅 {},{}",subscribeEntity.getPostId(),response.getReturnCode());
                subscribeService.changeLogisticsSubscribe(subscribeEntity);
            }
        }catch (Exception e){
            LOGGER.error("快递订阅失败 {}",subscribeEntity.getPostId(),e);
        }
    }

    /**
     * 快递100即时查询
     *
     * @param logisticsCode
     * @param postId
     * @return
     */
    public List<LogisticsProcessDto> queryWaybillTrace(String logisticsCode, String postId) {
        if (StringUtils.isBlank(postId)) {
            return Collections.emptyList();
        }

        LOGGER.info("快递100 即时查询：快递公司编码：{}， 快递单号: {} ", logisticsCode, postId);

        // 查询快递100公司信息
        Express100LogisticsEntity express100LogisticsEntity = express100LogisticsService.findByBaiqiLogisticsCode(logisticsCode);
        if (express100LogisticsEntity == null) {
            LOGGER.error("获取不到快递100物流公司信息");

            return Collections.emptyList();
        }

        List<LogisticsProcessDto> resultList = null;
        try {

            // 获取快递100查询结果
            Result result = KuaiDi100Util.queryWaybillTrace(express100LogisticsEntity.getLogisticsCode(), postId);

            if (result != null && CollectionUtils.isNotEmpty(result.getData())) {

                //
                List<ResultItem> list = result.getData();

                resultList = Lists.newArrayListWithCapacity(list.size());
                for (ResultItem resultItem : list) {
                    LogisticsProcessDto logisticsProcessDto = new LogisticsProcessDto();
                    logisticsProcessDto.setWaybillNo(postId);
                    logisticsProcessDto.setUploadTime(resultItem.getFtime());
                    logisticsProcessDto.setProcessInfo(resultItem.getContext());

                    resultList.add(logisticsProcessDto);
                }
            }
        } catch (Exception e) {
            LOGGER.warn("快递100运单走件流程异常， postId = " + postId, e);
        }

        return resultList;
    }

    /**
     * 即时查询状态
     * @param logisticsCode
     * @param postId
     * @return
     */
    public void queryStatus(String logisticsCode, String postId, String orderId){

        String status = "4";
        String condition = "";
        if (StringUtils.isBlank(postId)) {
            status = "4";
        }

        LOGGER.info("快递100 修复 即时查询：快递公司编码：{}， 快递单号: {} ", logisticsCode, postId);

        // 查询快递100公司信息
        Express100LogisticsEntity express100LogisticsEntity = express100LogisticsService.findByBaiqiLogisticsCode(logisticsCode);
        if (express100LogisticsEntity == null) {
            LOGGER.error("获取不到快递100物流公司信息");

            status = "4";
        }


        try {

            // 获取快递100查询结果
            Result result = KuaiDi100Util.queryWaybillTrace(express100LogisticsEntity.getLogisticsCode(), postId);

            LOGGER.info("快递100 查询 {}",result);
            if (result != null) {

                status = result.getState();
                condition = result.getCondition();
            }
        } catch (Exception e) {
            LOGGER.warn("快递100运单走件流程异常， postId = " + postId, e);
        }

        dealKuaiDi100Status2(logisticsCode,postId,status,condition,orderId);
    }



    /**
     * 处理快递100物流 即时查询的信息
     *
     * @param code    快递100中快递公司编码
     * @param postId 快递单号
     * @param status 快递单当前状态
     * @return boolean
     */
    public Boolean dealKuaiDi100Status2(String code, String postId, String status, String condition, String orderId) {

        LOGGER.info("快递100 修复 推送信息：快递公司编码：{}， 快递单号: {} ，快递单当前状态: {}, 快递单明细状态标记:{}", code, postId, status, condition);

        LogisticsOrderEntity entity = new LogisticsOrderEntity();
        KuaiDi100LogisticsOrderStatusEnum kuaiDi100LogisticsOrderStatusEnum = KuaiDi100LogisticsOrderStatusEnum
                .getByState(status);
        if (kuaiDi100LogisticsOrderStatusEnum != null) {
            entity.setLogisticsStatus(kuaiDi100LogisticsOrderStatusEnum.getDesc());
        } else {
            entity.setLogisticsStatus(status);
        }
        KuaiDi100OrderStatusEnum statusEnum = KuaiDi100OrderStatusEnum.getByKuaiDi100Status(status);
        if (statusEnum == null) {
            return true;
        }
        //快递100拒签状态特殊处理
        if (statusEnum == KuaiDi100OrderStatusEnum.UNDER_WAY && "JQ001".equals(condition)) {
            entity.setOrderStatus(KuaiDi100OrderStatusEnum.REJECTED.getStatus());
        } else {
            entity.setOrderStatus(statusEnum.getStatus());
        }
        entity.setPostId(postId);

        entity.setLogisticsCode(code);
        entity.setOrderId(orderId);
        if (!"0".equals(status)) {
            logisticsOrderService.updateStatus(entity);
        }
        return true;
    }
}
