package com.qiho.center.biz.bo;

import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

import static com.qiho.center.api.constant.OrderRemarkConstant.SIGN_FAILED_TIME_OUT_TASK_CLOSE;

/**
 *  签收失败超时关闭
 * 「签收失败」的物流子订单，超过72小时仍未更新物流状态，则将主订单状态更新为【交易关闭】
 * @Author chensong
 * @Create 2019-02-18
 */
@Component
public class LogisticsCloseBO {

    private static final Logger logger = LoggerFactory.getLogger(LogisticsCloseBO.class);

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    @Autowired
    private QihoOrderDAO qihoOrderDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoFundOrderDAO qihoFundOrderDAO;

    /**
     * 执行签收失败物流子订单超时关闭任务
     */
    public void doLogisticsCloseTask(){

        Date endTime = new DateTime().minusHours(72).toDate();
        Date startTime = new DateTime().minusHours(80).toDate();
        List<LogisticsOrderEntity> logisticsOrderList = qihoLogisticsOrderDAO.findByStatusModifiedTime(startTime, endTime, LogisticsOrderStatusEnum.FAILED.toString());
        if (CollectionUtils.isEmpty(logisticsOrderList)) {
            return;
        }

        logger.info("logistics close task process size = [{}]", logisticsOrderList.size());

        for (LogisticsOrderEntity logisticsOrder : logisticsOrderList) {
            try {
                closeOrder(logisticsOrder);
            } catch (Exception e){
                logger.error("签收失败物流子订单超时关闭任务异常，orderId=[{}]", logisticsOrder.getOrderId(), e);
            }
        }
    }


    /**
     * 关闭订单
     * @param logisticsOrder
     */
    private void closeOrder(LogisticsOrderEntity logisticsOrder) {

        String orderId = logisticsOrder.getOrderId();

        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        updateOrder.setRemark(SIGN_FAILED_TIME_OUT_TASK_CLOSE);
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        orderSnapshot.setRemark(SIGN_FAILED_TIME_OUT_TASK_CLOSE);

        List<QihoFundOrderEntity> list = qihoFundOrderDAO
                .findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fundOrder = list.get(0);
            if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
                qihoFundOrderDAO.updateFundOrderStatus(fundOrder.getFundId(), FundStatusEnum.FAILED.getCode());
                orderSnapshot.setFundStatus(FundStatusEnum.FAILED.getCode());
            }
        }
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }




}
