package com.qiho.center.biz.bo;

import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.base.Joiner;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.biz.model.OrderSyncParamBean;
import com.qiho.center.biz.model.SsjxAddressBean;
import com.qiho.center.biz.model.SsjxPushOrderBean;
import com.qiho.center.biz.service.order.QihoOrderPushThirdPartyFailService;
import com.qiho.center.biz.util.QihoStringUtil;
import com.qiho.center.common.constant.DomainConstantUtil;
import com.qiho.center.common.constant.Environment;
import com.qiho.center.common.entityd.qiho.order.QihoOrderPushThirdPartyFailEntity;
import com.qiho.center.common.util.BigDecimalUtil;
import com.qiho.center.common.util.HttpClientUtil;

import cn.com.duiba.wolf.utils.DateUtils;

/**
 * 盛世嘉祥订单同步
 * <p>
 * Created by sunchangji on 2018/1/11.
 */
@Component
public class SsjxPushOrderBo {

    private static final Logger LOGGER = LoggerFactory.getLogger(SsjxPushOrderBo.class);

    @Resource
    private QihoOrderPushThirdPartyFailService qihoOrderPushThirdPartyFailService;


    /**
     * 监听订单创建事件
     *
     * @param entity
     * @param smsRetCode
     */
    public void createOrderSnapshotListener(OrderSnapshotDto entity, String smsRetCode) {
        try {

            SsjxPushOrderBean bean = orderSnapToBean(entity);
            bean.setSmsCode(smsRetCode);
            bean.setSmsSendTime(System.currentTimeMillis() / 1000);

            // 测试环境打印短信状态
            if (Environment.isDaily() || Environment.isPrepub()){
                LOGGER.info("测试环境打印短信状态和推送盛世嘉祥订单数据。orderId: {}, smsCode:{}, param:{}", entity.getOrderId(), smsRetCode, bean);
            }

            OrderSyncParamBean paramBean = new OrderSyncParamBean(entity.getOrderId(), entity.getMerchantId());
            sendOrderInfo(bean, paramBean, smsRetCode);
        } catch (Exception e) {
            LOGGER.error("订单同步失败", e);
            recodeSyncOrderFail(new OrderSyncParamBean(entity.getOrderId(), entity.getMerchantId()), "订单同步失败",
                smsRetCode);
        }
    }

    private void sendOrderInfo(SsjxPushOrderBean orderBean, OrderSyncParamBean paramBean, String smsRetCode) {
        String orderInfoJson = JSONObject.toJSONString(orderBean);
        Map<String, String> params = JSONObject.parseObject(orderInfoJson, new TypeReference<Map<String, String>>() {
        });
        String orderId = paramBean.getOrderId();
        String resp = HttpClientUtil.sendPostByIgnoreCookies(DomainConstantUtil.getSsjxOrderSyncUrl(), params);
        if (StringUtils.isBlank(resp)) {
            String failMsg = "请求失败";
            LOGGER.warn("发送盛世嘉祥订单同步同步失败, url={}, orderId={},resp={}, param={}", DomainConstantUtil.getSsjxOrderSyncUrl(),
                    orderId, failMsg, params);
            recodeSyncOrderFail(paramBean, failMsg, smsRetCode);
            return;
        }
        JSONObject jsonResp = JSONObject.parseObject(resp);
        if (jsonResp.getInteger("code") == 200) {
            return;
        }
        String failMsg = jsonResp.toJSONString();
        LOGGER.info("发送盛世嘉祥订单同步同步失败, url={},orderId={},code={},resp={}, param={}",
            DomainConstantUtil.getSsjxOrderSyncUrl(), orderId, jsonResp.getInteger("code"), failMsg, params);
        recodeSyncOrderFail(paramBean, failMsg, smsRetCode);
    }

    private void recodeSyncOrderFail(OrderSyncParamBean paramBean, String failMsg, String smsRetCode) {
        //记录不成功的订单推送
        QihoOrderPushThirdPartyFailEntity entity = new QihoOrderPushThirdPartyFailEntity();
        entity.setOrderId(paramBean.getOrderId());
        entity.setMerchantId(paramBean.getMerchantId());
        entity.setSmsRetCode(smsRetCode);
        entity.setRemark(failMsg);
        qihoOrderPushThirdPartyFailService.insert(entity);
    }

    /**
     * 订单快照表转成同步盛世嘉祥订单数据格式
     *
     * @param snapDto
     *
     * @return
     */
    private SsjxPushOrderBean orderSnapToBean(OrderSnapshotDto snapDto) {
        SsjxPushOrderBean orderBean = new SsjxPushOrderBean();
        String skuName = snapDto.getSkuName() == null ? StringUtils.EMPTY : snapDto.getSkuName();
        String goodsSkuName = snapDto.getItemShortName() + skuName;
        orderBean.setGoodstype(snapDto.getItemShortName());
        String goods0 = goodsSkuName + ",,," + snapDto.getQuantity() + "|";
        String goodsName = goodsSkuName + "*" + snapDto.getQuantity();

        orderBean.setGoods0(goods0);
        orderBean.setGoodsName(goodsName);
        orderBean.setGoodstxm(snapDto.getSkuNo() + "*" + snapDto.getQuantity());

        orderBean.setGoodsPrice(
            BigDecimalUtil.integerToMove2PointLeft(snapDto.getOrderAmt()).stripTrailingZeros().toPlainString());
        orderBean.setName(snapDto.getConsumerName());
        orderBean.setTel(snapDto.getMobile());
        SsjxAddressBean addressBean = QihoStringUtil.analyzeSendTime(snapDto.getSendTime());
        if (addressBean != null) {
            orderBean.setSendTime(addressBean.getSendTime());
            orderBean.setProvince3(addressBean.getProvince3());
            orderBean.setCity3(addressBean.getCity3());
            orderBean.setArea3(addressBean.getArea3());
            orderBean.setTownship3(addressBean.getTownship3());
        }
        orderBean.setDetailedAddr(snapDto.getAddress());
        String addrText = Joiner.on("_").join(QihoStringUtil
            .excludeNullString(snapDto.getProvince(), snapDto.getCity(), snapDto.getDistrict(), snapDto.getAddress()));
        orderBean.setDetailedAddrText(addrText);
        orderBean.setMessage(snapDto.getMessage() == null ? StringUtils.EMPTY : snapDto.getMessage());
        orderBean.setIp(snapDto.getIp());
        orderBean.setCreateTime(DateUtils.getSecondStr(snapDto.getGmtCreate()));
        orderBean.setSid(snapDto.getSid());
        orderBean.setBqorderid(snapDto.getOrderId());
        return orderBean;
    }


}
