package com.qiho.center.biz.engine.action;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.StrategyRuleDto;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.ShotOrder.ShotOrderRuleEnum;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.util.HttpClientUtil;

/**
 * 订单详细地址是否无效验证
 *
 * @author peanut.huang
 * @date 2018/4/20.
 */
public class InvalidAddressAction extends ShotOrderAbstractAction  {

    private static final Logger LOGGER = LoggerFactory.getLogger(InvalidAddressAction.class);

    /**
     * 订单详细地址是否无效验证
     * @return
     */
    @Override
    public ResultBase<Boolean> valuteCostomByQuery(OrderSnapshotDto orderSnapshotDto, StrategyRuleDto strategyRuleDto) {

        // 详细地址
        String address = StringUtils.deleteWhitespace(orderSnapshotDto.getAddress());

        // 过滤直辖市，得到所属城市
        String city = filterCity(orderSnapshotDto.getProvince(), orderSnapshotDto.getCity());

        int cityCount = findCityCount(address, city);

        return ShotOrderRuleEnum.fromVal(strategyRuleDto.getRuleName()).valuteOrderByQuery(null, cityCount);
    }

    /**
     * 过滤直辖市
     *
     * @param province  当前省
     * @param city      当前市
     * @return
     */
    private String filterCity(String province, String city) {

        // 直辖市
        List<String> filterCites = Lists.newArrayList("北京市","上海市","天津市","重庆市");

        // 当前省在直辖市内，返回直辖市
        if(filterCites.contains(province)){
            return province;
        }

        // 返回当前省的市区
        return city;
    }


    /**
     * 从高德接口查询详细地址所对应的城市数量
     *
     * @param address   详细地址
     * @param city      订单城市
     * @return
     */
    private int findCityCount(String address, String city) {

        String path = getRequestPath(address, city);

        // 获取结果
        String result = HttpClientUtil.sendGet(path);

        if (StringUtils.isBlank(result)) {
            return 0;
        }

        JSONObject jsonObject = JSON.parseObject(result);

        Object count = jsonObject.get("count");
        if(count == null){
            LOGGER.warn("查询高德地理逆编码错误:{}", result);
            return 0;
        }
        return  Integer.valueOf(count.toString());

    }

    /**
     * 获取高德地图请求地址
     *
     *  数字签名未开启，暂不需要sig参数
     *
     * String paramFront = StringUtils.removeEnd(sb.toString(), "&");
     *
     * // 加上私钥
     * String md5Str = paramFront + DsConstants.GEO_SERVICE_PRIVATE_KEY;
     * // 生成签名
     * String sig = DigestUtils.md5Hex(md5Str);
     *
     * sb.append("sig=").append(sig);
     *
     *
     * @param address  详细地址
     * @param city     城市
     * @return
     */
    private String getRequestPath(String address, String city) {

        Map<String, String> paramMap = Maps.newTreeMap();

        // 地址编码
        String encodeAddress;
        try {
            encodeAddress = URLEncoder.encode(address,"utf-8");
        } catch (UnsupportedEncodingException e) {
            LOGGER.error("地址有效性检验时，地址编码错误：{}",address, e);
            encodeAddress = address;
        }

        paramMap.put("address", encodeAddress);
        paramMap.put("output", "JSON");
        paramMap.put("key", DsConstants.GEO_SERVICE_KEY);
        paramMap.put("city", city);

        // 高德地理编码接口请求地址
        StringBuilder sb = new StringBuilder(DsConstants.GEO_ADDRESS_URL);

        // 构建顺序参数
        paramMap.forEach((k,v) -> sb.append(k).append("=").append(v).append("&"));

        return StringUtils.removeEnd(sb.toString(), "&");
    }
}
