package com.qiho.center.biz.engine.impl;

import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.OrderStrategyDto;
import com.qiho.center.api.dto.StrategyRuleDto;
import com.qiho.center.api.dto.resultbase.ResultBase;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.ShotOrder.OrderProcessWayEnum;
import com.qiho.center.api.enums.ShotOrder.ShotOrderHandlerEnum;
import com.qiho.center.api.enums.ShotOrder.StrategyType;
import com.qiho.center.api.enums.StrategyUseType;
import com.qiho.center.api.params.OrderGainParams;
import com.qiho.center.biz.bo.NoteGainBo;
import com.qiho.center.biz.engine.BaseEngine;
import com.qiho.center.biz.engine.BaseHandler;
import com.qiho.center.biz.engine.Factory.ProcessFaceory;
import com.qiho.center.biz.engine.ProcessContext;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.service.order.OrderSnapshotService;
import com.qiho.center.biz.service.order.StrategyMerchantService;
import com.qiho.center.common.entityd.qiho.order.StrategyMerchantEntity;
import org.apache.commons.beanutils.BeanComparator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.comparators.ComparatorChain;
import org.apache.commons.collections.comparators.FixedOrderComparator;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Resource;
import java.util.*;

/**
 * Created by danke on 2017/6/21.
 * 踢单辅助校验链路入口
 */
public class ShotOrderEngine extends BaseOrderEngine implements BaseEngine<OrderSnapshotDto> {

    Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 命中后执行处理器的map
     */
    private Map<String,BaseEngine> strategyTask;


    /**
     * 校验链路入口类
     * @param valuteData
     * {@link com.qiho.center.biz.engine.chain.BusEntiyHandler}
     * @return
     */
    private BaseHandler baseHandler;

    @Resource
    private NoteGainBo noteGainBo;

    @Resource
    private OrderSnapshotService       orderSnapshotService;
    @Resource
    private StrategyMerchantService    strategyMerchantService;

    @Resource
    private OrderService orderService;


    @Override
    public OrderSnapshotDto process(OrderSnapshotDto valuteData) {
        if(null == valuteData){
            return valuteData;
        }

        // 订单只有在待审核 且OrderProcessWay字段为空时才进行踢单规则
        if (!StringUtils.equalsIgnoreCase(OrderStatusEnum.TO_APPROVE.getCode(), valuteData.getOrderStatus())
                || StringUtils.isNotBlank(valuteData.getOrderProcessWay())) {
            return valuteData;
        }

        try{
            //查询所有策略,循环校验
            List<OrderStrategyDto> orderStrategyDtos = getOrderStrategyList(StrategyType.SHOT_ORDER);
            //查询校验策略,如果为空,直接返回,
            if (CollectionUtils.isEmpty(orderStrategyDtos)) {
                auditOrderPass(valuteData.getOrderId());
                return valuteData;
            }


            //进行一次基础的排序,优先处理审核拒绝的规则-捞单短信-标记
            sortList(orderStrategyDtos);
            //循环执行校验策略
            boolean isHit = checkStrategy(valuteData,orderStrategyDtos);
            //如果所有规则均未命中,则应该默认审核通过
            if(!isHit){
                auditOrderPass(valuteData.getOrderId());
            }
        }catch (Exception e){
            logger.error("策略过滤链路失败, data={}",valuteData.toString(), e);
        }
        return valuteData;
    }


    /**
     * 订单是否匹配策略
     *
     * @param strategyId
     * @param merchantId
     * @return
     */
    private boolean isOrderMatchStrategy(Long strategyId, Long merchantId) {

        //find
        StrategyMerchantEntity strategyMerchantEntity = strategyMerchantService.find(merchantId, strategyId);

        return strategyMerchantEntity != null;
    }


    /**
     * 循环处理规则方案
     * @param valuteData
     * @param orderStrategyDtos
     * @return
     */
    private boolean checkStrategy(OrderSnapshotDto valuteData, List<OrderStrategyDto> orderStrategyDtos) {
        List<String> handlerNames = new ArrayList<>();//这个list是无奈之举,暂时用于标记打上哪种订单处理方式,后续优化
        boolean isHit = false;
        for (OrderStrategyDto orderStrategyDto : orderStrategyDtos){

            //适用部分商家
            if(StrategyUseType.PART.equals(StrategyUseType.findByValue(orderStrategyDto.getUseType()))){
                Long strategyId = orderStrategyDto.getId();
                Long merchantId = valuteData.getMerchantId();

                //判断当前订单商家是否匹配策略
                Boolean match = isOrderMatchStrategy(strategyId, merchantId);

                if(!match){
                    continue;
                }
            }


            //校验基础数据
            ResultBase<List<StrategyRuleDto>> resultBase = validateorderStrategyDto(orderStrategyDto);
            if(!resultBase.hasSuccessValue()) {
                logger.warn(resultBase.getErrorMsg());
                continue;
            }

            List<StrategyRuleDto> strategyRuleDtos =resultBase.getValue();
            ProcessContext processContext = ProcessFaceory.createProcessContext(valuteData,null,orderStrategyDto);
            Boolean flag = checkStrategyDetail(valuteData,strategyRuleDtos,orderStrategyDto);


            //命中后执行链路
            if(flag && null != strategyTask && null!= strategyTask.get(orderStrategyDto.getHandlerName())){
                handlerNames.add(orderStrategyDto.getHandlerName());
                BaseEngine baseEngine = strategyTask.get(orderStrategyDto.getHandlerName());
                baseEngine.process(processContext);
                isHit = true;

                if(processContext.getNeedBroken()){
                   return isHit;
                }
            }
        }
        //规则处理完毕之后,针对命中规则对用处理方式的后续处理
        processByHandlerName(handlerNames,valuteData);
        return isHit;
    }


    private boolean checkStrategyDetail(OrderSnapshotDto valuteData,List<StrategyRuleDto> strategyRuleDtos,OrderStrategyDto orderStrategyDto){
        Boolean flag = Boolean.TRUE;
        for (StrategyRuleDto strategyRuleDto : strategyRuleDtos){

            // 适用部分商家
            if(StrategyUseType.PART.equals(StrategyUseType.findByValue(orderStrategyDto.getUseType()))) {
                strategyRuleDto.setPartMerchant(true);
            }else {
                strategyRuleDto.setPartMerchant(false);
            }

                //真正校验执行
            ProcessContext processContext = ProcessFaceory.createProcessContext(valuteData,strategyRuleDto,orderStrategyDto);
            ResultBase<Boolean> checkResult = baseHandler.handleRequest(processContext);
            if (checkResult.hasSuccessValue()){
                flag = flag && checkResult.getValue();
            }else{
                flag = Boolean.FALSE;
                logger.warn("本条策略规则校验失败 msg={} strategyName={} ruleName={} fieldName={}",
                        checkResult.getErrorMsg(),orderStrategyDto.getStrategyName(),strategyRuleDto.getRuleName(),strategyRuleDto.getFieldName());
                break;//跳出本策略的循环,其中一环错误,校验无意义
            }
        }
        return flag;
    }

    /**
     * 规则处理完毕之后,针对命中规则对用处理方式的后续处理
     * 标记:打上人工审单
     * 短信捞单:发送短信,打上短信捞单,更新短信捞单开始时间
     * @param handlerNames
     */
    private void processByHandlerName(List<String> handlerNames,OrderSnapshotDto orderSnapshotDto){
        if(CollectionUtils.isNotEmpty(handlerNames)){
            int noteGainIndex = handlerNames.indexOf(ShotOrderHandlerEnum.NOTE_GAIN.getVal());
            int stampIndex = handlerNames.indexOf(ShotOrderHandlerEnum.STAMP.getVal());
            OrderGainParams orderGainParams = new OrderGainParams();//将快照处理方式更新为短信捞单
            orderGainParams.setOrderId(orderSnapshotDto.getOrderId());
            if (noteGainIndex > -1){//标记短信捞单处理,发送短信
                orderGainParams.setOrderProcessWayEnum(OrderProcessWayEnum.NOTE_GAIN);
                orderGainParams.setProcessTime(new Date());
                noteGainBo.sendGainNote(orderSnapshotDto);
            }
            if (noteGainIndex == -1 && stampIndex > -1){//标记人工审单处理
                orderGainParams.setOrderProcessWayEnum(OrderProcessWayEnum.MANUAL_CHECK);
            }
            orderSnapshotService.updateOrderProcessWayByOrderId(orderGainParams);
        }
    }

    private void sortList(List<OrderStrategyDto> orderStrategyDtos) {
        String[] shotOrderHandler = { ShotOrderHandlerEnum.AUDIT_REJECT.getVal(),ShotOrderHandlerEnum.NOTE_GAIN.getVal(),ShotOrderHandlerEnum.STAMP.getVal()};
        Comparator typeComparator = new FixedOrderComparator(shotOrderHandler);
        ComparatorChain moInfoComparator = new ComparatorChain();
        moInfoComparator.addComparator(new BeanComparator("handlerName",typeComparator));
        Collections.sort(orderStrategyDtos,moInfoComparator);
    }

    public BaseHandler getBaseHandler() {
        return baseHandler;
    }

    public void setBaseHandler(BaseHandler baseHandler) {
        this.baseHandler = baseHandler;
    }

    public Map<String, BaseEngine> getStrategyTask() {
        return strategyTask;
    }

    public void setStrategyTask(Map<String, BaseEngine> strategyTask) {
        this.strategyTask = strategyTask;
    }

}
