package com.qiho.center.biz.job;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.enums.AreaEnum;
import com.qiho.center.api.enums.LogisticsNameEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.monitor.LogisticsMonitorPlanService;
import com.qiho.center.biz.service.monitor.MonitorReportService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.entityd.qiho.monitor.LogisticsMonitorPlanEntity;
import com.qiho.center.common.entityd.qiho.monitor.LogisticsMonitorReportEntity;
import com.qiho.center.common.entityd.qiho.monitor.LogisticsMonitorStrategyEntity;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created by qianjue on 2017/11/2.
 */
@Component
public class LogisticsMonitorJob extends AbstractQihoSimpleElasticJob{

	@Resource
	LogisticsMonitorPlanService  logisticsMonitorPlanService;

	@Resource
	OrderService orderService;

	@Resource
	MonitorReportService  monitorReportService;

	@Resource
	LogisticsOrderService  logisticsOrderService;

	Logger logger = LoggerFactory.getLogger(LogisticsMonitorJob.class);

	@Override
	protected void doProcess(JobExecutionMultipleShardingContext shardingContext) {
		for (LogisticsNameEnum enumVal : LogisticsNameEnum.values()) {
			//查询当前时间内优先级最高的计划
			LogisticsMonitorPlanEntity planEntity = logisticsMonitorPlanService.queryValidMonitorPlan(enumVal.getCode());
			//如果不存在
			if(null==planEntity){
				return ;
			}
			//查询具体策略
			List<LogisticsMonitorStrategyEntity> strategyEntityList =
					logisticsMonitorPlanService.selectStrategyListByPlanId(planEntity.getId());
			if(CollectionUtils.isEmpty(strategyEntityList)){
				return ;
			}
			dealWithStrategy(planEntity,strategyEntityList);
		}
	}

	private void dealWithStrategy(LogisticsMonitorPlanEntity planEntity,
			List<LogisticsMonitorStrategyEntity> strategyEntityList) {
		for(LogisticsMonitorStrategyEntity strategy:strategyEntityList){
			try{
				if(strategy.getNextExecuteTime().after(new Date())){
					continue;
				}
				Map<String,Long> param = null;
				//如果是监控订单的状态,则需要查询订单的信息
				if(StringUtils.equals(strategy.getMonitorStatus(), OrderStatusEnum.TO_SEND.getCode())){
					param = dealWithOrderMonitor(strategy);
				}else{
					param = dealWithLogisticsOrderMonitor(strategy,planEntity.getLogisticsCode());
				}
				//插入报表数据
				insertReport(strategy,planEntity,param);
				//更新数据
				strategy.setLastExecuteTime(new Date());
				strategy.setNextExecuteTime(DateUtils.addMinutes(new Date(),strategy.getIntervalNum()));
				logisticsMonitorPlanService.updateStrategyExecuteTime(strategy);
			}catch(Exception e){
				logger.info("LogisticsMonitorJob run error,param = {}", strategy.toString(),e);
			}
		}
	}

	private void insertReport(LogisticsMonitorStrategyEntity strategy, LogisticsMonitorPlanEntity planEntity,
			Map<String, Long> param) {
		LogisticsMonitorReportEntity entity = new LogisticsMonitorReportEntity();
		entity.setLogisticsCode(planEntity.getLogisticsCode());
		entity.setPlanId(planEntity.getId());
		entity.setMonitorStatus(strategy.getMonitorStatus());
		entity.setWarehouseId(strategy.getWarehouseId());
		entity.setStrategyId(strategy.getId());
		entity.setOrderCountTotal(param.get("orderCountTotal"));
		entity.setOrderCountNormal(param.get("orderCountNormal"));
		entity.setOrderCountYellow(param.get("orderCountYellow"));
		entity.setOrderCountRed(param.get("orderCountRed"));
		entity.setOrderCountCrash(0L);
		monitorReportService.insert(entity);
	}

	/**
	 * 订单处理逻辑
	 * @param strategy
	 */
	private Map<String,Long> dealWithOrderMonitor(LogisticsMonitorStrategyEntity strategy) {
		Date now = new Date();
		Map<String,Long> maps = Maps.newHashMap();
		//处理省份列表
		List<String> provinceList = transfromAreaToList(strategy.getAreaInfo());
		//查询涉及的订单数
		Long orderCountTotal = orderService.queryMonitorOrderCount(strategy.getMonitorStatus(),null,null,provinceList);
		maps.put("orderCountTotal",orderCountTotal);
		//查询正常订单数
		Long orderCountNormal = orderService.queryMonitorOrderCount(strategy.getMonitorStatus(),
				DateUtils.addHours(now,-strategy.getGeneralLimit().intValue()),null,provinceList);
		maps.put("orderCountNormal",orderCountNormal);
		//查询黄牌订单数
		Long orderCountYellow = orderService.queryMonitorOrderCount(strategy.getMonitorStatus(),
				DateUtils.addHours(now,-strategy.getDeadlineLimit()),DateUtils.addHours(now,-strategy.getGeneralLimit()),provinceList);
		maps.put("orderCountYellow",orderCountYellow);
		//查询红牌订单数
		Long  orderCountRed = orderService.queryMonitorOrderCount(strategy.getMonitorStatus(),
				null,DateUtils.addHours(now,-strategy.getDeadlineLimit()),provinceList);
		maps.put("orderCountRed",orderCountRed);
		return maps;
	}

	private List<String> transfromAreaToList(String areaInfo) {
		List<String> result = Lists.newArrayList();
		List<String> areaCodeList = Splitter.on(",").trimResults().splitToList(areaInfo);
		if(CollectionUtils.isEmpty(areaCodeList)){
			return result;
		}
		areaCodeList.stream().forEach(code->{
			AreaEnum areaEnum = AreaEnum.getAreaEnumByCode(code);
			if(null != areaEnum){
				result.add(areaEnum.getName());
			}
		});
		return result;
	}

	/**
	 * 物流子订单处理逻辑
	 * @param strategy
	 */
	private Map<String,Long> dealWithLogisticsOrderMonitor(LogisticsMonitorStrategyEntity strategy,String logisticsCode){
		Date now = new Date();
		Map<String,Long> maps = Maps.newHashMap();
		//处理省份列表
		List<String> provinceList = transfromAreaToList(strategy.getAreaInfo());
		//查询涉及的订单数
		Long orderCountTotal = logisticsOrderService.queryMonitorLogisticsOrderCount(strategy.getMonitorStatus(),logisticsCode,null,null,provinceList);
		maps.put("orderCountTotal",orderCountTotal);
		//查询正常订单数
		Long orderCountNormal = logisticsOrderService.queryMonitorLogisticsOrderCount(strategy.getMonitorStatus(),logisticsCode,
				DateUtils.addHours(now,-strategy.getGeneralLimit().intValue()),null,provinceList);
		maps.put("orderCountNormal",orderCountNormal);
		//查询黄牌订单数
		Long orderCountYellow = logisticsOrderService.queryMonitorLogisticsOrderCount(strategy.getMonitorStatus(),logisticsCode,
				DateUtils.addHours(now,-strategy.getDeadlineLimit()),DateUtils.addHours(now,-strategy.getGeneralLimit()),provinceList);
		maps.put("orderCountYellow",orderCountYellow);
		//查询红牌订单数
		Long  orderCountRed = logisticsOrderService.queryMonitorLogisticsOrderCount(strategy.getMonitorStatus(),logisticsCode,
				null,DateUtils.addHours(now,-strategy.getDeadlineLimit()),provinceList);
		maps.put("orderCountRed",orderCountRed);
		return maps;
	}
}
